//
// Copyright(C) 2005-2014 Simon Howard
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
//
// This implements a cryptographically secure pseudorandom number
// generator for implementing secure demos. The approach taken is to
// use the AES (Rijndael) stream cipher in "counter" mode, encrypting
// an incrementing counter. The cipher key acts as the random seed.
// Cryptanalysis of AES used in this way has shown it to be an
// effective PRNG (see: Empirical Evidence concerning AES, Hellekalek
// & Wegenkittl, 2003).
//
// The AES implementation is from GnuPG. Original copyright notice is
// below.
//

/* Rijndael (AES) for GnuPG
 *	Copyright (C) 2000, 2001, 2008 Free Software Foundation, Inc.
 *
 * This file is part of GnuPG.
 *
 * GnuPG is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * GnuPG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *******************************************************************
 * The code here is based on the optimized implementation taken from
 * http://www.esat.kuleuven.ac.be/~rijmen/rijndael/ on Oct 2, 2000,
 * which carries this notice:
 *------------------------------------------
 * rijndael-alg-fst.c   v2.3   April '2000
 *
 * Optimised ANSI C code
 *
 * authors: v1.0: Antoon Bosselaers
 *          v2.0: Vincent Rijmen
 *          v2.3: Paulo Barreto
 *
 * This code is placed in the public domain.
 *------------------------------------------
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h> /* for memcmp() */

#include "aes_prng.h"
#include "doomtype.h"
#include "i_system.h"

#define MAXKC			(256/32)
#define MAXROUNDS		14


typedef struct
{
    int  ROUNDS;                   /* key-length-dependent number of rounds */
    uint32_t keySched[MAXROUNDS+1][4];	/* key schedule		*/
} RIJNDAEL_context;

static const byte S[256] =
{
    99, 124, 119, 123, 242, 107, 111, 197,
    48,   1, 103,  43, 254, 215, 171, 118,
    202, 130, 201, 125, 250,  89,  71, 240,
    173, 212, 162, 175, 156, 164, 114, 192,
    183, 253, 147,  38,  54,  63, 247, 204,
    52, 165, 229, 241, 113, 216,  49,  21,
    4, 199,  35, 195,  24, 150,   5, 154,
    7,  18, 128, 226, 235,  39, 178, 117,
    9, 131,  44,  26,  27, 110,  90, 160,
    82,  59, 214, 179,  41, 227,  47, 132,
    83, 209,   0, 237,  32, 252, 177,  91,
    106, 203, 190,  57,  74,  76,  88, 207,
    208, 239, 170, 251,  67,  77,  51, 133,
    69, 249,   2, 127,  80,  60, 159, 168,
    81, 163,  64, 143, 146, 157,  56, 245,
    188, 182, 218,  33,  16, 255, 243, 210,
    205,  12,  19, 236,  95, 151,  68,  23,
    196, 167, 126,  61, 100,  93,  25, 115,
    96, 129,  79, 220,  34,  42, 144, 136,
    70, 238, 184,  20, 222,  94,  11, 219,
    224,  50,  58,  10,  73,   6,  36,  92,
    194, 211, 172,  98, 145, 149, 228, 121,
    231, 200,  55, 109, 141, 213,  78, 169,
    108,  86, 244, 234, 101, 122, 174,   8,
    186, 120,  37,  46,  28, 166, 180, 198,
    232, 221, 116,  31,  75, 189, 139, 138,
    112,  62, 181, 102,  72,   3, 246,  14,
    97,  53,  87, 185, 134, 193,  29, 158,
    225, 248, 152,  17, 105, 217, 142, 148,
    155,  30, 135, 233, 206,  85,  40, 223,
    140, 161, 137,  13, 191, 230,  66, 104,
    65, 153,  45,  15, 176,  84, 187,  22
};


static const byte T1[256][4] =
{
    { 0xc6,0x63,0x63,0xa5 }, { 0xf8,0x7c,0x7c,0x84 },
    { 0xee,0x77,0x77,0x99 }, { 0xf6,0x7b,0x7b,0x8d },
    { 0xff,0xf2,0xf2,0x0d }, { 0xd6,0x6b,0x6b,0xbd },
    { 0xde,0x6f,0x6f,0xb1 }, { 0x91,0xc5,0xc5,0x54 },
    { 0x60,0x30,0x30,0x50 }, { 0x02,0x01,0x01,0x03 },
    { 0xce,0x67,0x67,0xa9 }, { 0x56,0x2b,0x2b,0x7d },
    { 0xe7,0xfe,0xfe,0x19 }, { 0xb5,0xd7,0xd7,0x62 },
    { 0x4d,0xab,0xab,0xe6 }, { 0xec,0x76,0x76,0x9a },
    { 0x8f,0xca,0xca,0x45 }, { 0x1f,0x82,0x82,0x9d },
    { 0x89,0xc9,0xc9,0x40 }, { 0xfa,0x7d,0x7d,0x87 },
    { 0xef,0xfa,0xfa,0x15 }, { 0xb2,0x59,0x59,0xeb },
    { 0x8e,0x47,0x47,0xc9 }, { 0xfb,0xf0,0xf0,0x0b },
    { 0x41,0xad,0xad,0xec }, { 0xb3,0xd4,0xd4,0x67 },
    { 0x5f,0xa2,0xa2,0xfd }, { 0x45,0xaf,0xaf,0xea },
    { 0x23,0x9c,0x9c,0xbf }, { 0x53,0xa4,0xa4,0xf7 },
    { 0xe4,0x72,0x72,0x96 }, { 0x9b,0xc0,0xc0,0x5b },
    { 0x75,0xb7,0xb7,0xc2 }, { 0xe1,0xfd,0xfd,0x1c },
    { 0x3d,0x93,0x93,0xae }, { 0x4c,0x26,0x26,0x6a },
    { 0x6c,0x36,0x36,0x5a }, { 0x7e,0x3f,0x3f,0x41 },
    { 0xf5,0xf7,0xf7,0x02 }, { 0x83,0xcc,0xcc,0x4f },
    { 0x68,0x34,0x34,0x5c }, { 0x51,0xa5,0xa5,0xf4 },
    { 0xd1,0xe5,0xe5,0x34 }, { 0xf9,0xf1,0xf1,0x08 },
    { 0xe2,0x71,0x71,0x93 }, { 0xab,0xd8,0xd8,0x73 },
    { 0x62,0x31,0x31,0x53 }, { 0x2a,0x15,0x15,0x3f },
    { 0x08,0x04,0x04,0x0c }, { 0x95,0xc7,0xc7,0x52 },
    { 0x46,0x23,0x23,0x65 }, { 0x9d,0xc3,0xc3,0x5e },
    { 0x30,0x18,0x18,0x28 }, { 0x37,0x96,0x96,0xa1 },
    { 0x0a,0x05,0x05,0x0f }, { 0x2f,0x9a,0x9a,0xb5 },
    { 0x0e,0x07,0x07,0x09 }, { 0x24,0x12,0x12,0x36 },
    { 0x1b,0x80,0x80,0x9b }, { 0xdf,0xe2,0xe2,0x3d },
    { 0xcd,0xeb,0xeb,0x26 }, { 0x4e,0x27,0x27,0x69 },
    { 0x7f,0xb2,0xb2,0xcd }, { 0xea,0x75,0x75,0x9f },
    { 0x12,0x09,0x09,0x1b }, { 0x1d,0x83,0x83,0x9e },
    { 0x58,0x2c,0x2c,0x74 }, { 0x34,0x1a,0x1a,0x2e },
    { 0x36,0x1b,0x1b,0x2d }, { 0xdc,0x6e,0x6e,0xb2 },
    { 0xb4,0x5a,0x5a,0xee }, { 0x5b,0xa0,0xa0,0xfb },
    { 0xa4,0x52,0x52,0xf6 }, { 0x76,0x3b,0x3b,0x4d },
    { 0xb7,0xd6,0xd6,0x61 }, { 0x7d,0xb3,0xb3,0xce },
    { 0x52,0x29,0x29,0x7b }, { 0xdd,0xe3,0xe3,0x3e },
    { 0x5e,0x2f,0x2f,0x71 }, { 0x13,0x84,0x84,0x97 },
    { 0xa6,0x53,0x53,0xf5 }, { 0xb9,0xd1,0xd1,0x68 },
    { 0x00,0x00,0x00,0x00 }, { 0xc1,0xed,0xed,0x2c },
    { 0x40,0x20,0x20,0x60 }, { 0xe3,0xfc,0xfc,0x1f },
    { 0x79,0xb1,0xb1,0xc8 }, { 0xb6,0x5b,0x5b,0xed },
    { 0xd4,0x6a,0x6a,0xbe }, { 0x8d,0xcb,0xcb,0x46 },
    { 0x67,0xbe,0xbe,0xd9 }, { 0x72,0x39,0x39,0x4b },
    { 0x94,0x4a,0x4a,0xde }, { 0x98,0x4c,0x4c,0xd4 },
    { 0xb0,0x58,0x58,0xe8 }, { 0x85,0xcf,0xcf,0x4a },
    { 0xbb,0xd0,0xd0,0x6b }, { 0xc5,0xef,0xef,0x2a },
    { 0x4f,0xaa,0xaa,0xe5 }, { 0xed,0xfb,0xfb,0x16 },
    { 0x86,0x43,0x43,0xc5 }, { 0x9a,0x4d,0x4d,0xd7 },
    { 0x66,0x33,0x33,0x55 }, { 0x11,0x85,0x85,0x94 },
    { 0x8a,0x45,0x45,0xcf }, { 0xe9,0xf9,0xf9,0x10 },
    { 0x04,0x02,0x02,0x06 }, { 0xfe,0x7f,0x7f,0x81 },
    { 0xa0,0x50,0x50,0xf0 }, { 0x78,0x3c,0x3c,0x44 },
    { 0x25,0x9f,0x9f,0xba }, { 0x4b,0xa8,0xa8,0xe3 },
    { 0xa2,0x51,0x51,0xf3 }, { 0x5d,0xa3,0xa3,0xfe },
    { 0x80,0x40,0x40,0xc0 }, { 0x05,0x8f,0x8f,0x8a },
    { 0x3f,0x92,0x92,0xad }, { 0x21,0x9d,0x9d,0xbc },
    { 0x70,0x38,0x38,0x48 }, { 0xf1,0xf5,0xf5,0x04 },
    { 0x63,0xbc,0xbc,0xdf }, { 0x77,0xb6,0xb6,0xc1 },
    { 0xaf,0xda,0xda,0x75 }, { 0x42,0x21,0x21,0x63 },
    { 0x20,0x10,0x10,0x30 }, { 0xe5,0xff,0xff,0x1a },
    { 0xfd,0xf3,0xf3,0x0e }, { 0xbf,0xd2,0xd2,0x6d },
    { 0x81,0xcd,0xcd,0x4c }, { 0x18,0x0c,0x0c,0x14 },
    { 0x26,0x13,0x13,0x35 }, { 0xc3,0xec,0xec,0x2f },
    { 0xbe,0x5f,0x5f,0xe1 }, { 0x35,0x97,0x97,0xa2 },
    { 0x88,0x44,0x44,0xcc }, { 0x2e,0x17,0x17,0x39 },
    { 0x93,0xc4,0xc4,0x57 }, { 0x55,0xa7,0xa7,0xf2 },
    { 0xfc,0x7e,0x7e,0x82 }, { 0x7a,0x3d,0x3d,0x47 },
    { 0xc8,0x64,0x64,0xac }, { 0xba,0x5d,0x5d,0xe7 },
    { 0x32,0x19,0x19,0x2b }, { 0xe6,0x73,0x73,0x95 },
    { 0xc0,0x60,0x60,0xa0 }, { 0x19,0x81,0x81,0x98 },
    { 0x9e,0x4f,0x4f,0xd1 }, { 0xa3,0xdc,0xdc,0x7f },
    { 0x44,0x22,0x22,0x66 }, { 0x54,0x2a,0x2a,0x7e },
    { 0x3b,0x90,0x90,0xab }, { 0x0b,0x88,0x88,0x83 },
    { 0x8c,0x46,0x46,0xca }, { 0xc7,0xee,0xee,0x29 },
    { 0x6b,0xb8,0xb8,0xd3 }, { 0x28,0x14,0x14,0x3c },
    { 0xa7,0xde,0xde,0x79 }, { 0xbc,0x5e,0x5e,0xe2 },
    { 0x16,0x0b,0x0b,0x1d }, { 0xad,0xdb,0xdb,0x76 },
    { 0xdb,0xe0,0xe0,0x3b }, { 0x64,0x32,0x32,0x56 },
    { 0x74,0x3a,0x3a,0x4e }, { 0x14,0x0a,0x0a,0x1e },
    { 0x92,0x49,0x49,0xdb }, { 0x0c,0x06,0x06,0x0a },
    { 0x48,0x24,0x24,0x6c }, { 0xb8,0x5c,0x5c,0xe4 },
    { 0x9f,0xc2,0xc2,0x5d }, { 0xbd,0xd3,0xd3,0x6e },
    { 0x43,0xac,0xac,0xef }, { 0xc4,0x62,0x62,0xa6 },
    { 0x39,0x91,0x91,0xa8 }, { 0x31,0x95,0x95,0xa4 },
    { 0xd3,0xe4,0xe4,0x37 }, { 0xf2,0x79,0x79,0x8b },
    { 0xd5,0xe7,0xe7,0x32 }, { 0x8b,0xc8,0xc8,0x43 },
    { 0x6e,0x37,0x37,0x59 }, { 0xda,0x6d,0x6d,0xb7 },
    { 0x01,0x8d,0x8d,0x8c }, { 0xb1,0xd5,0xd5,0x64 },
    { 0x9c,0x4e,0x4e,0xd2 }, { 0x49,0xa9,0xa9,0xe0 },
    { 0xd8,0x6c,0x6c,0xb4 }, { 0xac,0x56,0x56,0xfa },
    { 0xf3,0xf4,0xf4,0x07 }, { 0xcf,0xea,0xea,0x25 },
    { 0xca,0x65,0x65,0xaf }, { 0xf4,0x7a,0x7a,0x8e },
    { 0x47,0xae,0xae,0xe9 }, { 0x10,0x08,0x08,0x18 },
    { 0x6f,0xba,0xba,0xd5 }, { 0xf0,0x78,0x78,0x88 },
    { 0x4a,0x25,0x25,0x6f }, { 0x5c,0x2e,0x2e,0x72 },
    { 0x38,0x1c,0x1c,0x24 }, { 0x57,0xa6,0xa6,0xf1 },
    { 0x73,0xb4,0xb4,0xc7 }, { 0x97,0xc6,0xc6,0x51 },
    { 0xcb,0xe8,0xe8,0x23 }, { 0xa1,0xdd,0xdd,0x7c },
    { 0xe8,0x74,0x74,0x9c }, { 0x3e,0x1f,0x1f,0x21 },
    { 0x96,0x4b,0x4b,0xdd }, { 0x61,0xbd,0xbd,0xdc },
    { 0x0d,0x8b,0x8b,0x86 }, { 0x0f,0x8a,0x8a,0x85 },
    { 0xe0,0x70,0x70,0x90 }, { 0x7c,0x3e,0x3e,0x42 },
    { 0x71,0xb5,0xb5,0xc4 }, { 0xcc,0x66,0x66,0xaa },
    { 0x90,0x48,0x48,0xd8 }, { 0x06,0x03,0x03,0x05 },
    { 0xf7,0xf6,0xf6,0x01 }, { 0x1c,0x0e,0x0e,0x12 },
    { 0xc2,0x61,0x61,0xa3 }, { 0x6a,0x35,0x35,0x5f },
    { 0xae,0x57,0x57,0xf9 }, { 0x69,0xb9,0xb9,0xd0 },
    { 0x17,0x86,0x86,0x91 }, { 0x99,0xc1,0xc1,0x58 },
    { 0x3a,0x1d,0x1d,0x27 }, { 0x27,0x9e,0x9e,0xb9 },
    { 0xd9,0xe1,0xe1,0x38 }, { 0xeb,0xf8,0xf8,0x13 },
    { 0x2b,0x98,0x98,0xb3 }, { 0x22,0x11,0x11,0x33 },
    { 0xd2,0x69,0x69,0xbb }, { 0xa9,0xd9,0xd9,0x70 },
    { 0x07,0x8e,0x8e,0x89 }, { 0x33,0x94,0x94,0xa7 },
    { 0x2d,0x9b,0x9b,0xb6 }, { 0x3c,0x1e,0x1e,0x22 },
    { 0x15,0x87,0x87,0x92 }, { 0xc9,0xe9,0xe9,0x20 },
    { 0x87,0xce,0xce,0x49 }, { 0xaa,0x55,0x55,0xff },
    { 0x50,0x28,0x28,0x78 }, { 0xa5,0xdf,0xdf,0x7a },
    { 0x03,0x8c,0x8c,0x8f }, { 0x59,0xa1,0xa1,0xf8 },
    { 0x09,0x89,0x89,0x80 }, { 0x1a,0x0d,0x0d,0x17 },
    { 0x65,0xbf,0xbf,0xda }, { 0xd7,0xe6,0xe6,0x31 },
    { 0x84,0x42,0x42,0xc6 }, { 0xd0,0x68,0x68,0xb8 },
    { 0x82,0x41,0x41,0xc3 }, { 0x29,0x99,0x99,0xb0 },
    { 0x5a,0x2d,0x2d,0x77 }, { 0x1e,0x0f,0x0f,0x11 },
    { 0x7b,0xb0,0xb0,0xcb }, { 0xa8,0x54,0x54,0xfc },
    { 0x6d,0xbb,0xbb,0xd6 }, { 0x2c,0x16,0x16,0x3a }
};

static const byte T2[256][4] =
{
    { 0xa5,0xc6,0x63,0x63 }, { 0x84,0xf8,0x7c,0x7c },
    { 0x99,0xee,0x77,0x77 }, { 0x8d,0xf6,0x7b,0x7b },
    { 0x0d,0xff,0xf2,0xf2 }, { 0xbd,0xd6,0x6b,0x6b },
    { 0xb1,0xde,0x6f,0x6f }, { 0x54,0x91,0xc5,0xc5 },
    { 0x50,0x60,0x30,0x30 }, { 0x03,0x02,0x01,0x01 },
    { 0xa9,0xce,0x67,0x67 }, { 0x7d,0x56,0x2b,0x2b },
    { 0x19,0xe7,0xfe,0xfe }, { 0x62,0xb5,0xd7,0xd7 },
    { 0xe6,0x4d,0xab,0xab }, { 0x9a,0xec,0x76,0x76 },
    { 0x45,0x8f,0xca,0xca }, { 0x9d,0x1f,0x82,0x82 },
    { 0x40,0x89,0xc9,0xc9 }, { 0x87,0xfa,0x7d,0x7d },
    { 0x15,0xef,0xfa,0xfa }, { 0xeb,0xb2,0x59,0x59 },
    { 0xc9,0x8e,0x47,0x47 }, { 0x0b,0xfb,0xf0,0xf0 },
    { 0xec,0x41,0xad,0xad }, { 0x67,0xb3,0xd4,0xd4 },
    { 0xfd,0x5f,0xa2,0xa2 }, { 0xea,0x45,0xaf,0xaf },
    { 0xbf,0x23,0x9c,0x9c }, { 0xf7,0x53,0xa4,0xa4 },
    { 0x96,0xe4,0x72,0x72 }, { 0x5b,0x9b,0xc0,0xc0 },
    { 0xc2,0x75,0xb7,0xb7 }, { 0x1c,0xe1,0xfd,0xfd },
    { 0xae,0x3d,0x93,0x93 }, { 0x6a,0x4c,0x26,0x26 },
    { 0x5a,0x6c,0x36,0x36 }, { 0x41,0x7e,0x3f,0x3f },
    { 0x02,0xf5,0xf7,0xf7 }, { 0x4f,0x83,0xcc,0xcc },
    { 0x5c,0x68,0x34,0x34 }, { 0xf4,0x51,0xa5,0xa5 },
    { 0x34,0xd1,0xe5,0xe5 }, { 0x08,0xf9,0xf1,0xf1 },
    { 0x93,0xe2,0x71,0x71 }, { 0x73,0xab,0xd8,0xd8 },
    { 0x53,0x62,0x31,0x31 }, { 0x3f,0x2a,0x15,0x15 },
    { 0x0c,0x08,0x04,0x04 }, { 0x52,0x95,0xc7,0xc7 },
    { 0x65,0x46,0x23,0x23 }, { 0x5e,0x9d,0xc3,0xc3 },
    { 0x28,0x30,0x18,0x18 }, { 0xa1,0x37,0x96,0x96 },
    { 0x0f,0x0a,0x05,0x05 }, { 0xb5,0x2f,0x9a,0x9a },
    { 0x09,0x0e,0x07,0x07 }, { 0x36,0x24,0x12,0x12 },
    { 0x9b,0x1b,0x80,0x80 }, { 0x3d,0xdf,0xe2,0xe2 },
    { 0x26,0xcd,0xeb,0xeb }, { 0x69,0x4e,0x27,0x27 },
    { 0xcd,0x7f,0xb2,0xb2 }, { 0x9f,0xea,0x75,0x75 },
    { 0x1b,0x12,0x09,0x09 }, { 0x9e,0x1d,0x83,0x83 },
    { 0x74,0x58,0x2c,0x2c }, { 0x2e,0x34,0x1a,0x1a },
    { 0x2d,0x36,0x1b,0x1b }, { 0xb2,0xdc,0x6e,0x6e },
    { 0xee,0xb4,0x5a,0x5a }, { 0xfb,0x5b,0xa0,0xa0 },
    { 0xf6,0xa4,0x52,0x52 }, { 0x4d,0x76,0x3b,0x3b },
    { 0x61,0xb7,0xd6,0xd6 }, { 0xce,0x7d,0xb3,0xb3 },
    { 0x7b,0x52,0x29,0x29 }, { 0x3e,0xdd,0xe3,0xe3 },
    { 0x71,0x5e,0x2f,0x2f }, { 0x97,0x13,0x84,0x84 },
    { 0xf5,0xa6,0x53,0x53 }, { 0x68,0xb9,0xd1,0xd1 },
    { 0x00,0x00,0x00,0x00 }, { 0x2c,0xc1,0xed,0xed },
    { 0x60,0x40,0x20,0x20 }, { 0x1f,0xe3,0xfc,0xfc },
    { 0xc8,0x79,0xb1,0xb1 }, { 0xed,0xb6,0x5b,0x5b },
    { 0xbe,0xd4,0x6a,0x6a }, { 0x46,0x8d,0xcb,0xcb },
    { 0xd9,0x67,0xbe,0xbe }, { 0x4b,0x72,0x39,0x39 },
    { 0xde,0x94,0x4a,0x4a }, { 0xd4,0x98,0x4c,0x4c },
    { 0xe8,0xb0,0x58,0x58 }, { 0x4a,0x85,0xcf,0xcf },
    { 0x6b,0xbb,0xd0,0xd0 }, { 0x2a,0xc5,0xef,0xef },
    { 0xe5,0x4f,0xaa,0xaa }, { 0x16,0xed,0xfb,0xfb },
    { 0xc5,0x86,0x43,0x43 }, { 0xd7,0x9a,0x4d,0x4d },
    { 0x55,0x66,0x33,0x33 }, { 0x94,0x11,0x85,0x85 },
    { 0xcf,0x8a,0x45,0x45 }, { 0x10,0xe9,0xf9,0xf9 },
    { 0x06,0x04,0x02,0x02 }, { 0x81,0xfe,0x7f,0x7f },
    { 0xf0,0xa0,0x50,0x50 }, { 0x44,0x78,0x3c,0x3c },
    { 0xba,0x25,0x9f,0x9f }, { 0xe3,0x4b,0xa8,0xa8 },
    { 0xf3,0xa2,0x51,0x51 }, { 0xfe,0x5d,0xa3,0xa3 },
    { 0xc0,0x80,0x40,0x40 }, { 0x8a,0x05,0x8f,0x8f },
    { 0xad,0x3f,0x92,0x92 }, { 0xbc,0x21,0x9d,0x9d },
    { 0x48,0x70,0x38,0x38 }, { 0x04,0xf1,0xf5,0xf5 },
    { 0xdf,0x63,0xbc,0xbc }, { 0xc1,0x77,0xb6,0xb6 },
    { 0x75,0xaf,0xda,0xda }, { 0x63,0x42,0x21,0x21 },
    { 0x30,0x20,0x10,0x10 }, { 0x1a,0xe5,0xff,0xff },
    { 0x0e,0xfd,0xf3,0xf3 }, { 0x6d,0xbf,0xd2,0xd2 },
    { 0x4c,0x81,0xcd,0xcd }, { 0x14,0x18,0x0c,0x0c },
    { 0x35,0x26,0x13,0x13 }, { 0x2f,0xc3,0xec,0xec },
    { 0xe1,0xbe,0x5f,0x5f }, { 0xa2,0x35,0x97,0x97 },
    { 0xcc,0x88,0x44,0x44 }, { 0x39,0x2e,0x17,0x17 },
    { 0x57,0x93,0xc4,0xc4 }, { 0xf2,0x55,0xa7,0xa7 },
    { 0x82,0xfc,0x7e,0x7e }, { 0x47,0x7a,0x3d,0x3d },
    { 0xac,0xc8,0x64,0x64 }, { 0xe7,0xba,0x5d,0x5d },
    { 0x2b,0x32,0x19,0x19 }, { 0x95,0xe6,0x73,0x73 },
    { 0xa0,0xc0,0x60,0x60 }, { 0x98,0x19,0x81,0x81 },
    { 0xd1,0x9e,0x4f,0x4f }, { 0x7f,0xa3,0xdc,0xdc },
    { 0x66,0x44,0x22,0x22 }, { 0x7e,0x54,0x2a,0x2a },
    { 0xab,0x3b,0x90,0x90 }, { 0x83,0x0b,0x88,0x88 },
    { 0xca,0x8c,0x46,0x46 }, { 0x29,0xc7,0xee,0xee },
    { 0xd3,0x6b,0xb8,0xb8 }, { 0x3c,0x28,0x14,0x14 },
    { 0x79,0xa7,0xde,0xde }, { 0xe2,0xbc,0x5e,0x5e },
    { 0x1d,0x16,0x0b,0x0b }, { 0x76,0xad,0xdb,0xdb },
    { 0x3b,0xdb,0xe0,0xe0 }, { 0x56,0x64,0x32,0x32 },
    { 0x4e,0x74,0x3a,0x3a }, { 0x1e,0x14,0x0a,0x0a },
    { 0xdb,0x92,0x49,0x49 }, { 0x0a,0x0c,0x06,0x06 },
    { 0x6c,0x48,0x24,0x24 }, { 0xe4,0xb8,0x5c,0x5c },
    { 0x5d,0x9f,0xc2,0xc2 }, { 0x6e,0xbd,0xd3,0xd3 },
    { 0xef,0x43,0xac,0xac }, { 0xa6,0xc4,0x62,0x62 },
    { 0xa8,0x39,0x91,0x91 }, { 0xa4,0x31,0x95,0x95 },
    { 0x37,0xd3,0xe4,0xe4 }, { 0x8b,0xf2,0x79,0x79 },
    { 0x32,0xd5,0xe7,0xe7 }, { 0x43,0x8b,0xc8,0xc8 },
    { 0x59,0x6e,0x37,0x37 }, { 0xb7,0xda,0x6d,0x6d },
    { 0x8c,0x01,0x8d,0x8d }, { 0x64,0xb1,0xd5,0xd5 },
    { 0xd2,0x9c,0x4e,0x4e }, { 0xe0,0x49,0xa9,0xa9 },
    { 0xb4,0xd8,0x6c,0x6c }, { 0xfa,0xac,0x56,0x56 },
    { 0x07,0xf3,0xf4,0xf4 }, { 0x25,0xcf,0xea,0xea },
    { 0xaf,0xca,0x65,0x65 }, { 0x8e,0xf4,0x7a,0x7a },
    { 0xe9,0x47,0xae,0xae }, { 0x18,0x10,0x08,0x08 },
    { 0xd5,0x6f,0xba,0xba }, { 0x88,0xf0,0x78,0x78 },
    { 0x6f,0x4a,0x25,0x25 }, { 0x72,0x5c,0x2e,0x2e },
    { 0x24,0x38,0x1c,0x1c }, { 0xf1,0x57,0xa6,0xa6 },
    { 0xc7,0x73,0xb4,0xb4 }, { 0x51,0x97,0xc6,0xc6 },
    { 0x23,0xcb,0xe8,0xe8 }, { 0x7c,0xa1,0xdd,0xdd },
    { 0x9c,0xe8,0x74,0x74 }, { 0x21,0x3e,0x1f,0x1f },
    { 0xdd,0x96,0x4b,0x4b }, { 0xdc,0x61,0xbd,0xbd },
    { 0x86,0x0d,0x8b,0x8b }, { 0x85,0x0f,0x8a,0x8a },
    { 0x90,0xe0,0x70,0x70 }, { 0x42,0x7c,0x3e,0x3e },
    { 0xc4,0x71,0xb5,0xb5 }, { 0xaa,0xcc,0x66,0x66 },
    { 0xd8,0x90,0x48,0x48 }, { 0x05,0x06,0x03,0x03 },
    { 0x01,0xf7,0xf6,0xf6 }, { 0x12,0x1c,0x0e,0x0e },
    { 0xa3,0xc2,0x61,0x61 }, { 0x5f,0x6a,0x35,0x35 },
    { 0xf9,0xae,0x57,0x57 }, { 0xd0,0x69,0xb9,0xb9 },
    { 0x91,0x17,0x86,0x86 }, { 0x58,0x99,0xc1,0xc1 },
    { 0x27,0x3a,0x1d,0x1d }, { 0xb9,0x27,0x9e,0x9e },
    { 0x38,0xd9,0xe1,0xe1 }, { 0x13,0xeb,0xf8,0xf8 },
    { 0xb3,0x2b,0x98,0x98 }, { 0x33,0x22,0x11,0x11 },
    { 0xbb,0xd2,0x69,0x69 }, { 0x70,0xa9,0xd9,0xd9 },
    { 0x89,0x07,0x8e,0x8e }, { 0xa7,0x33,0x94,0x94 },
    { 0xb6,0x2d,0x9b,0x9b }, { 0x22,0x3c,0x1e,0x1e },
    { 0x92,0x15,0x87,0x87 }, { 0x20,0xc9,0xe9,0xe9 },
    { 0x49,0x87,0xce,0xce }, { 0xff,0xaa,0x55,0x55 },
    { 0x78,0x50,0x28,0x28 }, { 0x7a,0xa5,0xdf,0xdf },
    { 0x8f,0x03,0x8c,0x8c }, { 0xf8,0x59,0xa1,0xa1 },
    { 0x80,0x09,0x89,0x89 }, { 0x17,0x1a,0x0d,0x0d },
    { 0xda,0x65,0xbf,0xbf }, { 0x31,0xd7,0xe6,0xe6 },
    { 0xc6,0x84,0x42,0x42 }, { 0xb8,0xd0,0x68,0x68 },
    { 0xc3,0x82,0x41,0x41 }, { 0xb0,0x29,0x99,0x99 },
    { 0x77,0x5a,0x2d,0x2d }, { 0x11,0x1e,0x0f,0x0f },
    { 0xcb,0x7b,0xb0,0xb0 }, { 0xfc,0xa8,0x54,0x54 },
    { 0xd6,0x6d,0xbb,0xbb }, { 0x3a,0x2c,0x16,0x16 }
};

static const byte T3[256][4] =
{
    { 0x63,0xa5,0xc6,0x63 }, { 0x7c,0x84,0xf8,0x7c },
    { 0x77,0x99,0xee,0x77 }, { 0x7b,0x8d,0xf6,0x7b },
    { 0xf2,0x0d,0xff,0xf2 }, { 0x6b,0xbd,0xd6,0x6b },
    { 0x6f,0xb1,0xde,0x6f }, { 0xc5,0x54,0x91,0xc5 },
    { 0x30,0x50,0x60,0x30 }, { 0x01,0x03,0x02,0x01 },
    { 0x67,0xa9,0xce,0x67 }, { 0x2b,0x7d,0x56,0x2b },
    { 0xfe,0x19,0xe7,0xfe }, { 0xd7,0x62,0xb5,0xd7 },
    { 0xab,0xe6,0x4d,0xab }, { 0x76,0x9a,0xec,0x76 },
    { 0xca,0x45,0x8f,0xca }, { 0x82,0x9d,0x1f,0x82 },
    { 0xc9,0x40,0x89,0xc9 }, { 0x7d,0x87,0xfa,0x7d },
    { 0xfa,0x15,0xef,0xfa }, { 0x59,0xeb,0xb2,0x59 },
    { 0x47,0xc9,0x8e,0x47 }, { 0xf0,0x0b,0xfb,0xf0 },
    { 0xad,0xec,0x41,0xad }, { 0xd4,0x67,0xb3,0xd4 },
    { 0xa2,0xfd,0x5f,0xa2 }, { 0xaf,0xea,0x45,0xaf },
    { 0x9c,0xbf,0x23,0x9c }, { 0xa4,0xf7,0x53,0xa4 },
    { 0x72,0x96,0xe4,0x72 }, { 0xc0,0x5b,0x9b,0xc0 },
    { 0xb7,0xc2,0x75,0xb7 }, { 0xfd,0x1c,0xe1,0xfd },
    { 0x93,0xae,0x3d,0x93 }, { 0x26,0x6a,0x4c,0x26 },
    { 0x36,0x5a,0x6c,0x36 }, { 0x3f,0x41,0x7e,0x3f },
    { 0xf7,0x02,0xf5,0xf7 }, { 0xcc,0x4f,0x83,0xcc },
    { 0x34,0x5c,0x68,0x34 }, { 0xa5,0xf4,0x51,0xa5 },
    { 0xe5,0x34,0xd1,0xe5 }, { 0xf1,0x08,0xf9,0xf1 },
    { 0x71,0x93,0xe2,0x71 }, { 0xd8,0x73,0xab,0xd8 },
    { 0x31,0x53,0x62,0x31 }, { 0x15,0x3f,0x2a,0x15 },
    { 0x04,0x0c,0x08,0x04 }, { 0xc7,0x52,0x95,0xc7 },
    { 0x23,0x65,0x46,0x23 }, { 0xc3,0x5e,0x9d,0xc3 },
    { 0x18,0x28,0x30,0x18 }, { 0x96,0xa1,0x37,0x96 },
    { 0x05,0x0f,0x0a,0x05 }, { 0x9a,0xb5,0x2f,0x9a },
    { 0x07,0x09,0x0e,0x07 }, { 0x12,0x36,0x24,0x12 },
    { 0x80,0x9b,0x1b,0x80 }, { 0xe2,0x3d,0xdf,0xe2 },
    { 0xeb,0x26,0xcd,0xeb }, { 0x27,0x69,0x4e,0x27 },
    { 0xb2,0xcd,0x7f,0xb2 }, { 0x75,0x9f,0xea,0x75 },
    { 0x09,0x1b,0x12,0x09 }, { 0x83,0x9e,0x1d,0x83 },
    { 0x2c,0x74,0x58,0x2c }, { 0x1a,0x2e,0x34,0x1a },
    { 0x1b,0x2d,0x36,0x1b }, { 0x6e,0xb2,0xdc,0x6e },
    { 0x5a,0xee,0xb4,0x5a }, { 0xa0,0xfb,0x5b,0xa0 },
    { 0x52,0xf6,0xa4,0x52 }, { 0x3b,0x4d,0x76,0x3b },
    { 0xd6,0x61,0xb7,0xd6 }, { 0xb3,0xce,0x7d,0xb3 },
    { 0x29,0x7b,0x52,0x29 }, { 0xe3,0x3e,0xdd,0xe3 },
    { 0x2f,0x71,0x5e,0x2f }, { 0x84,0x97,0x13,0x84 },
    { 0x53,0xf5,0xa6,0x53 }, { 0xd1,0x68,0xb9,0xd1 },
    { 0x00,0x00,0x00,0x00 }, { 0xed,0x2c,0xc1,0xed },
    { 0x20,0x60,0x40,0x20 }, { 0xfc,0x1f,0xe3,0xfc },
    { 0xb1,0xc8,0x79,0xb1 }, { 0x5b,0xed,0xb6,0x5b },
    { 0x6a,0xbe,0xd4,0x6a }, { 0xcb,0x46,0x8d,0xcb },
    { 0xbe,0xd9,0x67,0xbe }, { 0x39,0x4b,0x72,0x39 },
    { 0x4a,0xde,0x94,0x4a }, { 0x4c,0xd4,0x98,0x4c },
    { 0x58,0xe8,0xb0,0x58 }, { 0xcf,0x4a,0x85,0xcf },
    { 0xd0,0x6b,0xbb,0xd0 }, { 0xef,0x2a,0xc5,0xef },
    { 0xaa,0xe5,0x4f,0xaa }, { 0xfb,0x16,0xed,0xfb },
    { 0x43,0xc5,0x86,0x43 }, { 0x4d,0xd7,0x9a,0x4d },
    { 0x33,0x55,0x66,0x33 }, { 0x85,0x94,0x11,0x85 },
    { 0x45,0xcf,0x8a,0x45 }, { 0xf9,0x10,0xe9,0xf9 },
    { 0x02,0x06,0x04,0x02 }, { 0x7f,0x81,0xfe,0x7f },
    { 0x50,0xf0,0xa0,0x50 }, { 0x3c,0x44,0x78,0x3c },
    { 0x9f,0xba,0x25,0x9f }, { 0xa8,0xe3,0x4b,0xa8 },
    { 0x51,0xf3,0xa2,0x51 }, { 0xa3,0xfe,0x5d,0xa3 },
    { 0x40,0xc0,0x80,0x40 }, { 0x8f,0x8a,0x05,0x8f },
    { 0x92,0xad,0x3f,0x92 }, { 0x9d,0xbc,0x21,0x9d },
    { 0x38,0x48,0x70,0x38 }, { 0xf5,0x04,0xf1,0xf5 },
    { 0xbc,0xdf,0x63,0xbc }, { 0xb6,0xc1,0x77,0xb6 },
    { 0xda,0x75,0xaf,0xda }, { 0x21,0x63,0x42,0x21 },
    { 0x10,0x30,0x20,0x10 }, { 0xff,0x1a,0xe5,0xff },
    { 0xf3,0x0e,0xfd,0xf3 }, { 0xd2,0x6d,0xbf,0xd2 },
    { 0xcd,0x4c,0x81,0xcd }, { 0x0c,0x14,0x18,0x0c },
    { 0x13,0x35,0x26,0x13 }, { 0xec,0x2f,0xc3,0xec },
    { 0x5f,0xe1,0xbe,0x5f }, { 0x97,0xa2,0x35,0x97 },
    { 0x44,0xcc,0x88,0x44 }, { 0x17,0x39,0x2e,0x17 },
    { 0xc4,0x57,0x93,0xc4 }, { 0xa7,0xf2,0x55,0xa7 },
    { 0x7e,0x82,0xfc,0x7e }, { 0x3d,0x47,0x7a,0x3d },
    { 0x64,0xac,0xc8,0x64 }, { 0x5d,0xe7,0xba,0x5d },
    { 0x19,0x2b,0x32,0x19 }, { 0x73,0x95,0xe6,0x73 },
    { 0x60,0xa0,0xc0,0x60 }, { 0x81,0x98,0x19,0x81 },
    { 0x4f,0xd1,0x9e,0x4f }, { 0xdc,0x7f,0xa3,0xdc },
    { 0x22,0x66,0x44,0x22 }, { 0x2a,0x7e,0x54,0x2a },
    { 0x90,0xab,0x3b,0x90 }, { 0x88,0x83,0x0b,0x88 },
    { 0x46,0xca,0x8c,0x46 }, { 0xee,0x29,0xc7,0xee },
    { 0xb8,0xd3,0x6b,0xb8 }, { 0x14,0x3c,0x28,0x14 },
    { 0xde,0x79,0xa7,0xde }, { 0x5e,0xe2,0xbc,0x5e },
    { 0x0b,0x1d,0x16,0x0b }, { 0xdb,0x76,0xad,0xdb },
    { 0xe0,0x3b,0xdb,0xe0 }, { 0x32,0x56,0x64,0x32 },
    { 0x3a,0x4e,0x74,0x3a }, { 0x0a,0x1e,0x14,0x0a },
    { 0x49,0xdb,0x92,0x49 }, { 0x06,0x0a,0x0c,0x06 },
    { 0x24,0x6c,0x48,0x24 }, { 0x5c,0xe4,0xb8,0x5c },
    { 0xc2,0x5d,0x9f,0xc2 }, { 0xd3,0x6e,0xbd,0xd3 },
    { 0xac,0xef,0x43,0xac }, { 0x62,0xa6,0xc4,0x62 },
    { 0x91,0xa8,0x39,0x91 }, { 0x95,0xa4,0x31,0x95 },
    { 0xe4,0x37,0xd3,0xe4 }, { 0x79,0x8b,0xf2,0x79 },
    { 0xe7,0x32,0xd5,0xe7 }, { 0xc8,0x43,0x8b,0xc8 },
    { 0x37,0x59,0x6e,0x37 }, { 0x6d,0xb7,0xda,0x6d },
    { 0x8d,0x8c,0x01,0x8d }, { 0xd5,0x64,0xb1,0xd5 },
    { 0x4e,0xd2,0x9c,0x4e }, { 0xa9,0xe0,0x49,0xa9 },
    { 0x6c,0xb4,0xd8,0x6c }, { 0x56,0xfa,0xac,0x56 },
    { 0xf4,0x07,0xf3,0xf4 }, { 0xea,0x25,0xcf,0xea },
    { 0x65,0xaf,0xca,0x65 }, { 0x7a,0x8e,0xf4,0x7a },
    { 0xae,0xe9,0x47,0xae }, { 0x08,0x18,0x10,0x08 },
    { 0xba,0xd5,0x6f,0xba }, { 0x78,0x88,0xf0,0x78 },
    { 0x25,0x6f,0x4a,0x25 }, { 0x2e,0x72,0x5c,0x2e },
    { 0x1c,0x24,0x38,0x1c }, { 0xa6,0xf1,0x57,0xa6 },
    { 0xb4,0xc7,0x73,0xb4 }, { 0xc6,0x51,0x97,0xc6 },
    { 0xe8,0x23,0xcb,0xe8 }, { 0xdd,0x7c,0xa1,0xdd },
    { 0x74,0x9c,0xe8,0x74 }, { 0x1f,0x21,0x3e,0x1f },
    { 0x4b,0xdd,0x96,0x4b }, { 0xbd,0xdc,0x61,0xbd },
    { 0x8b,0x86,0x0d,0x8b }, { 0x8a,0x85,0x0f,0x8a },
    { 0x70,0x90,0xe0,0x70 }, { 0x3e,0x42,0x7c,0x3e },
    { 0xb5,0xc4,0x71,0xb5 }, { 0x66,0xaa,0xcc,0x66 },
    { 0x48,0xd8,0x90,0x48 }, { 0x03,0x05,0x06,0x03 },
    { 0xf6,0x01,0xf7,0xf6 }, { 0x0e,0x12,0x1c,0x0e },
    { 0x61,0xa3,0xc2,0x61 }, { 0x35,0x5f,0x6a,0x35 },
    { 0x57,0xf9,0xae,0x57 }, { 0xb9,0xd0,0x69,0xb9 },
    { 0x86,0x91,0x17,0x86 }, { 0xc1,0x58,0x99,0xc1 },
    { 0x1d,0x27,0x3a,0x1d }, { 0x9e,0xb9,0x27,0x9e },
    { 0xe1,0x38,0xd9,0xe1 }, { 0xf8,0x13,0xeb,0xf8 },
    { 0x98,0xb3,0x2b,0x98 }, { 0x11,0x33,0x22,0x11 },
    { 0x69,0xbb,0xd2,0x69 }, { 0xd9,0x70,0xa9,0xd9 },
    { 0x8e,0x89,0x07,0x8e }, { 0x94,0xa7,0x33,0x94 },
    { 0x9b,0xb6,0x2d,0x9b }, { 0x1e,0x22,0x3c,0x1e },
    { 0x87,0x92,0x15,0x87 }, { 0xe9,0x20,0xc9,0xe9 },
    { 0xce,0x49,0x87,0xce }, { 0x55,0xff,0xaa,0x55 },
    { 0x28,0x78,0x50,0x28 }, { 0xdf,0x7a,0xa5,0xdf },
    { 0x8c,0x8f,0x03,0x8c }, { 0xa1,0xf8,0x59,0xa1 },
    { 0x89,0x80,0x09,0x89 }, { 0x0d,0x17,0x1a,0x0d },
    { 0xbf,0xda,0x65,0xbf }, { 0xe6,0x31,0xd7,0xe6 },
    { 0x42,0xc6,0x84,0x42 }, { 0x68,0xb8,0xd0,0x68 },
    { 0x41,0xc3,0x82,0x41 }, { 0x99,0xb0,0x29,0x99 },
    { 0x2d,0x77,0x5a,0x2d }, { 0x0f,0x11,0x1e,0x0f },
    { 0xb0,0xcb,0x7b,0xb0 }, { 0x54,0xfc,0xa8,0x54 },
    { 0xbb,0xd6,0x6d,0xbb }, { 0x16,0x3a,0x2c,0x16 }
};

static const byte T4[256][4] =
{
    { 0x63,0x63,0xa5,0xc6 }, { 0x7c,0x7c,0x84,0xf8 },
    { 0x77,0x77,0x99,0xee }, { 0x7b,0x7b,0x8d,0xf6 },
    { 0xf2,0xf2,0x0d,0xff }, { 0x6b,0x6b,0xbd,0xd6 },
    { 0x6f,0x6f,0xb1,0xde }, { 0xc5,0xc5,0x54,0x91 },
    { 0x30,0x30,0x50,0x60 }, { 0x01,0x01,0x03,0x02 },
    { 0x67,0x67,0xa9,0xce }, { 0x2b,0x2b,0x7d,0x56 },
    { 0xfe,0xfe,0x19,0xe7 }, { 0xd7,0xd7,0x62,0xb5 },
    { 0xab,0xab,0xe6,0x4d }, { 0x76,0x76,0x9a,0xec },
    { 0xca,0xca,0x45,0x8f }, { 0x82,0x82,0x9d,0x1f },
    { 0xc9,0xc9,0x40,0x89 }, { 0x7d,0x7d,0x87,0xfa },
    { 0xfa,0xfa,0x15,0xef }, { 0x59,0x59,0xeb,0xb2 },
    { 0x47,0x47,0xc9,0x8e }, { 0xf0,0xf0,0x0b,0xfb },
    { 0xad,0xad,0xec,0x41 }, { 0xd4,0xd4,0x67,0xb3 },
    { 0xa2,0xa2,0xfd,0x5f }, { 0xaf,0xaf,0xea,0x45 },
    { 0x9c,0x9c,0xbf,0x23 }, { 0xa4,0xa4,0xf7,0x53 },
    { 0x72,0x72,0x96,0xe4 }, { 0xc0,0xc0,0x5b,0x9b },
    { 0xb7,0xb7,0xc2,0x75 }, { 0xfd,0xfd,0x1c,0xe1 },
    { 0x93,0x93,0xae,0x3d }, { 0x26,0x26,0x6a,0x4c },
    { 0x36,0x36,0x5a,0x6c }, { 0x3f,0x3f,0x41,0x7e },
    { 0xf7,0xf7,0x02,0xf5 }, { 0xcc,0xcc,0x4f,0x83 },
    { 0x34,0x34,0x5c,0x68 }, { 0xa5,0xa5,0xf4,0x51 },
    { 0xe5,0xe5,0x34,0xd1 }, { 0xf1,0xf1,0x08,0xf9 },
    { 0x71,0x71,0x93,0xe2 }, { 0xd8,0xd8,0x73,0xab },
    { 0x31,0x31,0x53,0x62 }, { 0x15,0x15,0x3f,0x2a },
    { 0x04,0x04,0x0c,0x08 }, { 0xc7,0xc7,0x52,0x95 },
    { 0x23,0x23,0x65,0x46 }, { 0xc3,0xc3,0x5e,0x9d },
    { 0x18,0x18,0x28,0x30 }, { 0x96,0x96,0xa1,0x37 },
    { 0x05,0x05,0x0f,0x0a }, { 0x9a,0x9a,0xb5,0x2f },
    { 0x07,0x07,0x09,0x0e }, { 0x12,0x12,0x36,0x24 },
    { 0x80,0x80,0x9b,0x1b }, { 0xe2,0xe2,0x3d,0xdf },
    { 0xeb,0xeb,0x26,0xcd }, { 0x27,0x27,0x69,0x4e },
    { 0xb2,0xb2,0xcd,0x7f }, { 0x75,0x75,0x9f,0xea },
    { 0x09,0x09,0x1b,0x12 }, { 0x83,0x83,0x9e,0x1d },
    { 0x2c,0x2c,0x74,0x58 }, { 0x1a,0x1a,0x2e,0x34 },
    { 0x1b,0x1b,0x2d,0x36 }, { 0x6e,0x6e,0xb2,0xdc },
    { 0x5a,0x5a,0xee,0xb4 }, { 0xa0,0xa0,0xfb,0x5b },
    { 0x52,0x52,0xf6,0xa4 }, { 0x3b,0x3b,0x4d,0x76 },
    { 0xd6,0xd6,0x61,0xb7 }, { 0xb3,0xb3,0xce,0x7d },
    { 0x29,0x29,0x7b,0x52 }, { 0xe3,0xe3,0x3e,0xdd },
    { 0x2f,0x2f,0x71,0x5e }, { 0x84,0x84,0x97,0x13 },
    { 0x53,0x53,0xf5,0xa6 }, { 0xd1,0xd1,0x68,0xb9 },
    { 0x00,0x00,0x00,0x00 }, { 0xed,0xed,0x2c,0xc1 },
    { 0x20,0x20,0x60,0x40 }, { 0xfc,0xfc,0x1f,0xe3 },
    { 0xb1,0xb1,0xc8,0x79 }, { 0x5b,0x5b,0xed,0xb6 },
    { 0x6a,0x6a,0xbe,0xd4 }, { 0xcb,0xcb,0x46,0x8d },
    { 0xbe,0xbe,0xd9,0x67 }, { 0x39,0x39,0x4b,0x72 },
    { 0x4a,0x4a,0xde,0x94 }, { 0x4c,0x4c,0xd4,0x98 },
    { 0x58,0x58,0xe8,0xb0 }, { 0xcf,0xcf,0x4a,0x85 },
    { 0xd0,0xd0,0x6b,0xbb }, { 0xef,0xef,0x2a,0xc5 },
    { 0xaa,0xaa,0xe5,0x4f }, { 0xfb,0xfb,0x16,0xed },
    { 0x43,0x43,0xc5,0x86 }, { 0x4d,0x4d,0xd7,0x9a },
    { 0x33,0x33,0x55,0x66 }, { 0x85,0x85,0x94,0x11 },
    { 0x45,0x45,0xcf,0x8a }, { 0xf9,0xf9,0x10,0xe9 },
    { 0x02,0x02,0x06,0x04 }, { 0x7f,0x7f,0x81,0xfe },
    { 0x50,0x50,0xf0,0xa0 }, { 0x3c,0x3c,0x44,0x78 },
    { 0x9f,0x9f,0xba,0x25 }, { 0xa8,0xa8,0xe3,0x4b },
    { 0x51,0x51,0xf3,0xa2 }, { 0xa3,0xa3,0xfe,0x5d },
    { 0x40,0x40,0xc0,0x80 }, { 0x8f,0x8f,0x8a,0x05 },
    { 0x92,0x92,0xad,0x3f }, { 0x9d,0x9d,0xbc,0x21 },
    { 0x38,0x38,0x48,0x70 }, { 0xf5,0xf5,0x04,0xf1 },
    { 0xbc,0xbc,0xdf,0x63 }, { 0xb6,0xb6,0xc1,0x77 },
    { 0xda,0xda,0x75,0xaf }, { 0x21,0x21,0x63,0x42 },
    { 0x10,0x10,0x30,0x20 }, { 0xff,0xff,0x1a,0xe5 },
    { 0xf3,0xf3,0x0e,0xfd }, { 0xd2,0xd2,0x6d,0xbf },
    { 0xcd,0xcd,0x4c,0x81 }, { 0x0c,0x0c,0x14,0x18 },
    { 0x13,0x13,0x35,0x26 }, { 0xec,0xec,0x2f,0xc3 },
    { 0x5f,0x5f,0xe1,0xbe }, { 0x97,0x97,0xa2,0x35 },
    { 0x44,0x44,0xcc,0x88 }, { 0x17,0x17,0x39,0x2e },
    { 0xc4,0xc4,0x57,0x93 }, { 0xa7,0xa7,0xf2,0x55 },
    { 0x7e,0x7e,0x82,0xfc }, { 0x3d,0x3d,0x47,0x7a },
    { 0x64,0x64,0xac,0xc8 }, { 0x5d,0x5d,0xe7,0xba },
    { 0x19,0x19,0x2b,0x32 }, { 0x73,0x73,0x95,0xe6 },
    { 0x60,0x60,0xa0,0xc0 }, { 0x81,0x81,0x98,0x19 },
    { 0x4f,0x4f,0xd1,0x9e }, { 0xdc,0xdc,0x7f,0xa3 },
    { 0x22,0x22,0x66,0x44 }, { 0x2a,0x2a,0x7e,0x54 },
    { 0x90,0x90,0xab,0x3b }, { 0x88,0x88,0x83,0x0b },
    { 0x46,0x46,0xca,0x8c }, { 0xee,0xee,0x29,0xc7 },
    { 0xb8,0xb8,0xd3,0x6b }, { 0x14,0x14,0x3c,0x28 },
    { 0xde,0xde,0x79,0xa7 }, { 0x5e,0x5e,0xe2,0xbc },
    { 0x0b,0x0b,0x1d,0x16 }, { 0xdb,0xdb,0x76,0xad },
    { 0xe0,0xe0,0x3b,0xdb }, { 0x32,0x32,0x56,0x64 },
    { 0x3a,0x3a,0x4e,0x74 }, { 0x0a,0x0a,0x1e,0x14 },
    { 0x49,0x49,0xdb,0x92 }, { 0x06,0x06,0x0a,0x0c },
    { 0x24,0x24,0x6c,0x48 }, { 0x5c,0x5c,0xe4,0xb8 },
    { 0xc2,0xc2,0x5d,0x9f }, { 0xd3,0xd3,0x6e,0xbd },
    { 0xac,0xac,0xef,0x43 }, { 0x62,0x62,0xa6,0xc4 },
    { 0x91,0x91,0xa8,0x39 }, { 0x95,0x95,0xa4,0x31 },
    { 0xe4,0xe4,0x37,0xd3 }, { 0x79,0x79,0x8b,0xf2 },
    { 0xe7,0xe7,0x32,0xd5 }, { 0xc8,0xc8,0x43,0x8b },
    { 0x37,0x37,0x59,0x6e }, { 0x6d,0x6d,0xb7,0xda },
    { 0x8d,0x8d,0x8c,0x01 }, { 0xd5,0xd5,0x64,0xb1 },
    { 0x4e,0x4e,0xd2,0x9c }, { 0xa9,0xa9,0xe0,0x49 },
    { 0x6c,0x6c,0xb4,0xd8 }, { 0x56,0x56,0xfa,0xac },
    { 0xf4,0xf4,0x07,0xf3 }, { 0xea,0xea,0x25,0xcf },
    { 0x65,0x65,0xaf,0xca }, { 0x7a,0x7a,0x8e,0xf4 },
    { 0xae,0xae,0xe9,0x47 }, { 0x08,0x08,0x18,0x10 },
    { 0xba,0xba,0xd5,0x6f }, { 0x78,0x78,0x88,0xf0 },
    { 0x25,0x25,0x6f,0x4a }, { 0x2e,0x2e,0x72,0x5c },
    { 0x1c,0x1c,0x24,0x38 }, { 0xa6,0xa6,0xf1,0x57 },
    { 0xb4,0xb4,0xc7,0x73 }, { 0xc6,0xc6,0x51,0x97 },
    { 0xe8,0xe8,0x23,0xcb }, { 0xdd,0xdd,0x7c,0xa1 },
    { 0x74,0x74,0x9c,0xe8 }, { 0x1f,0x1f,0x21,0x3e },
    { 0x4b,0x4b,0xdd,0x96 }, { 0xbd,0xbd,0xdc,0x61 },
    { 0x8b,0x8b,0x86,0x0d }, { 0x8a,0x8a,0x85,0x0f },
    { 0x70,0x70,0x90,0xe0 }, { 0x3e,0x3e,0x42,0x7c },
    { 0xb5,0xb5,0xc4,0x71 }, { 0x66,0x66,0xaa,0xcc },
    { 0x48,0x48,0xd8,0x90 }, { 0x03,0x03,0x05,0x06 },
    { 0xf6,0xf6,0x01,0xf7 }, { 0x0e,0x0e,0x12,0x1c },
    { 0x61,0x61,0xa3,0xc2 }, { 0x35,0x35,0x5f,0x6a },
    { 0x57,0x57,0xf9,0xae }, { 0xb9,0xb9,0xd0,0x69 },
    { 0x86,0x86,0x91,0x17 }, { 0xc1,0xc1,0x58,0x99 },
    { 0x1d,0x1d,0x27,0x3a }, { 0x9e,0x9e,0xb9,0x27 },
    { 0xe1,0xe1,0x38,0xd9 }, { 0xf8,0xf8,0x13,0xeb },
    { 0x98,0x98,0xb3,0x2b }, { 0x11,0x11,0x33,0x22 },
    { 0x69,0x69,0xbb,0xd2 }, { 0xd9,0xd9,0x70,0xa9 },
    { 0x8e,0x8e,0x89,0x07 }, { 0x94,0x94,0xa7,0x33 },
    { 0x9b,0x9b,0xb6,0x2d }, { 0x1e,0x1e,0x22,0x3c },
    { 0x87,0x87,0x92,0x15 }, { 0xe9,0xe9,0x20,0xc9 },
    { 0xce,0xce,0x49,0x87 }, { 0x55,0x55,0xff,0xaa },
    { 0x28,0x28,0x78,0x50 }, { 0xdf,0xdf,0x7a,0xa5 },
    { 0x8c,0x8c,0x8f,0x03 }, { 0xa1,0xa1,0xf8,0x59 },
    { 0x89,0x89,0x80,0x09 }, { 0x0d,0x0d,0x17,0x1a },
    { 0xbf,0xbf,0xda,0x65 }, { 0xe6,0xe6,0x31,0xd7 },
    { 0x42,0x42,0xc6,0x84 }, { 0x68,0x68,0xb8,0xd0 },
    { 0x41,0x41,0xc3,0x82 }, { 0x99,0x99,0xb0,0x29 },
    { 0x2d,0x2d,0x77,0x5a }, { 0x0f,0x0f,0x11,0x1e },
    { 0xb0,0xb0,0xcb,0x7b }, { 0x54,0x54,0xfc,0xa8 },
    { 0xbb,0xbb,0xd6,0x6d }, { 0x16,0x16,0x3a,0x2c }
};

static const uint32_t rcon[30] =
{
    0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 0x36, 0x6c,
    0xd8, 0xab, 0x4d, 0x9a, 0x2f, 0x5e, 0xbc, 0x63, 0xc6, 0x97, 0x35,
    0x6a, 0xd4, 0xb3, 0x7d, 0xfa, 0xef, 0xc5, 0x91
};

static char *AES_SelfTest(void);

/* Perform the key setup.
 */
static boolean AES_SetKey(RIJNDAEL_context *ctx, const byte *key,
                          const unsigned keylen)
{
    int ROUNDS;
    byte k[MAXKC][4];
    int i,j, r, t, rconpointer = 0;
    byte tk[MAXKC][4];
    int KC;

    if( keylen == 128/8 )
    {
        ROUNDS = 10;
        KC = 4;
    }
    else if ( keylen == 192/8 )
    {
        ROUNDS = 12;
        KC = 6;
    }
    else if ( keylen == 256/8 )
    {
        ROUNDS = 14;
        KC = 8;
    }
    else
    {
	return false;
    }

    ctx->ROUNDS = ROUNDS;

    for (i = 0; i < keylen; i++)
    {
        k[i >> 2][i & 3] = key[i];
    }
#define W (ctx->keySched)

    for (j = KC-1; j >= 0; j--)
    {
        *((uint32_t*)tk[j]) = *((uint32_t*)k[j]);
    }
    r = 0;
    t = 0;
    /* copy values into round key array */
    for (j = 0; (j < KC) && (r < ROUNDS + 1); )
    {
        for (; (j < KC) && (t < 4); j++, t++)
        {
            W[r][t] = *((uint32_t*)tk[j]);
        }
        if (t == 4)
        {
            r++;
            t = 0;
        }
    }

    while (r < ROUNDS + 1)
    {
        /* while not enough round key material calculated */
        /* calculate new values */
        tk[0][0] ^= S[tk[KC-1][1]];
        tk[0][1] ^= S[tk[KC-1][2]];
        tk[0][2] ^= S[tk[KC-1][3]];
        tk[0][3] ^= S[tk[KC-1][0]];
        tk[0][0] ^= rcon[rconpointer++];

        if (KC != 8)
        {
            for (j = 1; j < KC; j++)
            {
                *((uint32_t*)tk[j]) ^= *((uint32_t*)tk[j-1]);
            }
        }
        else
        {
            for (j = 1; j < KC/2; j++)
            {
                *((uint32_t*)tk[j]) ^= *((uint32_t*)tk[j-1]);
            }
            tk[KC/2][0] ^= S[tk[KC/2 - 1][0]];
            tk[KC/2][1] ^= S[tk[KC/2 - 1][1]];
            tk[KC/2][2] ^= S[tk[KC/2 - 1][2]];
            tk[KC/2][3] ^= S[tk[KC/2 - 1][3]];
            for (j = KC/2 + 1; j < KC; j++)
            {
                *((uint32_t*)tk[j]) ^= *((uint32_t*)tk[j-1]);
            }
        }
        /* copy values into round key array */
        for (j = 0; (j < KC) && (r < ROUNDS + 1); )
        {
            for (; (j < KC) && (t < 4); j++, t++)
            {
                W[r][t] = *((uint32_t*)tk[j]);
            }
            if (t == 4)
            {
                r++;
                t = 0;
            }
        }
    }

#undef W
    return true;
}

/* Encrypt one block.  A and B need to be aligned on a 4 byte
   boundary.  A and B may be the same. */
static void AES_EncryptAligned(const RIJNDAEL_context *ctx,
                               byte *b, const byte *a)
{
#define rk (ctx->keySched)
    int ROUNDS = ctx->ROUNDS;
    int r;
    union
    {
        uint32_t u32;
        byte b[4];
    } temp[4];

    temp[0].u32 = *((uint32_t*)(a   )) ^ rk[0][0];
    temp[1].u32 = *((uint32_t*)(a+ 4)) ^ rk[0][1];
    temp[2].u32 = *((uint32_t*)(a+ 8)) ^ rk[0][2];
    temp[3].u32 = *((uint32_t*)(a+12)) ^ rk[0][3];
    *((uint32_t*)(b    )) = (*((uint32_t*)T1[temp[0].b[0]])
                           ^ *((uint32_t*)T2[temp[1].b[1]])
                           ^ *((uint32_t*)T3[temp[2].b[2]])
                           ^ *((uint32_t*)T4[temp[3].b[3]]));
    *((uint32_t*)(b + 4)) = (*((uint32_t*)T1[temp[1].b[0]])
                           ^ *((uint32_t*)T2[temp[2].b[1]])
                           ^ *((uint32_t*)T3[temp[3].b[2]])
                           ^ *((uint32_t*)T4[temp[0].b[3]]));
    *((uint32_t*)(b + 8)) = (*((uint32_t*)T1[temp[2].b[0]])
                           ^ *((uint32_t*)T2[temp[3].b[1]])
                           ^ *((uint32_t*)T3[temp[0].b[2]])
                           ^ *((uint32_t*)T4[temp[1].b[3]]));
    *((uint32_t*)(b +12)) = (*((uint32_t*)T1[temp[3].b[0]])
                           ^ *((uint32_t*)T2[temp[0].b[1]])
                           ^ *((uint32_t*)T3[temp[1].b[2]])
                           ^ *((uint32_t*)T4[temp[2].b[3]]));

    for (r = 1; r < ROUNDS-1; r++)
    {
        temp[0].u32 = *((uint32_t*)(b   )) ^ rk[r][0];
        temp[1].u32 = *((uint32_t*)(b+ 4)) ^ rk[r][1];
        temp[2].u32 = *((uint32_t*)(b+ 8)) ^ rk[r][2];
        temp[3].u32 = *((uint32_t*)(b+12)) ^ rk[r][3];

        *((uint32_t*)(b    )) = (*((uint32_t*)T1[temp[0].b[0]])
                               ^ *((uint32_t*)T2[temp[1].b[1]])
                               ^ *((uint32_t*)T3[temp[2].b[2]])
                               ^ *((uint32_t*)T4[temp[3].b[3]]));
        *((uint32_t*)(b + 4)) = (*((uint32_t*)T1[temp[1].b[0]])
                               ^ *((uint32_t*)T2[temp[2].b[1]])
                               ^ *((uint32_t*)T3[temp[3].b[2]])
                               ^ *((uint32_t*)T4[temp[0].b[3]]));
        *((uint32_t*)(b + 8)) = (*((uint32_t*)T1[temp[2].b[0]])
                               ^ *((uint32_t*)T2[temp[3].b[1]])
                               ^ *((uint32_t*)T3[temp[0].b[2]])
                               ^ *((uint32_t*)T4[temp[1].b[3]]));
        *((uint32_t*)(b +12)) = (*((uint32_t*)T1[temp[3].b[0]])
                               ^ *((uint32_t*)T2[temp[0].b[1]])
                               ^ *((uint32_t*)T3[temp[1].b[2]])
                               ^ *((uint32_t*)T4[temp[2].b[3]]));
    }

    /* Last round is special. */
    temp[0].u32 = *((uint32_t*)(b   )) ^ rk[ROUNDS-1][0];
    temp[1].u32 = *((uint32_t*)(b+ 4)) ^ rk[ROUNDS-1][1];
    temp[2].u32 = *((uint32_t*)(b+ 8)) ^ rk[ROUNDS-1][2];
    temp[3].u32 = *((uint32_t*)(b+12)) ^ rk[ROUNDS-1][3];
    b[ 0] = T1[temp[0].b[0]][1];
    b[ 1] = T1[temp[1].b[1]][1];
    b[ 2] = T1[temp[2].b[2]][1];
    b[ 3] = T1[temp[3].b[3]][1];
    b[ 4] = T1[temp[1].b[0]][1];
    b[ 5] = T1[temp[2].b[1]][1];
    b[ 6] = T1[temp[3].b[2]][1];
    b[ 7] = T1[temp[0].b[3]][1];
    b[ 8] = T1[temp[2].b[0]][1];
    b[ 9] = T1[temp[3].b[1]][1];
    b[10] = T1[temp[0].b[2]][1];
    b[11] = T1[temp[1].b[3]][1];
    b[12] = T1[temp[3].b[0]][1];
    b[13] = T1[temp[0].b[1]][1];
    b[14] = T1[temp[1].b[2]][1];
    b[15] = T1[temp[2].b[3]][1];
    *((uint32_t*)(b   )) ^= rk[ROUNDS][0];
    *((uint32_t*)(b+ 4)) ^= rk[ROUNDS][1];
    *((uint32_t*)(b+ 8)) ^= rk[ROUNDS][2];
    *((uint32_t*)(b+12)) ^= rk[ROUNDS][3];
#undef rk
}

static void AES_Encrypt(const RIJNDAEL_context *ctx,
                        byte *bx, const byte *ax)
{
    /* BX and AX are not necessary correctly aligned.  Thus we need to
       copy them here. */
    uint32_t a[4];
    uint32_t b[4];

    memcpy(a, ax, 16);
    AES_EncryptAligned(ctx, (byte *) b, (byte *) a);
    memcpy(bx, b, 16);
}

/* Test a single encryption and decryption with each key size. */

static char *AES_SelfTest(void)
{
    RIJNDAEL_context ctx;
    byte scratch[16];

    /* The test vectors are from the AES supplied ones; more or less
     * randomly taken from ecb_tbl.txt (I=42,81,14)
     */
    static const byte plaintext[16] = {
       0x01,0x4B,0xAF,0x22,0x78,0xA6,0x9D,0x33,
       0x1D,0x51,0x80,0x10,0x36,0x43,0xE9,0x9A
    };
    static const byte key[16] = {
        0xE8,0xE9,0xEA,0xEB,0xED,0xEE,0xEF,0xF0,
        0xF2,0xF3,0xF4,0xF5,0xF7,0xF8,0xF9,0xFA
    };
    static const byte ciphertext[16] = {
        0x67,0x43,0xC3,0xD1,0x51,0x9A,0xB4,0xF2,
        0xCD,0x9A,0x78,0xAB,0x09,0xA5,0x11,0xBD
    };

    static const byte plaintext_192[16] = {
        0x76,0x77,0x74,0x75,0xF1,0xF2,0xF3,0xF4,
        0xF8,0xF9,0xE6,0xE7,0x77,0x70,0x71,0x72
    };
    static const byte key_192[24] = {
        0x04,0x05,0x06,0x07,0x09,0x0A,0x0B,0x0C,
        0x0E,0x0F,0x10,0x11,0x13,0x14,0x15,0x16,
        0x18,0x19,0x1A,0x1B,0x1D,0x1E,0x1F,0x20
    };
    static const byte ciphertext_192[16] = {
        0x5D,0x1E,0xF2,0x0D,0xCE,0xD6,0xBC,0xBC,
        0x12,0x13,0x1A,0xC7,0xC5,0x47,0x88,0xAA
    };

    static const byte plaintext_256[16] = {
        0x06,0x9A,0x00,0x7F,0xC7,0x6A,0x45,0x9F,
        0x98,0xBA,0xF9,0x17,0xFE,0xDF,0x95,0x21
    };
    static const byte key_256[32] = {
        0x08,0x09,0x0A,0x0B,0x0D,0x0E,0x0F,0x10,
        0x12,0x13,0x14,0x15,0x17,0x18,0x19,0x1A,
        0x1C,0x1D,0x1E,0x1F,0x21,0x22,0x23,0x24,
        0x26,0x27,0x28,0x29,0x2B,0x2C,0x2D,0x2E
    };
    static const byte ciphertext_256[16] = {
        0x08,0x0E,0x95,0x17,0xEB,0x16,0x77,0x71,
        0x9A,0xCF,0x72,0x80,0x86,0x04,0x0A,0xE3
    };

    AES_SetKey(&ctx, key, sizeof(key));
    AES_Encrypt(&ctx, scratch, plaintext);
    if (memcmp (scratch, ciphertext, sizeof (ciphertext)))
        return "Rijndael-128 test encryption failed.";

    AES_SetKey(&ctx, key_192, sizeof(key_192));
    AES_Encrypt(&ctx, scratch, plaintext_192);
    if (memcmp (scratch, ciphertext_192, sizeof (ciphertext_192)))
        return "Rijndael-192 test encryption failed.";

    AES_SetKey(&ctx, key_256, sizeof(key_256));
    AES_Encrypt(&ctx, scratch, plaintext_256);
    if (memcmp (scratch, ciphertext_256, sizeof (ciphertext_256)))
        return "Rijndael-256 test encryption failed.";

    return NULL;
}

#ifndef TEST

static boolean prng_enabled = false;
static RIJNDAEL_context prng_context;
static uint32_t prng_input_counter;
static uint32_t prng_values[4];
static unsigned int prng_value_index = 0;

// Initialize Pseudo-RNG using the specified 128-bit key.

void PRNG_Start(prng_seed_t key)
{
    char *errormsg;

    errormsg = AES_SelfTest();

    if (errormsg != NULL)
    {
        I_Error("Failed to initialize PRNG: %s", errormsg);
    }

    AES_SetKey(&prng_context, key, sizeof(prng_seed_t));
    prng_value_index = 4;
    prng_input_counter = 0;
    prng_enabled = true;
}

void PRNG_Stop(void)
{
    prng_enabled = false;
}

// Generate a set of new PRNG values by encrypting a new block.

static void PRNG_Generate(void)
{
    byte input[16], output[16];
    unsigned int i;

    // Input for the cipher is a consecutively increasing 32-bit counter.

    for (i = 0; i < 4; ++i)
    {
        input[4*i] = prng_input_counter & 0xff;
        input[4*i + 1] = (prng_input_counter >> 8) & 0xff;
        input[4*i + 2] = (prng_input_counter >> 16) & 0xff;
        input[4*i + 3] = (prng_input_counter >> 24) & 0xff;
        ++prng_input_counter;
    }

    AES_Encrypt(&prng_context, output, input);

    for (i = 0; i < 4; ++i)
    {
        prng_values[i] = output[4*i]
                       | (output[4*i + 1] << 8)
                       | (output[4*i + 2] << 16)
                       | (output[4*i + 3] << 24);
    }

    prng_value_index = 0;
}

// Read a random 32-bit integer from the PRNG.

unsigned int PRNG_Random(void)
{
    unsigned int result;

    if (!prng_enabled)
    {
        return 0;
    }

    if (prng_value_index >= 4)
    {
        PRNG_Generate();
    }

    result = prng_values[prng_value_index];
    ++prng_value_index;

    return result;
}


#else /* #ifndef TEST */

int main(int argc, char *argv[])
{
    char *errormsg;

    errormsg = AES_SelfTest();

    if (errormsg == NULL)
    {
        printf("AES Self test passed.\n");
        return 0;
    }
    else
    {
        fprintf(stderr, "AES self test failed: %s\n", errormsg);
        return 1;
    }
}

#endif
