# coding=utf-8
from blueman.plugins.AppletPlugin import AppletPlugin
import dbus
from gi.repository import GLib
from blueman.gui.Notification import Notification
from blueman.Sdp import DIALUP_NET_SVCLASS_ID
from blueman.Functions import get_icon, composite_icon
import weakref
import logging


class ConnectionHandler:
    def __init__(self, parent, service, reply, err):
        self.parent = parent
        self.service = service
        self.reply = reply
        self.err = err
        self.rfcomm_dev = None
        self.timeout = None

        self.parent.bus.add_signal_receiver(self.on_interfaces_added, "InterfacesAdded",
                                            "org.freedesktop.DBus.ObjectManager")

        # for some reason these handlers take a reference and don't give it back
        #so i have to workaround :(
        w = weakref.ref(self)
        service.connect(reply_handler=lambda *args: w() and w().on_connect_reply(*args),
                        error_handler=lambda *args: w() and w().on_connect_error(*args))

    def __del__(self):
        logging.info("deleting")

    def on_connect_reply(self, rfcomm):
        self.rfcomm_dev = rfcomm
        self.timeout = GLib.timeout_add(10000, self.on_timeout)

    def on_connect_error(self, *args):
        self.err(*args)
        self.cleanup()

    def cleanup(self):
        if self.timeout:
            GLib.source_remove(self.timeout)
        self.parent.bus.remove_signal_receiver(self.on_interfaces_added, "InterfacesAdded",
                                               "org.freedesktop.DBus.ObjectManager")

        del self.service

    def on_mm_device_added(self, path, name="org.freedesktop.ModemManager"):
        logging.info(path)
        interface = "%s.Modem" % name
        props = self.parent.bus.call_blocking(name, path, "org.freedesktop.DBus.Properties", "GetAll", "s", [interface])

        try:
            drivers = props["Drivers"]
        except KeyError:
            drivers = [props["Driver"]]

        if self.rfcomm_dev and "bluetooth" in drivers and props["Device"] in self.rfcomm_dev:
            logging.info("It's our bluetooth modem!")

            modem = get_icon("modem", 24)
            blueman = get_icon("blueman", 48)

            icon = composite_icon(blueman, [(modem, 24, 24, 255)])

            Notification(_("Bluetooth Dialup"),
                         _("DUN connection on %s will now be available in Network Manager") % self.service.device['Alias'],
                         image_data=icon).show()

            self.reply(self.rfcomm_dev)
            self.cleanup()

    def on_interfaces_added(self, object_path, interfaces):
        if 'org.freedesktop.ModemManager1.Modem' in interfaces:
            self.on_mm_device_added(object_path, "org.freedesktop.ModemManager1")

    def on_timeout(self):
        self.timeout = None
        self.err(dbus.DBusException(_("Modem Manager did not support the connection")))
        self.cleanup()


class NMDUNSupport(AppletPlugin):
    __depends__ = ["StatusIcon", "DBusService"]
    __conflicts__ = ["PPPSupport"]
    __icon__ = "modem"
    __author__ = "Walmis"
    __description__ = _("Provides support for Dial Up Networking (DUN) with ModemManager and NetworkManager")
    __priority__ = 1

    def on_load(self):
        self.bus = dbus.SystemBus()

    def on_unload(self):
        pass

    def rfcomm_connect_handler(self, service, reply, err):
        if DIALUP_NET_SVCLASS_ID == service.short_uuid:
            ConnectionHandler(self, service, reply, err)
            return True
        else:
            return False
