/*
 * Copyright (c) 2005-2009 Laf-Widget Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Laf-Widget Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.lafwidget.text;

import java.awt.event.*;

import javax.swing.*;
import javax.swing.text.JTextComponent;

import org.jvnet.lafwidget.LafWidgetAdapter;
import org.jvnet.lafwidget.LafWidgetUtilities;

/**
 * Adds "select / deselect on Escape key press" behaviour on text components.
 * 
 * @author Kirill Grouchnikov
 * @since 2.1
 */
public class SelectOnEscapeWidget extends LafWidgetAdapter<JTextComponent> {
	/**
	 * The Escape key listener.
	 */
	protected KeyListener keyListener;

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.lafwidget.LafWidget#requiresCustomLafSupport()
	 */
	public boolean requiresCustomLafSupport() {
		return false;
	}

	@Override
	public void installUI() {
		super.installUI();

		InputMap currMap = SwingUtilities.getUIInputMap(this.jcomp,
				JComponent.WHEN_FOCUSED);

		InputMap newMap = new InputMap();
		if (currMap != null) {
			KeyStroke[] kss = currMap.allKeys();
			for (int i = 0; i < kss.length; i++) {
				KeyStroke stroke = kss[i];
				Object val = currMap.get(stroke);
				newMap.put(stroke, val);
			}
		}

		newMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0),
				"flipTextSelection");

		this.jcomp.getActionMap().put("flipTextSelection",
				new AbstractAction() {
					public void actionPerformed(ActionEvent e) {
						if (LafWidgetUtilities
								.hasTextFlipSelectOnEscapeProperty(jcomp)) {
							SwingUtilities.invokeLater(new Runnable() {
								public void run() {
									int selectionLength = jcomp
											.getSelectionEnd()
											- jcomp.getSelectionStart();
									if (selectionLength == 0) {
										jcomp.selectAll();
									} else {
										int lastPos = jcomp.getSelectionEnd();
										jcomp.setSelectionStart(0);
										jcomp.setSelectionEnd(0);
										jcomp.setCaretPosition(lastPos);
									}
								}
							});
						}
					}
				});

		SwingUtilities.replaceUIInputMap(this.jcomp, JComponent.WHEN_FOCUSED,
				newMap);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.lafwidget.LafWidgetAdapter#installListeners()
	 */
	@Override
	public void installListeners() {
		this.keyListener = new KeyAdapter() {
			@Override
			public void keyTyped(final KeyEvent e) {
				if (LafWidgetUtilities.hasTextFlipSelectOnEscapeProperty(jcomp)
						&& (e.getKeyChar() == KeyEvent.VK_ESCAPE)) {
					SwingUtilities.invokeLater(new Runnable() {
						public void run() {
							int selectionLength = jcomp.getSelectionEnd()
									- jcomp.getSelectionStart();
							if (selectionLength == 0) {
								jcomp.selectAll();
							} else {
								int lastPos = jcomp.getSelectionEnd();
								jcomp.setSelectionStart(0);
								jcomp.setSelectionEnd(0);
								jcomp.setCaretPosition(lastPos);
							}
							// dispatch to parent - especially useful
							// for text components in dialogs that should
							// be closed on pressing ESC.
							jcomp.getParent().dispatchEvent(e);
						}
					});
				}
			}
		};
		// this.jcomp.addKeyListener(this.keyListener);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.lafwidget.LafWidgetAdapter#uninstallListeners()
	 */
	@Override
	public void uninstallListeners() {
		this.keyListener = null;
		this.jcomp.removeKeyListener(this.keyListener);
	}
}
