/***************************************************************************

  video.c

  Functions to emulate the video hardware of the machine.

***************************************************************************/

#include "emu.h"
#include "machine/segacrpt.h"
#include "includes/senjyo.h"


/***************************************************************************

  Callbacks for the TileMap code

***************************************************************************/

TILE_GET_INFO_MEMBER(senjyo_state::get_fg_tile_info)
{
	UINT8 attr = m_fgcolorram[tile_index];
	int flags = (attr & 0x80) ? TILE_FLIPY : 0;

	if (m_is_senjyo && (tile_index & 0x1f) >= 32-8)
		flags |= TILE_FORCE_LAYER0;

	SET_TILE_INFO_MEMBER(0,
			m_fgvideoram[tile_index] + ((attr & 0x10) << 4),
			attr & 0x07,
			flags);
}

TILE_GET_INFO_MEMBER(senjyo_state::senjyo_bg1_tile_info)
{
	UINT8 code = m_bg1videoram[tile_index];

	SET_TILE_INFO_MEMBER(1,
			code,
			(code & 0x70) >> 4,
			0);
}

TILE_GET_INFO_MEMBER(senjyo_state::starforc_bg1_tile_info)
{
	/* Star Force has more tiles in bg1, so to get a uniform color code spread */
	/* they wired bit 7 of the tile code in place of bit 4 to get the color code */
	static const UINT8 colormap[8] = { 0, 2, 4, 6, 1, 3, 5, 7 };
	UINT8 code = m_bg1videoram[tile_index];

	SET_TILE_INFO_MEMBER(1,
			code,
			colormap[(code & 0xe0) >> 5],
			0);
}

TILE_GET_INFO_MEMBER(senjyo_state::get_bg2_tile_info)
{
	UINT8 code = m_bg2videoram[tile_index];

	SET_TILE_INFO_MEMBER(2,
			code,
			(code & 0xe0) >> 5,
			0);
}

TILE_GET_INFO_MEMBER(senjyo_state::get_bg3_tile_info)
{
	UINT8 code = m_bg3videoram[tile_index];

	SET_TILE_INFO_MEMBER(3,
			code,
			(code & 0xe0) >> 5,
			0);
}



/***************************************************************************

  Start the video hardware emulation.

***************************************************************************/

void senjyo_state::video_start()
{
	m_fg_tilemap = &machine().tilemap().create(m_gfxdecode, tilemap_get_info_delegate(FUNC(senjyo_state::get_fg_tile_info),this), TILEMAP_SCAN_ROWS, 8, 8, 32, 32);

	if (m_is_senjyo)
	{
		m_bg1_tilemap = &machine().tilemap().create(m_gfxdecode, tilemap_get_info_delegate(FUNC(senjyo_state::senjyo_bg1_tile_info),this), TILEMAP_SCAN_ROWS, 16, 16, 16, 32);
		m_bg2_tilemap = &machine().tilemap().create(m_gfxdecode, tilemap_get_info_delegate(FUNC(senjyo_state::get_bg2_tile_info),this),    TILEMAP_SCAN_ROWS, 16, 16, 16, 48);   /* only 16x32 used by Star Force */
		m_bg3_tilemap = &machine().tilemap().create(m_gfxdecode, tilemap_get_info_delegate(FUNC(senjyo_state::get_bg3_tile_info),this),    TILEMAP_SCAN_ROWS, 16, 16, 16, 56);   /* only 16x32 used by Star Force */
	}
	else
	{
		m_bg1_tilemap = &machine().tilemap().create(m_gfxdecode, tilemap_get_info_delegate(FUNC(senjyo_state::starforc_bg1_tile_info),this), TILEMAP_SCAN_ROWS, 16, 16, 16, 32);
		m_bg2_tilemap = &machine().tilemap().create(m_gfxdecode, tilemap_get_info_delegate(FUNC(senjyo_state::get_bg2_tile_info),this),      TILEMAP_SCAN_ROWS, 16, 16, 16, 32); /* only 16x32 used by Star Force */
		m_bg3_tilemap = &machine().tilemap().create(m_gfxdecode, tilemap_get_info_delegate(FUNC(senjyo_state::get_bg3_tile_info),this),      TILEMAP_SCAN_ROWS, 16, 16, 16, 32); /* only 16x32 used by Star Force */
	}

	m_fg_tilemap->set_transparent_pen(0);
	m_bg1_tilemap->set_transparent_pen(0);
	m_bg2_tilemap->set_transparent_pen(0);
	m_bg3_tilemap->set_transparent_pen(0);
	m_fg_tilemap->set_scroll_cols(32);
}



/***************************************************************************

  Memory handlers

***************************************************************************/

WRITE8_MEMBER(senjyo_state::senjyo_fgvideoram_w)
{
	m_fgvideoram[offset] = data;
	m_fg_tilemap->mark_tile_dirty(offset);
}
WRITE8_MEMBER(senjyo_state::senjyo_fgcolorram_w)
{
	m_fgcolorram[offset] = data;
	m_fg_tilemap->mark_tile_dirty(offset);
}
WRITE8_MEMBER(senjyo_state::senjyo_bg1videoram_w)
{
	m_bg1videoram[offset] = data;
	m_bg1_tilemap->mark_tile_dirty(offset);
}
WRITE8_MEMBER(senjyo_state::senjyo_bg2videoram_w)
{
	m_bg2videoram[offset] = data;
	m_bg2_tilemap->mark_tile_dirty(offset);
}
WRITE8_MEMBER(senjyo_state::senjyo_bg3videoram_w)
{
	m_bg3videoram[offset] = data;
	m_bg3_tilemap->mark_tile_dirty(offset);
}

WRITE8_MEMBER(senjyo_state::senjyo_bgstripes_w)
{
	*m_bgstripesram = data;
}

/***************************************************************************

  Display refresh

***************************************************************************/

void senjyo_state::draw_bgbitmap(bitmap_ind16 &bitmap,const rectangle &cliprect)
{
	int x,y,pen,strwid,count;


	if (m_bgstripes == 0xff) /* off */
		bitmap.fill(0, cliprect);
	else
	{
		int flip = flip_screen();

		pen = 0;
		count = 0;
		strwid = m_bgstripes;
		if (strwid == 0) strwid = 0x100;
		if (flip) strwid ^= 0xff;

		for (x = 0;x < 256;x++)
		{
			if (flip)
				for (y = 0;y < 256;y++)
					bitmap.pix16(y, 255 - x) = 384 + pen;
			else
				for (y = 0;y < 256;y++)
					bitmap.pix16(y, x) = 384 + pen;

			count += 0x10;
			if (count >= strwid)
			{
				pen = (pen + 1) & 0x0f;
				count -= strwid;
			}
		}
	}
}

void senjyo_state::draw_radar(bitmap_ind16 &bitmap,const rectangle &cliprect)
{
	int offs,x;

	for (offs = 0;offs < 0x400;offs++)
		for (x = 0;x < 8;x++)
			if (m_radarram[offs] & (1 << x))
			{
				int sx, sy;

				sx = (8 * (offs % 8) + x) + 256-64;
				sy = ((offs & 0x1ff) / 8) + 96;

				if (flip_screen())
				{
					sx = 255 - sx;
					sy = 255 - sy;
				}

				if (cliprect.contains(sx, sy))
					bitmap.pix16(sy, sx) = offs < 0x200 ? 512 : 513;
			}
}

void senjyo_state::draw_sprites(bitmap_ind16 &bitmap,const rectangle &cliprect,int priority)
{
	UINT8 *spriteram = m_spriteram;
	int offs;

	for (offs = m_spriteram.bytes() - 4; offs >= 0; offs -= 4)
	{
		int big,sx,sy,flipx,flipy;

		if (((spriteram[offs+1] & 0x30) >> 4) == priority)
		{
			if (m_is_senjyo) /* Senjyo */
				big = (spriteram[offs] & 0x80);
			else    /* Star Force */
				big = ((spriteram[offs] & 0xc0) == 0xc0);
			sx = spriteram[offs+3];
			if (big)
				sy = 224-spriteram[offs+2];
			else
				sy = 240-spriteram[offs+2];
			flipx = spriteram[offs+1] & 0x40;
			flipy = spriteram[offs+1] & 0x80;

			if (flip_screen())
			{
				flipx = !flipx;
				flipy = !flipy;

				if (big)
				{
					sx = 224 - sx;
					sy = 226 - sy;
				}
				else
				{
					sx = 240 - sx;
					sy = 242 - sy;
				}
			}


			m_gfxdecode->gfx(big ? 5 : 4)->transpen(bitmap,cliprect,
					spriteram[offs],
					spriteram[offs + 1] & 0x07,
					flipx,flipy,
					sx,sy,0);
		}
	}
}

UINT32 senjyo_state::screen_update_senjyo(screen_device &screen, bitmap_ind16 &bitmap, const rectangle &cliprect)
{
	int i;


	/* two colors for the radar dots (verified on the real board) */
	m_palette->set_pen_color(512,rgb_t(0xff,0x00,0x00));  /* red for enemies */
	m_palette->set_pen_color(513,rgb_t(0xff,0xff,0x00));  /* yellow for player */

	{
		int flip = flip_screen();
		int scrollx,scrolly;

		for (i = 0;i < 32;i++)
			m_fg_tilemap->set_scrolly(i, m_fgscroll[i]);

		scrollx = m_scrollx1[0];
		scrolly = m_scrolly1[0] + 256 * m_scrolly1[1];
		if (flip)
			scrollx = -scrollx;
		m_bg1_tilemap->set_scrollx(0, scrollx);
		m_bg1_tilemap->set_scrolly(0, scrolly);

		scrollx = m_scrollx2[0];
		scrolly = m_scrolly2[0] + 256 * m_scrolly2[1];
		if (m_scrollhack)   /* Star Force, but NOT the encrypted version */
		{
			scrollx = m_scrollx1[0];
			scrolly = m_scrolly1[0] + 256 * m_scrolly1[1];
		}
		if (flip)
			scrollx = -scrollx;
		m_bg2_tilemap->set_scrollx(0, scrollx);
		m_bg2_tilemap->set_scrolly(0, scrolly);

		scrollx = m_scrollx3[0];
		scrolly = m_scrolly3[0] + 256 * m_scrolly3[1];
		if (flip)
			scrollx = -scrollx;
		m_bg3_tilemap->set_scrollx(0, scrollx);
		m_bg3_tilemap->set_scrolly(0, scrolly);
	}

	draw_bgbitmap(bitmap, cliprect);
	draw_sprites(bitmap, cliprect, 0);
	m_bg3_tilemap->draw(screen, bitmap, cliprect, 0, 0);
	draw_sprites(bitmap, cliprect, 1);
	m_bg2_tilemap->draw(screen, bitmap, cliprect, 0, 0);
	draw_sprites(bitmap, cliprect, 2);
	m_bg1_tilemap->draw(screen, bitmap, cliprect, 0, 0);
	draw_sprites(bitmap, cliprect, 3);
	m_fg_tilemap->draw(screen, bitmap, cliprect, 0, 0);
	draw_radar(bitmap, cliprect);

#if 0
{
	char baf[80];
	UINT8 *senjyo_scrolly3 = m_scrolly3;

	sprintf(baf,"%02x %02x %02x %02x %02x %02x %02x %02x",
		senjyo_scrolly3[0x00],
		senjyo_scrolly3[0x01],
		senjyo_scrolly3[0x02],
		senjyo_scrolly3[0x03],
		senjyo_scrolly3[0x04],
		senjyo_scrolly3[0x05],
		senjyo_scrolly3[0x06],
		senjyo_scrolly3[0x07]);
	ui_draw_text(baf,0,0);
	sprintf(baf,"%02x %02x %02x %02x %02x %02x %02x %02x",
		senjyo_scrolly3[0x08],
		senjyo_scrolly3[0x09],
		senjyo_scrolly3[0x0a],
		senjyo_scrolly3[0x0b],
		senjyo_scrolly3[0x0c],
		senjyo_scrolly3[0x0d],
		senjyo_scrolly3[0x0e],
		senjyo_scrolly3[0x0f]);
	ui_draw_text(baf,0,10);
	sprintf(baf,"%02x %02x %02x %02x %02x %02x %02x %02x",
		senjyo_scrolly3[0x10],
		senjyo_scrolly3[0x11],
		senjyo_scrolly3[0x12],
		senjyo_scrolly3[0x13],
		senjyo_scrolly3[0x14],
		senjyo_scrolly3[0x15],
		senjyo_scrolly3[0x16],
		senjyo_scrolly3[0x17]);
	ui_draw_text(baf,0,20);
	sprintf(baf,"%02x %02x %02x %02x %02x %02x %02x %02x",
		senjyo_scrolly3[0x18],
		senjyo_scrolly3[0x19],
		senjyo_scrolly3[0x1a],
		senjyo_scrolly3[0x1b],
		senjyo_scrolly3[0x1c],
		senjyo_scrolly3[0x1d],
		senjyo_scrolly3[0x1e],
		senjyo_scrolly3[0x1f]);
	ui_draw_text(baf,0,30);
}
#endif
	return 0;
}
