#include "udm_config.h"

#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>


#ifdef HAVE_WINSOCK_H
#include <winsock.h>
#endif
#ifdef HAVE_IO_H   /* for Win */
#include <io.h>
#endif
#ifdef HAVE_PROCESS_H
#include <process.h>
#endif

#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif
#ifdef   HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif
#ifdef HAVE_MMAN_H
#include <sys/mman.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_NETINET_IN_SYSTM_H
#include <netinet/in_systm.h>
#endif
#ifdef HAVE_NETINET_IP_H
#include <netinet/ip.h>
#endif
#ifdef HAVE_NETINET_TCP_H
#include <netinet/tcp.h>
#endif
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif



#include "udm_common.h"
#include "udm_cache.h"
#include "udm_utils.h"
#include "udm_db.h"

#ifndef INADDR_NONE
#define INADDR_NONE ((unsigned long) -1)
#endif

/*
#define DEBUG_LOGD 1
*/

/*
#define DEBUG_LOGD_CMD 1
*/

#define STATE_CMD 1
#define STATE_WRD 2
#define MAXCLIENT 128
#define MAX_LOG_SIZE	1024*1024*30 /* 30 Mb */

typedef struct {
	int  fd;	/* Socket descriptor    */
	size_t  nbytes;	/* Bytes in buffer      */
	size_t  rbytes;	/* Required bytes       */
	size_t  mbytes;	/* Allocated bytes      */
	int  state;	/* Client state         */
	UDM_LOGD_CMD last_cmd;
	union {
		char * buf;	/* Buffer for data   */
		UDM_LOGD_CMD * cmd;
		UDM_LOGD_WRD * wrd;
	} data;	
} UDM_LOGD_CL;

#ifdef DEBUG_LOGD_CMD
void printcmd(int id,UDM_LOGD_CMD *c){
	fprintf(stderr,"cmd=%d nwords=%d stamp=%d url_id=%d\n",c->cmd,c->nwords,(int)c->stamp,c->url_id);
}
#endif

static char time_pid[100];
static char * time_pid_info(void){
	struct tm * tim;
	time_t t;
	t=time(NULL);
	tim=localtime(&t);
	strftime(time_pid,sizeof(time_pid),"%a %d %T",tim);
	sprintf(time_pid+strlen(time_pid)," [%d]",(int)getpid());
	return(time_pid);
}

static void init_client(UDM_LOGD_CL * client, int n){
	int i;
	for(i=0;i<n;i++){
		client[i].fd=0;
		client[i].state=0;
		client[i].nbytes=0;
		client[i].rbytes=0;
		client[i].mbytes=1;
		client[i].data.buf=(char*)malloc(1);
	}
}

static int del_fd=0;
static int wrd_fd=0;
static size_t del_size=0;
static size_t wrd_size=0;

static char pidname[]= UDM_VAR_DIR UDMSLASHSTR "cachelogd.pid";
static const char raw_dir[]=UDM_VAR_DIR UDMSLASHSTR "raw" UDMSLASHSTR;
static const char del_log_name[]=UDM_VAR_DIR UDMSLASHSTR "raw" UDMSLASHSTR "del.log";
static const char wrd_log_name[]=UDM_VAR_DIR UDMSLASHSTR "raw" UDMSLASHSTR "wrd.log";

static void close_logs(void){
	if(del_fd){
		close(del_fd);
		del_fd=0;
	}
	if(wrd_fd){
		close(wrd_fd);
		wrd_fd=0;
	}
}

static void exitproc(void){
	close_logs();
	unlink(pidname);
#if (WIN32|WINNT)
	WSACleanup();
#endif
}


static void open_logs(void){
	int flags=O_RDWR|O_CREAT|O_APPEND|UDM_BINARY;
	int perm=UDM_IWRITE;
	if((del_fd=open(del_log_name,flags,perm))<0){
		fprintf(stderr,"%s Can't open '%s' for writing\n",time_pid_info(),del_log_name);
		exit(1);
	}
	if((wrd_fd=open(wrd_log_name,flags,perm))<0){
		fprintf(stderr,"%s Can't open '%s' for writing\n",time_pid_info(),wrd_log_name);
		exit(1);
	}
	del_size=lseek(del_fd,(off_t)0,SEEK_END);
	wrd_size=lseek(wrd_fd,(off_t)0,SEEK_END);
	fprintf(stderr,"%s Open logs %d %d\n",time_pid_info(),del_size,wrd_size);
}

static void rotate_logs(void){
	char to[BUFSIZ];
	time_t stamp;

	fprintf(stderr,"%s Renaming logs: del_size=%d wrd_size=%d\n",time_pid_info(),del_size,wrd_size);
	if(del_size||wrd_size){
		
		close_logs();

		/* Rename logs */
		stamp=time(NULL);
		sprintf(to,"%s%d.wrd",raw_dir,(int)stamp);
		if(rename(wrd_log_name,to)!=0){
			fprintf(stderr,"%s Can't rename %s: %s\n",time_pid_info(),wrd_log_name,strerror(errno));
			exit(1);
		}
		sprintf(to,"%s%d.del",raw_dir,(int)stamp);
		if(rename(del_log_name,to)){
			fprintf(stderr,"%s Can't rename %s: %s\n",time_pid_info(),del_log_name,strerror(errno));
			exit(1);
		}

		open_logs();
	}else{
		fprintf(stderr,"%s No renaming required\n",time_pid_info());
	}
}


static int store_doc(UDM_LOGD_CL *client){
	UDM_LOGDEL  logdel;
	size_t wrote,nbytes;

	/* Write to del.log */
	logdel.stamp=client->last_cmd.stamp;
	logdel.url_id=client->last_cmd.url_id;
	nbytes=sizeof(logdel);
	wrote=write(del_fd,&logdel,nbytes);
	if(wrote!=nbytes){
		fprintf(stderr,"%s Can't write to del.log: %s\n",time_pid_info(),strerror(errno));
		exit(1);
	}
	del_size+=nbytes;
	if(wrd_size>=MAX_LOG_SIZE){
		fprintf(stderr,"%s Word log size became more than %dMb. Forse renaming.\n",time_pid_info(),MAX_LOG_SIZE/1024/1024);
		rotate_logs();
	}

	if(client->last_cmd.nwords==0)return 0;

	/* Write to word log */	
	/* Write command information */
	nbytes=sizeof(UDM_LOGD_CMD);
	wrote=write(wrd_fd,&client->last_cmd,nbytes);
	if(wrote!=nbytes){
		fprintf(stderr,"%s Can't write to wrd.log: %s\n",time_pid_info(),strerror(errno));
		exit(1);
	}
	wrd_size+=nbytes;

	/* Write words information */
	nbytes=client->last_cmd.nwords*sizeof(UDM_LOGD_WRD);
	wrote=write(wrd_fd,client->data.wrd,nbytes);
	if(wrote!=nbytes){
		fprintf(stderr,"%s Can't write to wrd.log: %s\n",time_pid_info(),strerror(errno));
		exit(1);
	}
	wrd_size+=nbytes;

	return 0;
}

/********************* SIG Handlers ****************/
static void sighandler(int sign);

static void init_signals(void){
	/* Set up signals handler*/
#if (WIN32|WINNT)
#else
	signal(SIGPIPE,sighandler);
	signal(SIGHUP,sighandler);
#endif
	signal(SIGINT,sighandler);
	signal(SIGTERM,sighandler);
}

static void sighandler(int sign){

	fprintf(stderr,"%s Signal %d arrived\n",time_pid_info(),sign);
	switch(sign){
#if (WIN32|WINNT)
#else
		case SIGPIPE:
			fprintf(stderr,"%s FATAL: This should not ever happen! Broken pipe!\n",time_pid_info());
			break;
		case SIGHUP:
			rotate_logs();
			break;
#endif
		default:
			fprintf(stderr,"%s Shutdown\n",time_pid_info());
			exit(0);
	}
	init_signals();
}

/*************************************************************/


static int client_action(int id,UDM_LOGD_CL * client){

	switch(client->state){
		
		case STATE_CMD:
			client->last_cmd=*(client->data.cmd);
#ifdef DEBUG_LOGD_CMD
			printcmd(id,client->data.cmd);
#endif
			if(client->last_cmd.nwords>0){
				client->state=STATE_WRD;
				client->rbytes=client->last_cmd.nwords*sizeof(UDM_LOGD_WRD);
			}else{
				store_doc(client);
				client->state=STATE_CMD;
				client->rbytes=sizeof(UDM_LOGD_CMD);
			}
			break;
		case STATE_WRD:
			/*
			switch(fork()){
				case 0:	store_doc(client);
					exit(0);
					break;
				case -1:fprintf(stderr,"%s Can't fork: '%s'\n",time_pid_info(),strerror(errno));
					break;
			}
			*/
			store_doc(client);
			client->state=STATE_CMD;
			client->rbytes=sizeof(UDM_LOGD_CMD);
			break;
	}
	return 0;
}


static void usage(void){

	fprintf(stderr,
"\n\
cachelogd from %s\n\
http://search.mnogo.ru (C) 1998-2001, LavTech Corp.\n\
\n\
Usage: cachelogd [OPTIONS]\n\
\n\
Options are:\n\
  -p xxx        listen port xxx\n\
  -h,-?         print this help page and exit\n\
\n\
Please mail bug reports and suggestions to <udmsearch@search.udm.net>.\n",UdmVersion());

	return;
}

	

int main(int argc,char **argv){
	int flags=O_RDWR|O_CREAT|O_APPEND|UDM_BINARY;
	int perm=UDM_IWRITE;
	struct	sockaddr_in server_addr;
	struct	sockaddr_in his_addr;
	struct	in_addr bind_address;

	int on=1,i;
	int ctl_sock, sel;
	struct timeval tval;

	int  pid_fd,ch,port=UDM_LOGD_PORT;
	char pidbuf[1024];
	fd_set mask;
	UDM_LOGD_CL log_client[MAXCLIENT];


	while ((ch = getopt(argc, argv, "p:?")) != -1){
		switch (ch) {
			case 'p':
				port=atoi(optarg);
				break;
			case 'h':
			case '?':
			default:
				usage();
				return 1;
				break;
		}
	}
	argc -= optind;argv += optind;

	if(argc>0){
		usage();
		return 1;
	}

	
#if (WIN32|WINNT)
	{
		WSADATA wsaData;

		if(WSAStartup(0x101,&wsaData)!=0){
			fprintf(stderr,"%s WSAStartup() error %d\n",time_pid_info(),WSAGetLastError);
			exit(1);
		}
	}
#endif


	/* Check that another instance isn't running */
	/* and create PID file.                       */

	pid_fd=open(pidname,flags,perm);
	if(pid_fd<0){
		fprintf(stderr,"%s Can't create '%s': %s\n",time_pid_info(),pidname,strerror(errno));
		if(errno==EEXIST){
			fprintf(stderr,"It seems that another cachelogd is already running!\n");
			fprintf(stderr,"Remove '%s' if it is not true.\n",pidname);
		}
		exit(1);
	}
	sprintf(pidbuf,"%d\n",(int)getpid());
	write(pid_fd,&pidbuf,strlen(pidbuf));
	close(pid_fd);

	/* Initialize variables */

	atexit(&exitproc);
	init_client(log_client,MAXCLIENT);
	open_logs();
	init_signals();

	

	fprintf(stderr,"%s cachelogd started on port %d. Accepting %d connections.\n",time_pid_info(),port,sizeof(mask));

	ctl_sock = socket(AF_INET, SOCK_STREAM, 0);
	if (ctl_sock < 0) {
		fprintf(stderr,"%s socket() error %d\n",time_pid_info(),errno);
		unlink(pidname);
		exit(1);
	}
	if (setsockopt(ctl_sock, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on)) != 0){
		fprintf(stderr,"%s setsockopt() error %d\n",time_pid_info(),errno);
		unlink(pidname);
		exit(1);
	}


	/* Prepare to start TCP server */
	
	bind_address.s_addr 	= htonl(INADDR_ANY);
	server_addr.sin_family	= AF_INET;
	server_addr.sin_addr	= bind_address;
	server_addr.sin_port	= ntohs(port);
	
	if (bind(ctl_sock, (struct sockaddr *)&server_addr, sizeof(server_addr))) {
		fprintf(stderr,"%s bind() error %d %s\n",time_pid_info(),errno,strerror(errno));
		unlink(pidname);
		exit(1);
	}
	if (listen(ctl_sock, 32) < 0) {
		fprintf(stderr,"%s listen() error %d %s\n",time_pid_info(),errno,strerror(errno));
		unlink(pidname);
		exit(1);
	}

	FD_ZERO(&mask);
	FD_SET(ctl_sock,&mask);

	
	while (1) {
		fd_set msk;
		
		tval.tv_sec  		= 300;
		tval.tv_usec 		= 0;
		msk=mask;
		sel=select(32,&msk,0,0,&tval);
		
		if(sel==0){
			/* Time limit expired */
 			continue;
		}
		if(sel<1){
			/* FIXME: add errno checking */
			continue;
		}
		
		/* Check whether new client has connected */
		if(FD_ISSET(ctl_sock,&msk)){
			int cl,addrlen;
			addrlen = sizeof(his_addr);
			for(cl=0;cl<MAXCLIENT;cl++){
				if(log_client[cl].fd==0){
					fprintf(stderr,"%s Client #%d connected\n",time_pid_info(),cl);
					log_client[cl].fd=accept(ctl_sock, (struct sockaddr *)&his_addr, &addrlen);
					FD_SET(log_client[cl].fd,&mask);
					log_client[cl].state=STATE_CMD;
					log_client[cl].rbytes=sizeof(UDM_LOGD_CMD);
					break;
				}
			}
		}
		
		
		/* Process all ready clients */
		for(i=0;i<MAXCLIENT;i++){
			if(FD_ISSET(log_client[i].fd,&msk)){
				UDM_LOGD_CL * cl;
				int lbytes;
				
				cl=&log_client[i];
#ifdef DEBUG_LOGD
				printf("[%d] try read %d r=%d n=%d\n",i,cl->rbytes-cl->nbytes,cl->rbytes,cl->nbytes);
#endif

				if((cl->mbytes)!=(cl->rbytes)){
					cl->data.buf=(char*)realloc(cl->data.buf,cl->rbytes);
					cl->mbytes=cl->rbytes;
				}

				lbytes=recv(cl->fd,&cl->data.buf[cl->nbytes],cl->rbytes-cl->nbytes,0);
#ifdef DEBUG_LOGD
				printf("[%d] read %d bytes n=%d\n",i,lbytes,cl->nbytes);
#endif				
				if(lbytes){
					cl->nbytes+=lbytes;
#ifdef DEBUG_LOGD
					printf("[%d] AAA n[i]=%d r[i]=%d\n",i,cl->nbytes,cl->rbytes);
#endif					
					if(cl->nbytes==cl->rbytes){
#ifdef DEBUG_LOGD
						printf("[%d] struct read %d bytes\n",i,cl->nbytes);
#endif
						client_action(i,cl);
						cl->nbytes=0;
					}
				}else{
					fprintf(stderr,"%s Client #%d left\n",time_pid_info(),i);
					FD_CLR(cl->fd,&mask);
					closesocket(cl->fd);
					UDM_FREE(cl->data.buf);
					init_client(cl,1);
				}
			}
		}
	}
}
