/*
 * Copyright (c) 2014, 2019, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2.0,
 * as published by the Free Software Foundation.
 *
 * This program is also distributed with certain software (including
 * but not limited to OpenSSL) that is licensed under separate terms, as
 * designated in a particular file or component or in included license
 * documentation.  The authors of MySQL hereby grant you an additional
 * permission to link the program and your derivative works with the
 * separately licensed software that they have included with MySQL.
 * This program is distributed in the hope that it will be useful,  but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 * the GNU General Public License, version 2.0, for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "modules/devapi/mod_mysqlx_resultset.h"
#include <memory>
#include <utility>
#include "modules/devapi/base_constants.h"
#include "mysqlshdk/include/shellcore/base_shell.h"
#include "mysqlshdk/libs/db/charset.h"
#include "mysqlshdk/libs/db/row_copy.h"
#include "mysqlshdk/libs/db/session.h"
#include "mysqlshdk/libs/utils/strformat.h"
#include "mysqlxtest_utils.h"
#include "scripting/common.h"
#include "scripting/obj_date.h"
#include "shellcore/utils_help.h"

using shcore::Value;
using std::placeholders::_1;

namespace mysqlsh {
namespace mysqlx {

// -----------------------------------------------------------------------

// Documentation of BaseResult class
REGISTER_HELP_CLASS(BaseResult, mysqlx);
REGISTER_HELP(
    BASERESULT_BRIEF,
    "Base class for the different types of results returned by the server.");

BaseResult::BaseResult(std::shared_ptr<mysqlshdk::db::mysqlx::Result> result)
    : _result(result) {
  add_property("affectedItemsCount", "getAffectedItemsCount");
  add_property("executionTime", "getExecutionTime");
  add_property("warningCount", "getWarningCount");
  add_property("warningsCount", "getWarningsCount");
  add_property("warnings", "getWarnings");
}

BaseResult::~BaseResult() {}

// Documentation of getWarnings function
REGISTER_HELP_PROPERTY(warnings, BaseResult);
REGISTER_HELP(BASERESULT_WARNINGS_BRIEF, "Same as <<<getWarnings>>>");
REGISTER_HELP_FUNCTION(getWarnings, BaseResult);
REGISTER_HELP(BASERESULT_GETWARNINGS_BRIEF,
              "Retrieves the warnings generated by the executed operation.");
REGISTER_HELP(
    BASERESULT_GETWARNINGS_RETURNS,
    "@returns A list containing a warning object for each generated warning.");
REGISTER_HELP(BASERESULT_GETWARNINGS_DETAIL,
              "This is the same value than C API mysql_warning_count, see "
              "https://dev.mysql.com/doc/refman/en/"
              "mysql-warning-count.html");
REGISTER_HELP(BASERESULT_GETWARNINGS_DETAIL1,
              "Each warning object contains a key/value pair describing the "
              "information related to a specific warning.");
REGISTER_HELP(BASERESULT_GETWARNINGS_DETAIL2,
              "This information includes: Level, Code and Message.");

/**
 * $(BASERESULT_GETWARNINGS_BRIEF)
 *
 * $(BASERESULT_GETWARNINGS_RETURNS)
 *
 * $(BASERESULT_GETWARNINGS_DETAIL)
 *
 * $(BASERESULT_GETWARNINGS_DETAIL1)
 *
 * $(BASERESULT_GETWARNINGS_DETAIL2)
 */
#if DOXYGEN_JS
List BaseResult::getWarnings() {}
#elif DOXYGEN_PY
list BaseResult::get_warnings() {}
#endif

shcore::Value BaseResult::get_member(const std::string &prop) const {
  shcore::Value ret_val;

  if (prop == "affectedItemsCount") {
    ret_val = Value(get_affected_items_count());
  } else if (prop == "executionTime") {
    return shcore::Value(get_execution_time());
  } else if (prop == "warningCount") {
    log_warning("'%s' is deprecated, use '%s' instead.",
                get_function_name("warningCount").c_str(),
                get_function_name("warningsCount").c_str());
    ret_val = Value(get_warnings_count());
  } else if (prop == "warningsCount") {
    ret_val = Value(get_warnings_count());
  } else if (prop == "warnings") {
    std::shared_ptr<shcore::Value::Array_type> array(
        new shcore::Value::Array_type);
    if (_result) {
      while (std::unique_ptr<mysqlshdk::db::Warning> warning =
                 _result->fetch_one_warning()) {
        mysqlsh::Row *warning_row = new mysqlsh::Row();
        switch (warning->level) {
          case mysqlshdk::db::Warning::Level::Note:
            warning_row->add_item("level", shcore::Value("Note"));
            break;
          case mysqlshdk::db::Warning::Level::Warn:
            warning_row->add_item("level", shcore::Value("Warning"));
            break;
          case mysqlshdk::db::Warning::Level::Error:
            warning_row->add_item("level", shcore::Value("Error"));
            break;
        }
        warning_row->add_item("code", shcore::Value(warning->code));
        warning_row->add_item("message", shcore::Value(warning->msg));

        array->push_back(shcore::Value::wrap(warning_row));
      }
    }
    ret_val = shcore::Value(array);
  } else {
    ret_val = ShellBaseResult::get_member(prop);
  }

  return ret_val;
}

// Documentation of getAffectedItemCount function
REGISTER_HELP_PROPERTY(affectedItemsCount, BaseResult);
REGISTER_HELP(BASERESULT_AFFECTEDITEMSCOUNT_BRIEF,
              "Same as <<<getAffectedItemsCount>>>");

REGISTER_HELP_FUNCTION(getAffectedItemsCount, BaseResult);
REGISTER_HELP(BASERESULT_GETAFFECTEDITEMSCOUNT_BRIEF,
              "The the number of affected items for the last operation.");
REGISTER_HELP(BASERESULT_GETAFFECTEDITEMSCOUNT_RETURNS,
              "@returns the number of affected items.");
REGISTER_HELP(BASERESULT_GETAFFECTEDITEMSCOUNT_DETAIL,
              "Returns the number of records affected by the executed "
              "operation");

/**
 * $(BASERESULT_GETAFFECTEDITEMSCOUNT_BRIEF)
 *
 * $(BASERESULT_GETAFFECTEDITEMSCOUNT_RETURNS)
 *
 * $(BASERESULT_GETAFFECTEDITEMSCOUNT_DETAIL)
 */
#if DOXYGEN_JS
Integer BaseResult::getAffectedItemsCount() {}
#elif DOXYGEN_PY
int BaseResult::get_affected_items_count() {}
#endif

int64_t BaseResult::get_affected_items_count() const {
  if (!_result) return -1;
  return _result->get_affected_row_count();
}

// Documentation of getExecutionTime function
REGISTER_HELP_PROPERTY(executionTime, BaseResult);
REGISTER_HELP(BASERESULT_EXECUTIONTIME_BRIEF, "Same as <<<getExecutionTime>>>");
REGISTER_HELP_FUNCTION(getExecutionTime, BaseResult);
REGISTER_HELP(BASERESULT_GETEXECUTIONTIME_BRIEF,
              "Retrieves a string value indicating the execution time of the "
              "executed operation.");

/**
 * $(BASERESULT_GETEXECUTIONTIME_BRIEF)
 */
#if DOXYGEN_JS
String BaseResult::getExecutionTime() {}
#elif DOXYGEN_PY
str BaseResult::get_execution_time() {}
#endif

std::string BaseResult::get_execution_time() const {
  return mysqlshdk::utils::format_seconds(_result->get_execution_time());
}

// Documentation of getWarningCount function
REGISTER_HELP_PROPERTY(warningCount, BaseResult);
REGISTER_HELP(BASERESULT_WARNINGCOUNT_BRIEF, "Same as <<<getWarningCount>>>");
REGISTER_HELP(BASERESULT_WARNINGCOUNT_DETAIL,
              "${BASERESULT_WARNINGCOUNT_DEPRECATED}");
REGISTER_HELP(BASERESULT_WARNINGCOUNT_DEPRECATED,
              "@attention This property will be removed in a future release, "
              "use the <b><<<warningsCount>>></b> property instead.");

REGISTER_HELP_FUNCTION(getWarningCount, BaseResult);
REGISTER_HELP(BASERESULT_GETWARNINGCOUNT_BRIEF,
              "The number of warnings produced by the last "
              "statement execution.");
REGISTER_HELP(BASERESULT_GETWARNINGCOUNT_RETURNS,
              "@returns the number of warnings.");
REGISTER_HELP(BASERESULT_GETWARNINGCOUNT_DETAIL,
              "This is the same value than C API mysql_warning_count, see "
              "https://dev.mysql.com/doc/refman/en/"
              "mysql-warning-count.html");
REGISTER_HELP(BASERESULT_GETWARNINGCOUNT_DETAIL1,
              "See <<<getWarnings>>>() for more details.");
REGISTER_HELP(BASERESULT_GETWARNINGCOUNT_DETAIL2,
              "${BASERESULT_GETWARNINGCOUNT_DEPRECATED}");
REGISTER_HELP(BASERESULT_GETWARNINGCOUNT_DEPRECATED,
              "@attention This function will be removed in a future release, "
              "use the <b><<<getWarningsCount>>></b> function instead.");

/**
 * $(BASERESULT_GETWARNINGCOUNT_BRIEF)
 *
 * $(BASERESULT_GETWARNINGCOUNT_RETURNS)
 *
 * $(BASERESULT_GETWARNINGCOUNT_DETAIL)
 *
 */
#if DOXYGEN_JS
/**
 * @attention This function will be removed in a future release, use the&nbsp;
 * <b>getWarningsCount</b> function instead.
 */
#elif DOXYGEN_PY
/**
 * @attention This function will be removed in a future release, use the&nbsp;
 * <b>get_warnings_count</b> function instead.
 */
#endif
/**
 * \sa warnings
 */
#if DOXYGEN_JS
Integer BaseResult::getWarningCount() {}
#elif DOXYGEN_PY
int BaseResult::get_warning_count() {}
#endif

// Documentation of getWarningCount function
REGISTER_HELP_PROPERTY(warningsCount, BaseResult);
REGISTER_HELP(BASERESULT_WARNINGSCOUNT_BRIEF, "Same as <<<getWarningsCount>>>");

REGISTER_HELP_FUNCTION(getWarningsCount, BaseResult);
REGISTER_HELP(BASERESULT_GETWARNINGSCOUNT_BRIEF,
              "The number of warnings produced by the last statement "
              "execution.");
REGISTER_HELP(BASERESULT_GETWARNINGSCOUNT_RETURNS,
              "@returns the number of warnings.");
REGISTER_HELP(BASERESULT_GETWARNINGSCOUNT_DETAIL,
              "This is the same value than C API mysql_warning_count, see "
              "https://dev.mysql.com/doc/refman/en/"
              "mysql-warning-count.html");
REGISTER_HELP(BASERESULT_GETWARNINGSCOUNT_DETAIL1,
              "See <<<getWarnings>>>() for more details.");

/**
 * $(BASERESULT_GETWARNINGSCOUNT_BRIEF)
 *
 * $(BASERESULT_GETWARNINGSCOUNT_RETURNS)
 *
 * $(BASERESULT_GETWARNINGSCOUNT_DETAIL)
 *
 * \sa warnings
 */
#if DOXYGEN_JS
Integer BaseResult::getWarningsCount() {}
#elif DOXYGEN_PY
int BaseResult::get_warnings_count() {}
#endif

uint64_t BaseResult::get_warnings_count() const {
  if (_result) return _result->get_warning_count();
  return 0;
}

void BaseResult::append_json(shcore::JSON_dumper &dumper) const {
  bool create_object = (dumper.deep_level() == 0);

  if (create_object) dumper.start_object();

  dumper.append_value("executionTime", get_member("executionTime"));
  dumper.append_value("affectedItemsCount", get_member("affectedItemsCount"));

  if (mysqlsh::current_shell_options()->get().show_warnings) {
    dumper.append_value("warningCount", get_member("warningsCount"));
    dumper.append_value("warningsCount", get_member("warningsCount"));
    dumper.append_value("warnings", get_member("warnings"));
  }

  if (create_object) dumper.end_object();
}

// -----------------------------------------------------------------------

// Documentation of Result class
REGISTER_HELP_SUB_CLASS(Result, mysqlx, BaseResult);
REGISTER_HELP(RESULT_BRIEF,
              "Allows retrieving information about non query operations "
              "performed on the database.");
REGISTER_HELP(RESULT_DETAIL,
              "An instance of this class will be returned on the CRUD "
              "operations that change the content of the database:");
REGISTER_HELP(RESULT_DETAIL1, "@li On Table: insert, update and delete");
REGISTER_HELP(RESULT_DETAIL2, "@li On Collection: add, modify and remove");
REGISTER_HELP(RESULT_DETAIL3,
              "Other functions on the Session class also return an "
              "instance of this class:");
REGISTER_HELP(RESULT_DETAIL4, "@li Transaction handling functions");
REGISTER_HELP(RESULT_DETAIL5, "@li Transaction handling functions");

Result::Result(std::shared_ptr<mysqlshdk::db::mysqlx::Result> result)
    : BaseResult(result) {
  add_property("affectedItemCount", "getAffectedItemCount");
  add_property("autoIncrementValue", "getAutoIncrementValue");
  add_property("generatedIds", "getGeneratedIds");
}

shcore::Value Result::get_member(const std::string &prop) const {
  Value ret_val;

  if (prop == "affectedItemCount") {
    ret_val = Value(get_affected_items_count());
    log_warning("'%s' is deprecated, use '%s' instead.",
                get_function_name("affectedItemCount").c_str(),
                get_function_name("affectedItemsCount").c_str());
  } else if (prop == "autoIncrementValue") {
    ret_val = Value(get_auto_increment_value());
  } else if (prop == "generatedIds") {
    shcore::Value::Array_type_ref ret_val(new shcore::Value::Array_type);
    std::vector<std::string> doc_ids = get_generated_ids();

    for (auto doc_id : doc_ids) ret_val->push_back(shcore::Value(doc_id));

    return shcore::Value(ret_val);
  } else {
    ret_val = BaseResult::get_member(prop);
  }

  return ret_val;
}

// Documentation of getAffectedItemCount function
REGISTER_HELP_PROPERTY(affectedItemCount, Result);
REGISTER_HELP(RESULT_AFFECTEDITEMCOUNT_BRIEF,
              "Same as <<<getAffectedItemCount>>>");
REGISTER_HELP(RESULT_AFFECTEDITEMCOUNT_DETAIL,
              "${RESULT_AFFECTEDITEMCOUNT_DEPRECATED}");
REGISTER_HELP(RESULT_AFFECTEDITEMCOUNT_DEPRECATED,
              "@attention This property will be removed in a future release, "
              "use the <b><<<affectedItemsCount>>></b> property instead.");
REGISTER_HELP_FUNCTION(getAffectedItemCount, Result);
REGISTER_HELP(RESULT_GETAFFECTEDITEMCOUNT_BRIEF,
              "The the number of affected items for the last operation.");
REGISTER_HELP(RESULT_GETAFFECTEDITEMCOUNT_RETURNS,
              "@returns the number of affected items.");
REGISTER_HELP(RESULT_GETAFFECTEDITEMCOUNT_DETAIL,
              "This is the value of the C API mysql_affected_rows(), see "
              "https://dev.mysql.com/doc/refman/en/"
              "mysql-affected-rows.html");
REGISTER_HELP(RESULT_GETAFFECTEDITEMCOUNT_DETAIL1,
              "${RESULT_GETAFFECTEDITEMCOUNT_DEPRECATED}");
REGISTER_HELP(RESULT_GETAFFECTEDITEMCOUNT_DEPRECATED,
              "@attention This function will be removed in a future release, "
              "use the <b><<<getAffectedItemsCount>>></b> function instead.");

/**
 * $(RESULT_GETAFFECTEDITEMCOUNT_BRIEF)
 *
 * $(RESULT_GETAFFECTEDITEMCOUNT_RETURNS)
 *
 * $(RESULT_GETAFFECTEDITEMCOUNT_DETAIL)
 *
 */
#if DOXYGEN_JS
/**
 * @attention This function will be removed in a future release, use the
 * &nbsp;<b>getAffectedItemsCount</b> function instead.
 */
#elif DOXYGEN_PY
/**
 * @attention This function will be removed in a future release, use the
 * &nbsp;<b>get_affected_items_count</b> function instead.
 */
#endif
/**
 */
#if DOXYGEN_JS
Integer Result::getAffectedItemCount() {}
#elif DOXYGEN_PY
int Result::get_affected_item_count() {}
#endif

// Documentation of getAutoIncrementValue function
REGISTER_HELP_PROPERTY(autoIncrementValue, Result);
REGISTER_HELP(RESULT_AUTOINCREMENTVALUE_BRIEF,
              "Same as <<<getAutoIncrementValue>>>");
REGISTER_HELP_FUNCTION(getAutoIncrementValue, Result);
REGISTER_HELP(RESULT_GETAUTOINCREMENTVALUE_BRIEF,
              "The last insert id auto generated (from an insert operation)");
REGISTER_HELP(RESULT_GETAUTOINCREMENTVALUE_RETURNS,
              "@returns the integer representing the last insert id");
REGISTER_HELP(RESULT_GETAUTOINCREMENTVALUE_DETAIL,
              "For more details, see "
              "https://dev.mysql.com/doc/refman/en/"
              "information-functions.html#function_last-insert-id");
REGISTER_HELP(RESULT_GETAUTOINCREMENTVALUE_DETAIL1,
              "Note that this value will be available only when the result is "
              "for a Table.insert operation.");

/**
 * $(RESULT_GETAUTOINCREMENTVALUE_BRIEF)
 *
 * $(RESULT_GETAUTOINCREMENTVALUE_RETURNS)
 *
 * $(RESULT_GETAUTOINCREMENTVALUE_DETAIL)
 *
 * $(RESULT_GETAUTOINCREMENTVALUE_DETAIL1)
 */
#if DOXYGEN_JS
Integer Result::getAutoIncrementValue() {}
#elif DOXYGEN_PY
int Result::get_auto_increment_value() {}
#endif

int64_t Result::get_auto_increment_value() const {
  if (_result) return _result->get_auto_increment_value();
  return 0;
}

// Documentation of getLastDocumentId function
REGISTER_HELP_PROPERTY(generatedIds, Result);
REGISTER_HELP_FUNCTION(getGeneratedIds, Result);
REGISTER_HELP(RESULT_GENERATEDIDS_BRIEF, "Same as <<<getGeneratedIds>>>.");
REGISTER_HELP(RESULT_GETGENERATEDIDS_BRIEF,
              "Returns the list of document ids generated on the server.");
REGISTER_HELP(RESULT_GETGENERATEDIDS_RETURNS,
              "@returns a list of strings containing the generated ids.");
REGISTER_HELP(
    RESULT_GETGENERATEDIDS_DETAIL,
    "When adding documents into a collection, it is required that an ID is "
    "associated to the document, if a document is added without an '_id' "
    "field, an error will be generated.");
REGISTER_HELP(
    RESULT_GETGENERATEDIDS_DETAIL1,
    "At MySQL 8.0.11 if the documents being added do not have an '_id' field, "
    "the server will automatically generate an ID and assign it to the "
    "document.");
REGISTER_HELP(
    RESULT_GETGENERATEDIDS_DETAIL2,
    "This function returns a list of the IDs that were generated for the "
    "server to satisfy this requirement.");
/**
 * $(RESULT_GETGENERATEDIDS_BRIEF)
 *
 * $(RESULT_GETGENERATEDIDS_RETURNS)
 *
 * $(RESULT_GETGENERATEDIDS_DETAIL)
 *
 * $(RESULT_GETGENERATEDIDS_DETAIL1)
 *
 * $(RESULT_GETGENERATEDIDS_DETAIL2)
 */
#if DOXYGEN_JS
List Result::getGeneratedIds(){};
#elif DOXYGEN_PY
list Result::get_generated_ids(){};
#endif
const std::vector<std::string> Result::get_generated_ids() const {
  if (_result)
    return _result->get_generated_ids();
  else
    return {};
}

void Result::append_json(shcore::JSON_dumper &dumper) const {
  dumper.start_object();

  BaseResult::append_json(dumper);

  dumper.append_value("affectedItemCount", get_member("affectedItemsCount"));
  dumper.append_value("autoIncrementValue", get_member("autoIncrementValue"));
  dumper.append_value("generatedIds", get_member("generatedIds"));

  dumper.end_object();
}

// -----------------------------------------------------------------------

// Documentation of DocResult class
REGISTER_HELP_SUB_CLASS(DocResult, mysqlx, BaseResult);
REGISTER_HELP(DOCRESULT_BRIEF,
              "Allows traversing the DbDoc objects returned by a "
              "Collection.find operation.");

DocResult::DocResult(std::shared_ptr<mysqlshdk::db::mysqlx::Result> result)
    : BaseResult(result) {
  add_method("fetchOne", std::bind(&DocResult::fetch_one, this, _1));
  add_method("fetchAll", std::bind(&DocResult::fetch_all, this, _1));
}

// Documentation of fetchOne function
REGISTER_HELP_FUNCTION(fetchOne, DocResult);
REGISTER_HELP(DOCRESULT_FETCHONE_BRIEF,
              "Retrieves the next DbDoc on the DocResult.");
REGISTER_HELP(
    DOCRESULT_FETCHONE_RETURNS,
    "@returns A DbDoc object representing the next Document in the result.");

/**
 * $(DOCRESULT_FETCHONE_BRIEF)
 *
 * $(DOCRESULT_FETCHONE_RETURNS)
 */
#if DOXYGEN_JS
Document DocResult::fetchOne() {}
#elif DOXYGEN_PY
Document DocResult::fetch_one() {}
#endif
shcore::Value DocResult::fetch_one(const shcore::Argument_list &args) const {
  Value ret_val = Value::Null();

  args.ensure_count(0, get_function_name("fetchOne").c_str());

  try {
    if (_result) {
      if (const mysqlshdk::db::IRow *r = _result->fetch_one()) {
        ret_val = Value::parse(r->get_string(0));
      }
    }
  }
  CATCH_AND_TRANSLATE_FUNCTION_EXCEPTION(get_function_name("fetchOne"));

  return ret_val;
}

// Documentation of fetchAll function
REGISTER_HELP_FUNCTION(fetchAll, DocResult);
REGISTER_HELP(DOCRESULT_FETCHALL_BRIEF,
              "Returns a list of DbDoc objects which contains an element for "
              "every unread document.");
REGISTER_HELP(DOCRESULT_FETCHALL_RETURNS, "@returns A List of DbDoc objects.");
REGISTER_HELP(DOCRESULT_FETCHALL_DETAIL,
              "If this function is called right after executing a query, it "
              "will return a DbDoc for every document on the resultset.");
REGISTER_HELP(DOCRESULT_FETCHALL_DETAIL1,
              "If fetchOne is called before this function, when this function "
              "is called it will return a DbDoc for each of the remaining "
              "documents on the resultset.");

/**
 * $(DOCRESULT_FETCHALL_BRIEF)
 *
 * $(DOCRESULT_FETCHALL_RETURNS)
 *
 * $(DOCRESULT_FETCHALL_DETAIL)
 *
 * $(DOCRESULT_FETCHALL_DETAIL1)
 */
#if DOXYGEN_JS
List DocResult::fetchAll() {}
#elif DOXYGEN_PY
list DocResult::fetch_all() {}
#endif
shcore::Value DocResult::fetch_all(const shcore::Argument_list &args) const {
  Value::Array_type_ref array(new Value::Array_type());

  args.ensure_count(0, get_function_name("fetchAll").c_str());

  // Gets the next document
  Value record = fetch_one(args);
  while (record) {
    array->push_back(record);
    record = fetch_one(args);
  }

  return Value(array);
}

shcore::Value DocResult::get_metadata() const {
  if (!_metadata) {
    shcore::Value data_type = mysqlsh::Constant::get_constant(
        "mysqlx", "Type", "JSON", shcore::Argument_list());

    const mysqlshdk::db::Column &column_md(_result->get_metadata()[0]);

    _metadata = shcore::Value::wrap(new mysqlsh::Column(column_md, data_type));
  }

  return _metadata;
}

void DocResult::append_json(shcore::JSON_dumper &dumper) const {
  dumper.start_object();

  dumper.append_value("documents", fetch_all(shcore::Argument_list()));

  BaseResult::append_json(dumper);

  dumper.end_object();
}

// -----------------------------------------------------------------------

// Documentation of RowResult class
REGISTER_HELP_SUB_CLASS(RowResult, mysqlx, BaseResult);
REGISTER_HELP(
    ROWRESULT_BRIEF,
    "Allows traversing the Row objects returned by a Table.select operation.");

RowResult::RowResult(std::shared_ptr<mysqlshdk::db::mysqlx::Result> result)
    : BaseResult(result) {
  add_property("columnCount", "getColumnCount");
  add_property("columns", "getColumns");
  add_property("columnNames", "getColumnNames");

  add_method("fetchOne", std::bind(&RowResult::fetch_one, this, _1));
  add_method("fetchAll", std::bind(&RowResult::fetch_all, this, _1));
  expose("fetchOneObject", &RowResult::_fetch_one_object);

  _column_names.reset(new std::vector<std::string>());
  for (auto &cmd : _result->get_metadata())
    _column_names->push_back(cmd.get_column_label());
}

shcore::Value RowResult::get_member(const std::string &prop) const {
  Value ret_val;
  if (prop == "columnCount") {
    ret_val = shcore::Value(get_column_count());
  } else if (prop == "columnNames") {
    std::shared_ptr<shcore::Value::Array_type> array(
        new shcore::Value::Array_type);
    for (auto &column : *_column_names) {
      array->push_back(shcore::Value(column));
    }

    ret_val = shcore::Value(array);
  } else if (prop == "columns")
    ret_val = shcore::Value(get_columns());
  else {
    ret_val = BaseResult::get_member(prop);
  }
  return ret_val;
}

// Documentation of getColumnCount function
REGISTER_HELP_PROPERTY(columnCount, RowResult);
REGISTER_HELP(ROWRESULT_COLUMNCOUNT_BRIEF, "Same as <<<getColumnCount>>>");
REGISTER_HELP_FUNCTION(getColumnCount, RowResult);
REGISTER_HELP(ROWRESULT_GETCOLUMNCOUNT_BRIEF,
              "Retrieves the number of columns on the current result.");
REGISTER_HELP(ROWRESULT_GETCOLUMNCOUNT_RETURNS,
              "@returns the number of columns on the current result.");

/**
 * $(ROWRESULT_GETCOLUMNCOUNT_BRIEF)
 *
 * $(ROWRESULT_GETCOLUMNCOUNT_RETURNS)
 */
#if DOXYGEN_JS
Integer RowResult::getColumnCount() {}
#elif DOXYGEN_PY
int RowResult::get_column_count() {}
#endif
int64_t RowResult::get_column_count() const {
  return _result->get_metadata().size();
}

// Documentation of getColumnNames function
REGISTER_HELP_PROPERTY(columnNames, RowResult);
REGISTER_HELP(ROWRESULT_COLUMNNAMES_BRIEF, "Same as <<<getColumnNames>>>");
REGISTER_HELP_FUNCTION(getColumnNames, RowResult);
REGISTER_HELP(ROWRESULT_GETCOLUMNNAMES_BRIEF,
              "Gets the columns on the current result.");
REGISTER_HELP(ROWRESULT_GETCOLUMNNAMES_RETURNS,
              "@returns A list with the names of the columns returned on the "
              "active result.");

/**
 * $(ROWRESULT_GETCOLUMNNAMES_BRIEF)
 *
 * $(ROWRESULT_GETCOLUMNNAMES_RETURNS)
 */
#if DOXYGEN_JS
List RowResult::getColumnNames() {}
#elif DOXYGEN_PY
list RowResult::get_column_names() {}
#endif

// Documentation of getColumns function
REGISTER_HELP_PROPERTY(columns, RowResult);
REGISTER_HELP(ROWRESULT_COLUMNS_BRIEF, "Same as <<<getColumns>>>");
REGISTER_HELP_FUNCTION(getColumns, RowResult);
REGISTER_HELP(ROWRESULT_GETCOLUMNS_BRIEF,
              "Gets the column metadata for the columns on the active result.");
REGISTER_HELP(ROWRESULT_GETCOLUMNS_RETURNS,
              "@returns a list of Column objects containing information about "
              "the columns included on the active result.");

/**
 * $(ROWRESULT_GETCOLUMNS_BRIEF)
 *
 * $(ROWRESULT_GETCOLUMNS_RETURNS)
 */
#if DOXYGEN_JS
List RowResult::getColumns() {}
#elif DOXYGEN_PY
list RowResult::get_columns() {}
#endif
shcore::Value::Array_type_ref RowResult::get_columns() const {
  if (!_columns) {
    _columns.reset(new shcore::Value::Array_type);

    for (auto &column_meta : _result->get_metadata()) {
      std::string type_name;
      switch (column_meta.get_type()) {
        case mysqlshdk::db::Type::Null:
          break;
        case mysqlshdk::db::Type::String:
          type_name = "STRING";
          break;
        case mysqlshdk::db::Type::Integer:
        case mysqlshdk::db::Type::UInteger:
          type_name = "INT";
          switch (column_meta.get_length()) {
            case 3:
            case 4:
              type_name = "TINYINT";
              break;
            case 5:
            case 6:
              type_name = "SMALLINT";
              break;
            case 8:
            case 9:
              type_name = "MEDIUMINT";
              break;
            case 10:
            case 11:
              type_name = "INT";
              break;
            case 20:
              type_name = "BIGINT";
              break;
          }
          break;
        case mysqlshdk::db::Type::Float:
          type_name = "FLOAT";
          break;
        case mysqlshdk::db::Type::Double:
          type_name = "DOUBLE";
          break;
        case mysqlshdk::db::Type::Decimal:
          type_name = "DECIMAL";
          break;
        case mysqlshdk::db::Type::Bytes:
          type_name = "BYTES";
          break;
        case mysqlshdk::db::Type::Geometry:
          type_name = "GEOMETRY";
          break;
        case mysqlshdk::db::Type::Json:
          type_name = "JSON";
          break;
        case mysqlshdk::db::Type::DateTime:
          type_name = "DATETIME";
          break;
        case mysqlshdk::db::Type::Date:
          type_name = "DATE";
          break;
        case mysqlshdk::db::Type::Time:
          type_name = "TIME";
          break;
        case mysqlshdk::db::Type::Bit:
          type_name = "BIT";
          break;
        case mysqlshdk::db::Type::Enum:
          type_name = "ENUM";
          break;
        case mysqlshdk::db::Type::Set:
          type_name = "SET";
          break;
      }
      assert(!type_name.empty());
      shcore::Value data_type = mysqlsh::Constant::get_constant(
          "mysqlx", "Type", type_name, shcore::Argument_list());
      assert(data_type);

      _columns->push_back(
          shcore::Value::wrap(new mysqlsh::Column(column_meta, data_type)));
    }
  }

  return _columns;
}

// Documentation of fetchOne function
REGISTER_HELP_FUNCTION(fetchOne, RowResult);
REGISTER_HELP(ROWRESULT_FETCHONE_BRIEF,
              "Retrieves the next Row on the RowResult.");
REGISTER_HELP(
    ROWRESULT_FETCHONE_RETURNS,
    "@returns A Row object representing the next record on the result.");

/**
 * $(ROWRESULT_FETCHONE_BRIEF)
 *
 * $(ROWRESULT_FETCHONE_RETURNS)
 */
#if DOXYGEN_JS
Row RowResult::fetchOne() {}
#elif DOXYGEN_PY
Row RowResult::fetch_one() {}
#endif
shcore::Value RowResult::fetch_one(const shcore::Argument_list &args) const {
  shcore::Value ret_val;
  args.ensure_count(0, get_function_name("fetchOne").c_str());

  try {
    auto row = fetch_one_row();
    if (row) {
      ret_val = shcore::Value::wrap(row.release());
    }
  }
  CATCH_AND_TRANSLATE_FUNCTION_EXCEPTION(get_function_name("fetchOne"));

  return ret_val;
}

REGISTER_HELP_FUNCTION(fetchOneObject, RowResult);
REGISTER_HELP(ROWRESULT_FETCHONEOBJECT_BRIEF,
              "${BASERESULT_FETCHONEOBJECT_BRIEF}");
REGISTER_HELP(ROWRESULT_FETCHONEOBJECT_RETURNS,
              "${BASERESULT_FETCHONEOBJECT_RETURNS}");
REGISTER_HELP(ROWRESULT_FETCHONEOBJECT_DETAIL,
              "${BASERESULT_FETCHONEOBJECT_DETAIL}");
/**
 * $(BASERESULT_FETCHONEOBJECT_BRIEF)
 *
 * $(BASERESULT_FETCHONEOBJECT)
 */
#if DOXYGEN_JS
Dictionary RowResult::fetchOneObject() {}
#elif DOXYGEN_PY
dict RowResult::fetch_one_object() {}
#endif
shcore::Dictionary_t RowResult::_fetch_one_object() {
  return ShellBaseResult::fetch_one_object();
}

// Documentation of fetchAll function
REGISTER_HELP_FUNCTION(fetchAll, RowResult);
REGISTER_HELP(ROWRESULT_FETCHALL_BRIEF,
              "Returns a list of DbDoc objects which contains an element for "
              "every unread document.");
REGISTER_HELP(ROWRESULT_FETCHALL_RETURNS, "@returns A List of DbDoc objects.");

/**
 * $(ROWRESULT_FETCHALL_BRIEF)
 *
 * $(ROWRESULT_FETCHALL_RETURNS)
 */
#if DOXYGEN_JS
List RowResult::fetchAll() {}
#elif DOXYGEN_PY
list RowResult::fetch_all() {}
#endif
shcore::Value RowResult::fetch_all(const shcore::Argument_list &args) const {
  Value::Array_type_ref array(new Value::Array_type());

  args.ensure_count(0, get_function_name("fetchAll").c_str());

  // Gets the next row
  Value record = fetch_one(args);
  while (record) {
    array->push_back(record);
    record = fetch_one(args);
  }

  return Value(array);
}

void RowResult::append_json(shcore::JSON_dumper &dumper) const {
  bool create_object = (dumper.deep_level() == 0);

  if (create_object) dumper.start_object();

  BaseResult::append_json(dumper);

  dumper.append_value("rows", fetch_all(shcore::Argument_list()));

  if (create_object) dumper.end_object();
}

// Documentation of SqlResult class
REGISTER_HELP_SUB_CLASS(SqlResult, mysqlx, RowResult);
REGISTER_HELP(SQLRESULT_BRIEF,
              "Allows browsing through the result information after performing "
              "an operation on the database done through Session.sql");

SqlResult::SqlResult(std::shared_ptr<mysqlshdk::db::mysqlx::Result> result)
    : RowResult(result) {
  add_method("hasData", std::bind(&SqlResult::has_data, this, _1));
  add_method("nextDataSet", std::bind(&SqlResult::next_data_set, this, _1));
  add_method("nextResult", std::bind(&SqlResult::next_result, this, _1));
  add_property("autoIncrementValue", "getAutoIncrementValue");
  add_property("affectedRowCount", "getAffectedRowCount");
}

// Documentation of getAutoIncrementValue function
REGISTER_HELP_PROPERTY(autoIncrementValue, SqlResult);
REGISTER_HELP(SQLRESULT_AUTOINCREMENTVALUE_BRIEF,
              "Same as <<<getAutoIncrementValue>>>");
REGISTER_HELP_FUNCTION(getAutoIncrementValue, SqlResult);
REGISTER_HELP(SQLRESULT_GETAUTOINCREMENTVALUE_BRIEF,
              "Returns the identifier for the last record inserted.");
REGISTER_HELP(SQLRESULT_GETAUTOINCREMENTVALUE_DETAIL,
              "Note that this value will only be set if the executed statement "
              "inserted a record in the database and an ID was automatically "
              "generated.");

/**
 * $(SQLRESULT_GETAUTOINCREMENTVALUE_BRIEF)
 *
 * $(SQLRESULT_GETAUTOINCREMENTVALUE_DETAIL)
 */
#if DOXYGEN_JS
Integer SqlResult::getAutoIncrementValue() {}
#elif DOXYGEN_PY
int SqlResult::get_auto_increment_value() {}
#endif
int64_t SqlResult::get_auto_increment_value() const {
  if (_result) return _result->get_auto_increment_value();
  return 0;
}

// Documentation of getAffectedRowCount function
REGISTER_HELP_PROPERTY(affectedRowCount, SqlResult);
REGISTER_HELP(SQLRESULT_AFFECTEDROWCOUNT_BRIEF,
              "Same as <<<getAffectedRowCount>>>");
REGISTER_HELP(SQLRESULT_AFFECTEDROWCOUNT_DETAIL,
              "${SQLRESULT_AFFECTEDROWCOUNT_DEPRECATED}");
REGISTER_HELP(SQLRESULT_AFFECTEDROWCOUNT_DEPRECATED,
              "@attention This property will be removed in a future release, "
              "use the <b><<<affectedItemsCount>>></b> property instead.");
REGISTER_HELP_FUNCTION(getAffectedRowCount, SqlResult);
REGISTER_HELP(SQLRESULT_GETAFFECTEDROWCOUNT_BRIEF,
              "Returns the number of rows affected by the executed query.");
REGISTER_HELP(SQLRESULT_GETAFFECTEDROWCOUNT_DETAIL,
              "${SQLRESULT_GETAFFECTEDROWCOUNT_DEPRECATED}");
REGISTER_HELP(SQLRESULT_GETAFFECTEDROWCOUNT_DEPRECATED,
              "@attention This function will be removed in a future release, "
              "use the <b><<<getAffectedItemsCount>>></b> function instead.");

/**
 * $(SQLRESULT_GETAFFECTEDROWCOUNT_BRIEF)
 */
#if DOXYGEN_JS
/**
 * @attention This function will be removed in a future release, use the
 * &nbsp;<b>getAffectedItemsCount</b> function instead.
 */
#elif DOXYGEN_PY
/**
 * @attention This function will be removed in a future release, use the
 * &nbsp;<b>get_affected_items_count</b> function instead.
 */
#endif
#if DOXYGEN_JS
Integer SqlResult::getAffectedRowCount() {}
#elif DOXYGEN_PY
int SqlResult::get_affected_row_count() {}
#endif
int64_t SqlResult::get_affected_row_count() const {
  if (_result) return _result->get_affected_row_count();
  return 0;
}

shcore::Value SqlResult::get_member(const std::string &prop) const {
  Value ret_val;
  if (prop == "autoIncrementValue") {
    ret_val = Value(get_auto_increment_value());
  } else if (prop == "affectedRowCount") {
    ret_val = Value(get_affected_row_count());
    log_warning("'%s' is deprecated, use '%s' instead.",
                get_function_name("affectedRowCount").c_str(),
                get_function_name("affectedItemsCount").c_str());
  } else {
    ret_val = RowResult::get_member(prop);
  }

  return ret_val;
}

// Documentation of hasData function
REGISTER_HELP_FUNCTION(hasData, SqlResult);
REGISTER_HELP(SQLRESULT_HASDATA_BRIEF,
              "Returns true if the last statement execution has a result set.");

/**
 * $(SQLRESULT_HASDATA_BRIEF)
 *
 */
#if DOXYGEN_JS
Bool SqlResult::hasData() {}
#elif DOXYGEN_PY
bool SqlResult::has_data() {}
#endif
shcore::Value SqlResult::has_data(const shcore::Argument_list &args) const {
  args.ensure_count(0, get_function_name("hasData").c_str());

  return Value(_result && _result->has_resultset());
}

// Documentation of nextDataSet function
REGISTER_HELP_FUNCTION(nextDataSet, SqlResult);
REGISTER_HELP(SQLRESULT_NEXTDATASET_BRIEF,
              "Prepares the SqlResult to start reading data from the next "
              "Result (if many results were returned).");
REGISTER_HELP(SQLRESULT_NEXTDATASET_DETAIL,
              "${SQLRESULT_NEXTDATASET_DEPRECATED}");
REGISTER_HELP(SQLRESULT_NEXTDATASET_DEPRECATED,
              "@attention This function will be removed in a future release, "
              "use the <b><<<nextResult>>></b> function instead.");
REGISTER_HELP(SQLRESULT_NEXTDATASET_RETURNS,
              "@returns A boolean value indicating whether there is another "
              "result or not.");

/**
 * $(SQLRESULT_NEXTDATASET_BRIEF)
 *
 * $(SQLRESULT_NEXTDATASET_RETURNS)
 */
#if DOXYGEN_JS
/**
 * @attention This function will be removed in a future release, use the
 * &nbsp;<b>nextResult</b> function instead.
 */
#elif DOXYGEN_PY
/**
 * @attention This function will be removed in a future release, use the
 * &nbsp;<b>next_result</b> function instead.
 */
#endif

#if DOXYGEN_JS
Bool SqlResult::nextDataSet() {}
#elif DOXYGEN_PY
bool SqlResult::next_data_set() {}
#endif
shcore::Value SqlResult::next_data_set(const shcore::Argument_list &args) {
  args.ensure_count(0, get_function_name("nextDataSet").c_str());

  log_warning("'%s' is deprecated, use '%s' instead.",
              get_function_name("nextDataSet").c_str(),
              get_function_name("nextResult").c_str());

  return shcore::Value(_result->next_resultset());
}

// Documentation of nextDataSet function
REGISTER_HELP_FUNCTION(nextResult, SqlResult);
REGISTER_HELP(SQLRESULT_NEXTRESULT_BRIEF,
              "Prepares the SqlResult to start reading data from the next "
              "Result (if many results were returned).");
REGISTER_HELP(SQLRESULT_NEXTRESULT_RETURNS,
              "@returns A boolean value indicating whether there is another "
              "result or not.");

/**
 * $(SQLRESULT_NEXTRESULT_BRIEF)
 *
 * $(SQLRESULT_NEXTRESULT_RETURNS)
 */
#if DOXYGEN_JS
Bool SqlResult::nextResult() {}
#elif DOXYGEN_PY
bool SqlResult::next_result() {}
#endif
shcore::Value SqlResult::next_result(const shcore::Argument_list &args) {
  args.ensure_count(0, get_function_name("nextResult").c_str());

  return shcore::Value(_result->next_resultset());
}

void SqlResult::append_json(shcore::JSON_dumper &dumper) const {
  dumper.start_object();

  RowResult::append_json(dumper);

  dumper.append_value("hasData", has_data(shcore::Argument_list()));
  dumper.append_value("affectedRowCount", get_member("affectedItemsCount"));
  dumper.append_value("autoIncrementValue", get_member("autoIncrementValue"));

  dumper.end_object();
}

}  // namespace mysqlx
}  // namespace mysqlsh
