# coding: utf-8
# Copyright (c) 2016, 2019, Oracle and/or its affiliates. All rights reserved.


from oci.util import formatted_flat_dict, NONE_SENTINEL, value_allowed_none_or_none_sentinel  # noqa: F401
from oci.decorators import init_model_state_from_kwargs


@init_model_state_from_kwargs
class FlowLogConfig(object):
    """
    *Flow logs* record information about traffic that is either allowed or rejected by the
    :class:`SecurityList` that control traffic in and out of a
    :class:`Vnic`.

    A *flow log configuration* (`FlowLogConfig`) contains information about where to store flow
    logs (an Object Storage bucket in your tenancy), and the type of logs to store.

    **Important:** For logs to be placed in the Object Storage bucket listed in the configuration,
    an administrator must create an IAM policy in your tenancy that lets the Networking service
    put objects in that bucket. Otherwise, no flow logs can be written to the bucket.
    Here's the required policy (which consists of three separate statements):

    `define tenancy VcnFlowLogs as ocid1.tenancy.oc1..aaaaaaaacaswototo7jx2hx24wyanauiyzreowh2cmsbmrkakeaabdyk6c6q`

    `define dynamic-group FlowLogsDynamicGroup as ocid1.dynamicgroup.oc1..aaaaaaaalzalw7g5jf5ujqlzuho3adrrhlioekcjigrcbv44gim5euyuostq`

    `admit dynamic-group FlowLogsDynamicGroup of tenancy VcnFlowLogs to manage objects in tenancy where target.bucket.name='yourbucketname'`

    To enable flow logs for a subnet: after creating a flow
    log configuration, attach the flow log configuration to that subnet. See
    :class:`FlowLogConfigAttachment` and
    :func:`create_flow_log_config_attachment`.

    **Warning:** Oracle recommends that you avoid using any confidential information when you
    supply string values using the API.
    """

    #: A constant which can be used with the lifecycle_state property of a FlowLogConfig.
    #: This constant has a value of "PROVISIONING"
    LIFECYCLE_STATE_PROVISIONING = "PROVISIONING"

    #: A constant which can be used with the lifecycle_state property of a FlowLogConfig.
    #: This constant has a value of "AVAILABLE"
    LIFECYCLE_STATE_AVAILABLE = "AVAILABLE"

    #: A constant which can be used with the lifecycle_state property of a FlowLogConfig.
    #: This constant has a value of "TERMINATING"
    LIFECYCLE_STATE_TERMINATING = "TERMINATING"

    #: A constant which can be used with the lifecycle_state property of a FlowLogConfig.
    #: This constant has a value of "TERMINATED"
    LIFECYCLE_STATE_TERMINATED = "TERMINATED"

    #: A constant which can be used with the flow_log_type property of a FlowLogConfig.
    #: This constant has a value of "ALL"
    FLOW_LOG_TYPE_ALL = "ALL"

    def __init__(self, **kwargs):
        """
        Initializes a new FlowLogConfig object with values from keyword arguments.
        The following keyword arguments are supported (corresponding to the getters/setters of this class):

        :param compartment_id:
            The value to assign to the compartment_id property of this FlowLogConfig.
        :type compartment_id: str

        :param defined_tags:
            The value to assign to the defined_tags property of this FlowLogConfig.
        :type defined_tags: dict(str, dict(str, object))

        :param display_name:
            The value to assign to the display_name property of this FlowLogConfig.
        :type display_name: str

        :param freeform_tags:
            The value to assign to the freeform_tags property of this FlowLogConfig.
        :type freeform_tags: dict(str, str)

        :param id:
            The value to assign to the id property of this FlowLogConfig.
        :type id: str

        :param lifecycle_state:
            The value to assign to the lifecycle_state property of this FlowLogConfig.
            Allowed values for this property are: "PROVISIONING", "AVAILABLE", "TERMINATING", "TERMINATED", 'UNKNOWN_ENUM_VALUE'.
            Any unrecognized values returned by a service will be mapped to 'UNKNOWN_ENUM_VALUE'.
        :type lifecycle_state: str

        :param flow_log_type:
            The value to assign to the flow_log_type property of this FlowLogConfig.
            Allowed values for this property are: "ALL", 'UNKNOWN_ENUM_VALUE'.
            Any unrecognized values returned by a service will be mapped to 'UNKNOWN_ENUM_VALUE'.
        :type flow_log_type: str

        :param destination:
            The value to assign to the destination property of this FlowLogConfig.
        :type destination: FlowLogDestination

        :param time_created:
            The value to assign to the time_created property of this FlowLogConfig.
        :type time_created: datetime

        """
        self.swagger_types = {
            'compartment_id': 'str',
            'defined_tags': 'dict(str, dict(str, object))',
            'display_name': 'str',
            'freeform_tags': 'dict(str, str)',
            'id': 'str',
            'lifecycle_state': 'str',
            'flow_log_type': 'str',
            'destination': 'FlowLogDestination',
            'time_created': 'datetime'
        }

        self.attribute_map = {
            'compartment_id': 'compartmentId',
            'defined_tags': 'definedTags',
            'display_name': 'displayName',
            'freeform_tags': 'freeformTags',
            'id': 'id',
            'lifecycle_state': 'lifecycleState',
            'flow_log_type': 'flowLogType',
            'destination': 'destination',
            'time_created': 'timeCreated'
        }

        self._compartment_id = None
        self._defined_tags = None
        self._display_name = None
        self._freeform_tags = None
        self._id = None
        self._lifecycle_state = None
        self._flow_log_type = None
        self._destination = None
        self._time_created = None

    @property
    def compartment_id(self):
        """
        **[Required]** Gets the compartment_id of this FlowLogConfig.
        The `OCID`__ of the compartment containing the flow
        log configuration.

        __ https://docs.cloud.oracle.com/Content/General/Concepts/identifiers.htm


        :return: The compartment_id of this FlowLogConfig.
        :rtype: str
        """
        return self._compartment_id

    @compartment_id.setter
    def compartment_id(self, compartment_id):
        """
        Sets the compartment_id of this FlowLogConfig.
        The `OCID`__ of the compartment containing the flow
        log configuration.

        __ https://docs.cloud.oracle.com/Content/General/Concepts/identifiers.htm


        :param compartment_id: The compartment_id of this FlowLogConfig.
        :type: str
        """
        self._compartment_id = compartment_id

    @property
    def defined_tags(self):
        """
        Gets the defined_tags of this FlowLogConfig.
        Defined tags for this resource. Each key is predefined and scoped to a
        namespace. For more information, see `Resource Tags`__.

        Example: `{\"Operations\": {\"CostCenter\": \"42\"}}`

        __ https://docs.cloud.oracle.com/Content/General/Concepts/resourcetags.htm


        :return: The defined_tags of this FlowLogConfig.
        :rtype: dict(str, dict(str, object))
        """
        return self._defined_tags

    @defined_tags.setter
    def defined_tags(self, defined_tags):
        """
        Sets the defined_tags of this FlowLogConfig.
        Defined tags for this resource. Each key is predefined and scoped to a
        namespace. For more information, see `Resource Tags`__.

        Example: `{\"Operations\": {\"CostCenter\": \"42\"}}`

        __ https://docs.cloud.oracle.com/Content/General/Concepts/resourcetags.htm


        :param defined_tags: The defined_tags of this FlowLogConfig.
        :type: dict(str, dict(str, object))
        """
        self._defined_tags = defined_tags

    @property
    def display_name(self):
        """
        **[Required]** Gets the display_name of this FlowLogConfig.
        A user-friendly name. Does not have to be unique, and it's changeable. Avoid
        entering confidential information.


        :return: The display_name of this FlowLogConfig.
        :rtype: str
        """
        return self._display_name

    @display_name.setter
    def display_name(self, display_name):
        """
        Sets the display_name of this FlowLogConfig.
        A user-friendly name. Does not have to be unique, and it's changeable. Avoid
        entering confidential information.


        :param display_name: The display_name of this FlowLogConfig.
        :type: str
        """
        self._display_name = display_name

    @property
    def freeform_tags(self):
        """
        Gets the freeform_tags of this FlowLogConfig.
        Free-form tags for this resource. Each tag is a simple key-value pair with no
        predefined name, type, or namespace. For more information, see `Resource Tags`__.

        Example: `{\"Department\": \"Finance\"}`

        __ https://docs.cloud.oracle.com/Content/General/Concepts/resourcetags.htm


        :return: The freeform_tags of this FlowLogConfig.
        :rtype: dict(str, str)
        """
        return self._freeform_tags

    @freeform_tags.setter
    def freeform_tags(self, freeform_tags):
        """
        Sets the freeform_tags of this FlowLogConfig.
        Free-form tags for this resource. Each tag is a simple key-value pair with no
        predefined name, type, or namespace. For more information, see `Resource Tags`__.

        Example: `{\"Department\": \"Finance\"}`

        __ https://docs.cloud.oracle.com/Content/General/Concepts/resourcetags.htm


        :param freeform_tags: The freeform_tags of this FlowLogConfig.
        :type: dict(str, str)
        """
        self._freeform_tags = freeform_tags

    @property
    def id(self):
        """
        **[Required]** Gets the id of this FlowLogConfig.
        The flow log configuration's `OCID`__.

        __ https://docs.cloud.oracle.com/Content/General/Concepts/identifiers.htm


        :return: The id of this FlowLogConfig.
        :rtype: str
        """
        return self._id

    @id.setter
    def id(self, id):
        """
        Sets the id of this FlowLogConfig.
        The flow log configuration's `OCID`__.

        __ https://docs.cloud.oracle.com/Content/General/Concepts/identifiers.htm


        :param id: The id of this FlowLogConfig.
        :type: str
        """
        self._id = id

    @property
    def lifecycle_state(self):
        """
        **[Required]** Gets the lifecycle_state of this FlowLogConfig.
        The flow log configuration's current state.

        Allowed values for this property are: "PROVISIONING", "AVAILABLE", "TERMINATING", "TERMINATED", 'UNKNOWN_ENUM_VALUE'.
        Any unrecognized values returned by a service will be mapped to 'UNKNOWN_ENUM_VALUE'.


        :return: The lifecycle_state of this FlowLogConfig.
        :rtype: str
        """
        return self._lifecycle_state

    @lifecycle_state.setter
    def lifecycle_state(self, lifecycle_state):
        """
        Sets the lifecycle_state of this FlowLogConfig.
        The flow log configuration's current state.


        :param lifecycle_state: The lifecycle_state of this FlowLogConfig.
        :type: str
        """
        allowed_values = ["PROVISIONING", "AVAILABLE", "TERMINATING", "TERMINATED"]
        if not value_allowed_none_or_none_sentinel(lifecycle_state, allowed_values):
            lifecycle_state = 'UNKNOWN_ENUM_VALUE'
        self._lifecycle_state = lifecycle_state

    @property
    def flow_log_type(self):
        """
        **[Required]** Gets the flow_log_type of this FlowLogConfig.
        Type or types of flow logs to store. `ALL` includes records for both accepted traffic and
        rejected traffic.

        Allowed values for this property are: "ALL", 'UNKNOWN_ENUM_VALUE'.
        Any unrecognized values returned by a service will be mapped to 'UNKNOWN_ENUM_VALUE'.


        :return: The flow_log_type of this FlowLogConfig.
        :rtype: str
        """
        return self._flow_log_type

    @flow_log_type.setter
    def flow_log_type(self, flow_log_type):
        """
        Sets the flow_log_type of this FlowLogConfig.
        Type or types of flow logs to store. `ALL` includes records for both accepted traffic and
        rejected traffic.


        :param flow_log_type: The flow_log_type of this FlowLogConfig.
        :type: str
        """
        allowed_values = ["ALL"]
        if not value_allowed_none_or_none_sentinel(flow_log_type, allowed_values):
            flow_log_type = 'UNKNOWN_ENUM_VALUE'
        self._flow_log_type = flow_log_type

    @property
    def destination(self):
        """
        **[Required]** Gets the destination of this FlowLogConfig.
        Where to store the flow logs.


        :return: The destination of this FlowLogConfig.
        :rtype: FlowLogDestination
        """
        return self._destination

    @destination.setter
    def destination(self, destination):
        """
        Sets the destination of this FlowLogConfig.
        Where to store the flow logs.


        :param destination: The destination of this FlowLogConfig.
        :type: FlowLogDestination
        """
        self._destination = destination

    @property
    def time_created(self):
        """
        **[Required]** Gets the time_created of this FlowLogConfig.
        The date and time the flow log configuration was created, in the format defined by RFC3339.

        Example: `2016-08-25T21:10:29.600Z`


        :return: The time_created of this FlowLogConfig.
        :rtype: datetime
        """
        return self._time_created

    @time_created.setter
    def time_created(self, time_created):
        """
        Sets the time_created of this FlowLogConfig.
        The date and time the flow log configuration was created, in the format defined by RFC3339.

        Example: `2016-08-25T21:10:29.600Z`


        :param time_created: The time_created of this FlowLogConfig.
        :type: datetime
        """
        self._time_created = time_created

    def __repr__(self):
        return formatted_flat_dict(self)

    def __eq__(self, other):
        if other is None:
            return False

        return self.__dict__ == other.__dict__

    def __ne__(self, other):
        return not self == other
