# coding: utf-8
# Copyright (c) 2016, 2019, Oracle and/or its affiliates. All rights reserved.


from oci.util import formatted_flat_dict, NONE_SENTINEL, value_allowed_none_or_none_sentinel  # noqa: F401
from oci.decorators import init_model_state_from_kwargs


@init_model_state_from_kwargs
class PrivateEndpoint(object):
    """
    A *private endpoint* (PE) is a way for an Oracle service to give a customer a private access point for
    the service within the customer's VCN. The PE consists of a VNIC with a private IP
    in the customer's VCN. The PE is associated with an
    :class:`EndpointService`
    that the service team has previously registered. The customer accesses the service
    by sending traffic to the PE's IP address on the registered port. That traffic is then
    sent to the :class:`PrivateAccessGateway` on the service VCN.
    The gateway then sends the traffic to the PE's associated `EndpointService` for
    processing.

    **Regarding DNS for the private endpoint:** When creating a PE, the private endpoint service assigns
    a fully qualified domain name (FQDN) to the PE and creates the related DNS zone and record in
    the customer's VCN. This enables the customer to use the FQDN instead of the PE's private IP
    address to access the service. Here are details about how the private endpoint service determines
    the value to use for the PE's FQDN:

    * Both the :class:`EndpointService` object and the
    :func:`create_private_endpoint_details`
    object have an `endpointFqdn` attribute.

    * If you don't specify an FQDN for `CreatePrivateEndpointDetails` during PE creation, the
    endpoint service's `endpointFqdn` is used for the PE's `endpointFqdn`.

    * If you specify an FQDN for `CreatePrivateEndpointDetails` during PE creation, that value is used.
    It always takes precedence over any value set in the `EndpointService` object.

    * If the `EndpointService` object does not have an FQDN value set, and you don't provide a value
    in `CreatePrivateEndpointDetails` during creation, an error is returned.

    * **Special scenario:**  If the endpoint service allows multiple PE's to be created per customer VCN
    (see the `areMultiplePrivateEndpointsPerVcnAllowed` attribute in the `EndpointService`),
    the `EndpointService` is prohibited from also having an `endpointFqdn` value. This restriction ensures
    that each FQDN in the customer's VCN resolves to a single PE. Therefore, for this particular
    scenario, you must assign each PE a unique FQDN within the scope of the customer's VCN.

    To use any of the API operations, you must be authorized in an IAM policy. If you're not authorized,
    talk to an administrator. If you're an administrator who needs to write policies to give users access, see
    `Getting Started with Policies`__.

    **Warning:** Oracle recommends that you avoid using any confidential information when you
    supply string values using the API.

    __ https://docs.cloud.oracle.com/Content/Identity/Concepts/policygetstarted.htm
    """

    #: A constant which can be used with the lifecycle_state property of a PrivateEndpoint.
    #: This constant has a value of "PROVISIONING"
    LIFECYCLE_STATE_PROVISIONING = "PROVISIONING"

    #: A constant which can be used with the lifecycle_state property of a PrivateEndpoint.
    #: This constant has a value of "AVAILABLE"
    LIFECYCLE_STATE_AVAILABLE = "AVAILABLE"

    #: A constant which can be used with the lifecycle_state property of a PrivateEndpoint.
    #: This constant has a value of "TERMINATING"
    LIFECYCLE_STATE_TERMINATING = "TERMINATING"

    #: A constant which can be used with the lifecycle_state property of a PrivateEndpoint.
    #: This constant has a value of "TERMINATED"
    LIFECYCLE_STATE_TERMINATED = "TERMINATED"

    def __init__(self, **kwargs):
        """
        Initializes a new PrivateEndpoint object with values from keyword arguments.
        The following keyword arguments are supported (corresponding to the getters/setters of this class):

        :param compartment_id:
            The value to assign to the compartment_id property of this PrivateEndpoint.
        :type compartment_id: str

        :param defined_tags:
            The value to assign to the defined_tags property of this PrivateEndpoint.
        :type defined_tags: dict(str, dict(str, object))

        :param freeform_tags:
            The value to assign to the freeform_tags property of this PrivateEndpoint.
        :type freeform_tags: dict(str, str)

        :param display_name:
            The value to assign to the display_name property of this PrivateEndpoint.
        :type display_name: str

        :param description:
            The value to assign to the description property of this PrivateEndpoint.
        :type description: str

        :param id:
            The value to assign to the id property of this PrivateEndpoint.
        :type id: str

        :param endpoint_service_id:
            The value to assign to the endpoint_service_id property of this PrivateEndpoint.
        :type endpoint_service_id: str

        :param time_created:
            The value to assign to the time_created property of this PrivateEndpoint.
        :type time_created: datetime

        :param vcn_id:
            The value to assign to the vcn_id property of this PrivateEndpoint.
        :type vcn_id: str

        :param subnet_id:
            The value to assign to the subnet_id property of this PrivateEndpoint.
        :type subnet_id: str

        :param private_endpoint_vnic_id:
            The value to assign to the private_endpoint_vnic_id property of this PrivateEndpoint.
        :type private_endpoint_vnic_id: str

        :param private_endpoint_ip:
            The value to assign to the private_endpoint_ip property of this PrivateEndpoint.
        :type private_endpoint_ip: str

        :param endpoint_fqdn:
            The value to assign to the endpoint_fqdn property of this PrivateEndpoint.
        :type endpoint_fqdn: str

        :param lifecycle_state:
            The value to assign to the lifecycle_state property of this PrivateEndpoint.
            Allowed values for this property are: "PROVISIONING", "AVAILABLE", "TERMINATING", "TERMINATED", 'UNKNOWN_ENUM_VALUE'.
            Any unrecognized values returned by a service will be mapped to 'UNKNOWN_ENUM_VALUE'.
        :type lifecycle_state: str

        :param nsg_ids:
            The value to assign to the nsg_ids property of this PrivateEndpoint.
        :type nsg_ids: list[str]

        :param reverse_connection_configuration:
            The value to assign to the reverse_connection_configuration property of this PrivateEndpoint.
        :type reverse_connection_configuration: ReverseConnectionConfiguration

        """
        self.swagger_types = {
            'compartment_id': 'str',
            'defined_tags': 'dict(str, dict(str, object))',
            'freeform_tags': 'dict(str, str)',
            'display_name': 'str',
            'description': 'str',
            'id': 'str',
            'endpoint_service_id': 'str',
            'time_created': 'datetime',
            'vcn_id': 'str',
            'subnet_id': 'str',
            'private_endpoint_vnic_id': 'str',
            'private_endpoint_ip': 'str',
            'endpoint_fqdn': 'str',
            'lifecycle_state': 'str',
            'nsg_ids': 'list[str]',
            'reverse_connection_configuration': 'ReverseConnectionConfiguration'
        }

        self.attribute_map = {
            'compartment_id': 'compartmentId',
            'defined_tags': 'definedTags',
            'freeform_tags': 'freeformTags',
            'display_name': 'displayName',
            'description': 'description',
            'id': 'id',
            'endpoint_service_id': 'endpointServiceId',
            'time_created': 'timeCreated',
            'vcn_id': 'vcnId',
            'subnet_id': 'subnetId',
            'private_endpoint_vnic_id': 'privateEndpointVnicId',
            'private_endpoint_ip': 'privateEndpointIp',
            'endpoint_fqdn': 'endpointFqdn',
            'lifecycle_state': 'lifecycleState',
            'nsg_ids': 'nsgIds',
            'reverse_connection_configuration': 'reverseConnectionConfiguration'
        }

        self._compartment_id = None
        self._defined_tags = None
        self._freeform_tags = None
        self._display_name = None
        self._description = None
        self._id = None
        self._endpoint_service_id = None
        self._time_created = None
        self._vcn_id = None
        self._subnet_id = None
        self._private_endpoint_vnic_id = None
        self._private_endpoint_ip = None
        self._endpoint_fqdn = None
        self._lifecycle_state = None
        self._nsg_ids = None
        self._reverse_connection_configuration = None

    @property
    def compartment_id(self):
        """
        **[Required]** Gets the compartment_id of this PrivateEndpoint.
        The `OCID`__ of the compartment that contains the
        private endpoint.

        __ https://docs.cloud.oracle.com/Content/General/Concepts/identifiers.htm


        :return: The compartment_id of this PrivateEndpoint.
        :rtype: str
        """
        return self._compartment_id

    @compartment_id.setter
    def compartment_id(self, compartment_id):
        """
        Sets the compartment_id of this PrivateEndpoint.
        The `OCID`__ of the compartment that contains the
        private endpoint.

        __ https://docs.cloud.oracle.com/Content/General/Concepts/identifiers.htm


        :param compartment_id: The compartment_id of this PrivateEndpoint.
        :type: str
        """
        self._compartment_id = compartment_id

    @property
    def defined_tags(self):
        """
        Gets the defined_tags of this PrivateEndpoint.
        Defined tags for this resource. Each key is predefined and scoped to a
        namespace. For more information, see `Resource Tags`__.

        Example: `{\"Operations\": {\"CostCenter\": \"42\"}}`

        __ https://docs.cloud.oracle.com/Content/General/Concepts/resourcetags.htm


        :return: The defined_tags of this PrivateEndpoint.
        :rtype: dict(str, dict(str, object))
        """
        return self._defined_tags

    @defined_tags.setter
    def defined_tags(self, defined_tags):
        """
        Sets the defined_tags of this PrivateEndpoint.
        Defined tags for this resource. Each key is predefined and scoped to a
        namespace. For more information, see `Resource Tags`__.

        Example: `{\"Operations\": {\"CostCenter\": \"42\"}}`

        __ https://docs.cloud.oracle.com/Content/General/Concepts/resourcetags.htm


        :param defined_tags: The defined_tags of this PrivateEndpoint.
        :type: dict(str, dict(str, object))
        """
        self._defined_tags = defined_tags

    @property
    def freeform_tags(self):
        """
        Gets the freeform_tags of this PrivateEndpoint.
        Free-form tags for this resource. Each tag is a simple key-value pair with no
        predefined name, type, or namespace. For more information, see `Resource Tags`__.

        Example: `{\"Department\": \"Finance\"}`

        __ https://docs.cloud.oracle.com/Content/General/Concepts/resourcetags.htm


        :return: The freeform_tags of this PrivateEndpoint.
        :rtype: dict(str, str)
        """
        return self._freeform_tags

    @freeform_tags.setter
    def freeform_tags(self, freeform_tags):
        """
        Sets the freeform_tags of this PrivateEndpoint.
        Free-form tags for this resource. Each tag is a simple key-value pair with no
        predefined name, type, or namespace. For more information, see `Resource Tags`__.

        Example: `{\"Department\": \"Finance\"}`

        __ https://docs.cloud.oracle.com/Content/General/Concepts/resourcetags.htm


        :param freeform_tags: The freeform_tags of this PrivateEndpoint.
        :type: dict(str, str)
        """
        self._freeform_tags = freeform_tags

    @property
    def display_name(self):
        """
        Gets the display_name of this PrivateEndpoint.
        A user-friendly name. Does not have to be unique.


        :return: The display_name of this PrivateEndpoint.
        :rtype: str
        """
        return self._display_name

    @display_name.setter
    def display_name(self, display_name):
        """
        Sets the display_name of this PrivateEndpoint.
        A user-friendly name. Does not have to be unique.


        :param display_name: The display_name of this PrivateEndpoint.
        :type: str
        """
        self._display_name = display_name

    @property
    def description(self):
        """
        Gets the description of this PrivateEndpoint.
        A description of this private endpoint.


        :return: The description of this PrivateEndpoint.
        :rtype: str
        """
        return self._description

    @description.setter
    def description(self, description):
        """
        Sets the description of this PrivateEndpoint.
        A description of this private endpoint.


        :param description: The description of this PrivateEndpoint.
        :type: str
        """
        self._description = description

    @property
    def id(self):
        """
        **[Required]** Gets the id of this PrivateEndpoint.
        The `OCID`__ of the private endpoint.

        __ https://docs.cloud.oracle.com/Content/General/Concepts/identifiers.htm


        :return: The id of this PrivateEndpoint.
        :rtype: str
        """
        return self._id

    @id.setter
    def id(self, id):
        """
        Sets the id of this PrivateEndpoint.
        The `OCID`__ of the private endpoint.

        __ https://docs.cloud.oracle.com/Content/General/Concepts/identifiers.htm


        :param id: The id of this PrivateEndpoint.
        :type: str
        """
        self._id = id

    @property
    def endpoint_service_id(self):
        """
        **[Required]** Gets the endpoint_service_id of this PrivateEndpoint.
        The `OCID`__ of the endpoint service that is associated
        with the private endpoint.

        __ https://docs.cloud.oracle.com/Content/General/Concepts/identifiers.htm


        :return: The endpoint_service_id of this PrivateEndpoint.
        :rtype: str
        """
        return self._endpoint_service_id

    @endpoint_service_id.setter
    def endpoint_service_id(self, endpoint_service_id):
        """
        Sets the endpoint_service_id of this PrivateEndpoint.
        The `OCID`__ of the endpoint service that is associated
        with the private endpoint.

        __ https://docs.cloud.oracle.com/Content/General/Concepts/identifiers.htm


        :param endpoint_service_id: The endpoint_service_id of this PrivateEndpoint.
        :type: str
        """
        self._endpoint_service_id = endpoint_service_id

    @property
    def time_created(self):
        """
        Gets the time_created of this PrivateEndpoint.
        The date and time the private endpoint was created, in the format defined by RFC3339.

        Example: `2016-08-25T21:10:29.600Z`


        :return: The time_created of this PrivateEndpoint.
        :rtype: datetime
        """
        return self._time_created

    @time_created.setter
    def time_created(self, time_created):
        """
        Sets the time_created of this PrivateEndpoint.
        The date and time the private endpoint was created, in the format defined by RFC3339.

        Example: `2016-08-25T21:10:29.600Z`


        :param time_created: The time_created of this PrivateEndpoint.
        :type: datetime
        """
        self._time_created = time_created

    @property
    def vcn_id(self):
        """
        **[Required]** Gets the vcn_id of this PrivateEndpoint.
        The `OCID`__ of the customer VCN that the private
        endpoint belongs to.

        __ https://docs.cloud.oracle.com/Content/General/Concepts/identifiers.htm


        :return: The vcn_id of this PrivateEndpoint.
        :rtype: str
        """
        return self._vcn_id

    @vcn_id.setter
    def vcn_id(self, vcn_id):
        """
        Sets the vcn_id of this PrivateEndpoint.
        The `OCID`__ of the customer VCN that the private
        endpoint belongs to.

        __ https://docs.cloud.oracle.com/Content/General/Concepts/identifiers.htm


        :param vcn_id: The vcn_id of this PrivateEndpoint.
        :type: str
        """
        self._vcn_id = vcn_id

    @property
    def subnet_id(self):
        """
        **[Required]** Gets the subnet_id of this PrivateEndpoint.
        The `OCID`__ of the subnet that the private endpoint
        belongs to.

        __ https://docs.cloud.oracle.com/Content/General/Concepts/identifiers.htm


        :return: The subnet_id of this PrivateEndpoint.
        :rtype: str
        """
        return self._subnet_id

    @subnet_id.setter
    def subnet_id(self, subnet_id):
        """
        Sets the subnet_id of this PrivateEndpoint.
        The `OCID`__ of the subnet that the private endpoint
        belongs to.

        __ https://docs.cloud.oracle.com/Content/General/Concepts/identifiers.htm


        :param subnet_id: The subnet_id of this PrivateEndpoint.
        :type: str
        """
        self._subnet_id = subnet_id

    @property
    def private_endpoint_vnic_id(self):
        """
        **[Required]** Gets the private_endpoint_vnic_id of this PrivateEndpoint.
        The `OCID`__ of the private endpoint's VNIC, which
        resides in the customer's VCN.

        __ https://docs.cloud.oracle.com/Content/General/Concepts/identifiers.htm


        :return: The private_endpoint_vnic_id of this PrivateEndpoint.
        :rtype: str
        """
        return self._private_endpoint_vnic_id

    @private_endpoint_vnic_id.setter
    def private_endpoint_vnic_id(self, private_endpoint_vnic_id):
        """
        Sets the private_endpoint_vnic_id of this PrivateEndpoint.
        The `OCID`__ of the private endpoint's VNIC, which
        resides in the customer's VCN.

        __ https://docs.cloud.oracle.com/Content/General/Concepts/identifiers.htm


        :param private_endpoint_vnic_id: The private_endpoint_vnic_id of this PrivateEndpoint.
        :type: str
        """
        self._private_endpoint_vnic_id = private_endpoint_vnic_id

    @property
    def private_endpoint_ip(self):
        """
        **[Required]** Gets the private_endpoint_ip of this PrivateEndpoint.
        The private IP address (in the customer's VCN) that represents the access point for the
        associated endpoint service.


        :return: The private_endpoint_ip of this PrivateEndpoint.
        :rtype: str
        """
        return self._private_endpoint_ip

    @private_endpoint_ip.setter
    def private_endpoint_ip(self, private_endpoint_ip):
        """
        Sets the private_endpoint_ip of this PrivateEndpoint.
        The private IP address (in the customer's VCN) that represents the access point for the
        associated endpoint service.


        :param private_endpoint_ip: The private_endpoint_ip of this PrivateEndpoint.
        :type: str
        """
        self._private_endpoint_ip = private_endpoint_ip

    @property
    def endpoint_fqdn(self):
        """
        Gets the endpoint_fqdn of this PrivateEndpoint.
        The three-label FQDN to use for the private endpoint. The customer VCN's DNS records are
        updated with this FQDN.

        If you provide a value for this attribute, it overrides the `endpointFqdn` in the associated
        :class:`EndpointService`. For more information, see the discussion
        of DNS and FQDNs in :class:`PrivateEndpoint`.

        You can change the PE's FQDN (see
        :func:`update_private_endpoint_details`).

        Example: `xyz.oraclecloud.com`


        :return: The endpoint_fqdn of this PrivateEndpoint.
        :rtype: str
        """
        return self._endpoint_fqdn

    @endpoint_fqdn.setter
    def endpoint_fqdn(self, endpoint_fqdn):
        """
        Sets the endpoint_fqdn of this PrivateEndpoint.
        The three-label FQDN to use for the private endpoint. The customer VCN's DNS records are
        updated with this FQDN.

        If you provide a value for this attribute, it overrides the `endpointFqdn` in the associated
        :class:`EndpointService`. For more information, see the discussion
        of DNS and FQDNs in :class:`PrivateEndpoint`.

        You can change the PE's FQDN (see
        :func:`update_private_endpoint_details`).

        Example: `xyz.oraclecloud.com`


        :param endpoint_fqdn: The endpoint_fqdn of this PrivateEndpoint.
        :type: str
        """
        self._endpoint_fqdn = endpoint_fqdn

    @property
    def lifecycle_state(self):
        """
        **[Required]** Gets the lifecycle_state of this PrivateEndpoint.
        The private endpoint's current lifecycle state.

        Allowed values for this property are: "PROVISIONING", "AVAILABLE", "TERMINATING", "TERMINATED", 'UNKNOWN_ENUM_VALUE'.
        Any unrecognized values returned by a service will be mapped to 'UNKNOWN_ENUM_VALUE'.


        :return: The lifecycle_state of this PrivateEndpoint.
        :rtype: str
        """
        return self._lifecycle_state

    @lifecycle_state.setter
    def lifecycle_state(self, lifecycle_state):
        """
        Sets the lifecycle_state of this PrivateEndpoint.
        The private endpoint's current lifecycle state.


        :param lifecycle_state: The lifecycle_state of this PrivateEndpoint.
        :type: str
        """
        allowed_values = ["PROVISIONING", "AVAILABLE", "TERMINATING", "TERMINATED"]
        if not value_allowed_none_or_none_sentinel(lifecycle_state, allowed_values):
            lifecycle_state = 'UNKNOWN_ENUM_VALUE'
        self._lifecycle_state = lifecycle_state

    @property
    def nsg_ids(self):
        """
        Gets the nsg_ids of this PrivateEndpoint.
        A list of the OCIDs of the network security groups that the private endpoint's VNIC belongs to.
        For more information about NSGs, see
        :class:`NetworkSecurityGroup`.


        :return: The nsg_ids of this PrivateEndpoint.
        :rtype: list[str]
        """
        return self._nsg_ids

    @nsg_ids.setter
    def nsg_ids(self, nsg_ids):
        """
        Sets the nsg_ids of this PrivateEndpoint.
        A list of the OCIDs of the network security groups that the private endpoint's VNIC belongs to.
        For more information about NSGs, see
        :class:`NetworkSecurityGroup`.


        :param nsg_ids: The nsg_ids of this PrivateEndpoint.
        :type: list[str]
        """
        self._nsg_ids = nsg_ids

    @property
    def reverse_connection_configuration(self):
        """
        Gets the reverse_connection_configuration of this PrivateEndpoint.

        :return: The reverse_connection_configuration of this PrivateEndpoint.
        :rtype: ReverseConnectionConfiguration
        """
        return self._reverse_connection_configuration

    @reverse_connection_configuration.setter
    def reverse_connection_configuration(self, reverse_connection_configuration):
        """
        Sets the reverse_connection_configuration of this PrivateEndpoint.

        :param reverse_connection_configuration: The reverse_connection_configuration of this PrivateEndpoint.
        :type: ReverseConnectionConfiguration
        """
        self._reverse_connection_configuration = reverse_connection_configuration

    def __repr__(self):
        return formatted_flat_dict(self)

    def __eq__(self, other):
        if other is None:
            return False

        return self.__dict__ == other.__dict__

    def __ne__(self, other):
        return not self == other
