

-- Object: web_object
-- Desc:   Stores the content of the individual web pages
DROP TABLE IF EXISTS web_object;
CREATE TABLE IF NOT EXISTS web_object (
  id_web_object INTEGER PRIMARY KEY AUTOINCREMENT,
  path TEXT UNIQUE NOT NULL,
  content_type TEXT NOT NULL,
  content BLOB,
  allow_embedded_code_execution INTEGER DEFAULT 0
  );
  
CREATE INDEX IF NOT EXISTS idx_web_object_path ON web_object(path);
  

-- Object: manual_type
-- Desc:   Stores the different document types by product, e.g. MySQL Server, MySQL Workbench
DROP TABLE IF EXISTS manual_type;
CREATE TABLE manual_type (
  id_manual_type INTEGER PRIMARY KEY,
  name TEXT NOT NULL UNIQUE
  );

INSERT INTO manual_type(id_manual_type, name)
  VALUES (1, "MySQL Manual");
INSERT INTO manual_type(id_manual_type, name)
  VALUES (10, "MySQL Server");
INSERT INTO manual_type(id_manual_type, name)
  VALUES (20, "MySQL Workbench");
INSERT INTO manual_type(id_manual_type, name)
  VALUES (100, "MySQL Connector/Net");
INSERT INTO manual_type(id_manual_type, name)
  VALUES (110, "MySQL Connector/Java");

-- Object: language
-- Desc:   A list of languages the documentation is available
DROP TABLE IF EXISTS language;
CREATE TABLE language (
  id_language INTEGER PRIMARY KEY,
  name TEXT NOT NULL UNIQUE,
  name_translated TEXT,
  code TEXT NOT NULL
  );

INSERT INTO language(id_language, name, name_translated, code)
  VALUES (1, "English", "English", "en");
INSERT INTO language(id_language, name, name_translated, code)
  VALUES (2, "German", "Deutsch", "de");

-- Object: manual
-- Desc:   All stored manuals
DROP TABLE IF EXISTS manual;
CREATE TABLE manual (
  id_manual INTEGER PRIMARY KEY AUTOINCREMENT,
  id_manual_type INTEGER NOT NULL,
  id_language INTEGER NOT NULL,
  description TEXT NOT NULL UNIQUE,
  directory TEXT NOT NULL UNIQUE,
  version_main INTEGER,
  version_minor INTEGER,
  version_caption TEXT NOT NULL,
  generation_date DATE,
  installed INTEGER,
  FOREIGN KEY (id_manual_type) 
    REFERENCES manual_type(id_manual_type) ON DELETE CASCADE,
  FOREIGN KEY (id_language) 
    REFERENCES language(id_language) ON DELETE CASCADE
  );

-- INSERT INTO manual(id_manual, id_manual_type, id_language, description, 
--   directory, version_main, version_minor, version_caption, generation_date,
--   installed)
--   VALUES (1, 10, 1, "MySQL Server 5.5 Reference Manual", 
--     "refman55-html-guitools", 5, 5, "5.5.8", NULL,
--     0);

-- INSERT INTO manual(id_manual, id_manual_type, id_language, description, 
--   directory, version_main, version_minor, version_caption, generation_date,
--   installed)
--   VALUES (2, 20, 1, "MySQL Workbench 5.2 Reference Manual", 
--     "wb-html-guitools", 5, 2, "5.2.31", NULL,
--     0);


-- Object: page
-- Desc:   Stores the individual pages of the doc manual
DROP TABLE IF EXISTS page;
CREATE TABLE page (
  id_page INTEGER PRIMARY KEY AUTOINCREMENT,
  id_manual INTEGER NOT NULL,
  id_web_object INTEGER NOT NULL,
  title TEXT,
  chapter TEXT,
  FOREIGN KEY (id_manual) 
    REFERENCES manual(id_manual),
  FOREIGN KEY (id_web_object) 
    REFERENCES web_object(id_web_object)
  );

CREATE INDEX idx_page ON page(title);

-- Object: page_content
-- Desc:   Stores the content of the individual pages
DROP TABLE IF EXISTS page_content;
CREATE VIRTUAL TABLE page_content USING fts3(
  id_page INTEGER PRIMARY KEY,
  title TEXT,
  content TEXT,
  tokenize=porter
  );

-- Object: page_rating
-- Desc:   Rating of a specific page used for full text search
DROP TABLE IF EXISTS page_rating;
CREATE TABLE page_rating (
  id_page_rating INTEGER PRIMARY KEY AUTOINCREMENT,
  id_manual_type INTEGER NOT NULL,
  title TEXT NOT NULL,
  rating INTEGER,
  FOREIGN KEY (id_manual_type) 
    REFERENCES manual_type(id_manual_type) ON DELETE CASCADE
);

CREATE INDEX idx_page_rating ON page_rating(title);

-- Rating for syntax pages
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "ALTER DATABASE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "ALTER EVENT Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "ALTER FUNCTION Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "ALTER LOGFILE GROUP Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "ALTER PROCEDURE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "ALTER SERVER Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "ALTER TABLE Syntax", 10000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "ALTER TABLESPACE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "ALTER VIEW Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "ANALYZE TABLE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "BACKUP TABLE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "BEGIN ... END  Compound Statement Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "BINLOG Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "CACHE INDEX Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "CALL Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "CHANGE MASTER TO Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "CHECK TABLE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "CHECKSUM TABLE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "Comment Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "START TRANSACTION,  COMMIT, and  ROLLBACK Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "CREATE DATABASE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "CREATE EVENT Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "CREATE FUNCTION Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "CREATE FUNCTION Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "CREATE INDEX Syntax", 5000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "CREATE LOGFILE GROUP Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "CREATE PROCEDURE and  CREATE FUNCTION Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "CREATE SERVER Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "CREATE TABLE Syntax", 15000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "CREATE TABLESPACE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "CREATE TRIGGER Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "CREATE USER Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "CREATE VIEW Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "DEALLOCATE PREPARE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "DECLARE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "DELETE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "DESCRIBE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "DO Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "DROP DATABASE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "DROP EVENT Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "DROP FUNCTION Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "DROP INDEX Syntax", 5000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "DROP LOGFILE GROUP Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "DROP PROCEDURE and  DROP FUNCTION Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "DROP SERVER Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "DROP TABLE Syntax", 10000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "DROP TABLESPACE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "DROP TRIGGER Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "DROP USER Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "DROP VIEW Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "Event Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "EXECUTE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "EXPLAIN Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "FLUSH Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "GRANT Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "HANDLER Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "HELP Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "Index Hint Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "INSERT DELAYED Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "INSERT ... ON  DUPLICATE KEY UPDATE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "INSERT ...  SELECT Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "INSERT Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "INSTALL PLUGIN Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "JOIN Syntax", 5000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "KILL Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "LOAD DATA FROM MASTER Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "LOAD DATA INFILE  Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "LOAD INDEX INTO  CACHE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "LOAD TABLE tbl_name FROM  MASTER Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "LOCK TABLES and  UNLOCK  TABLES Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "MASTER_POS_WAIT() Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "OPTIMIZE TABLE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "PREPARE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "PURGE BINARY LOGS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "RENAME DATABASE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "RENAME TABLE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "RENAME USER Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "REPAIR TABLE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "REPLACE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "RESET MASTER Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "RESET SLAVE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "RESET Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "RESTORE TABLE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "RETURN Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "REVOKE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SAVEPOINT and  ROLLBACK TO  SAVEPOINT Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SELECT Syntax", 100000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SET GLOBAL sql_slave_skip_counter Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SET Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SET PASSWORD Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SET sql_log_bin Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SET TRANSACTION Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW AUTHORS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW BINARY LOGS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW BINLOG EVENTS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW CHARACTER SET Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW COLLATION Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW COLUMNS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW CONTRIBUTORS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW CREATE DATABASE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW CREATE EVENT Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW CREATE FUNCTION Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW CREATE PROCEDURE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW CREATE TABLE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW CREATE TRIGGER Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW CREATE VIEW Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW DATABASES Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW ENGINE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW ENGINES Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW ERRORS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW EVENTS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW FUNCTION CODE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW FUNCTION STATUS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW GRANTS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW INDEX Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW INNODB STATUS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW MASTER STATUS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW OPEN TABLES Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW PLUGINS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW PRIVILEGES Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW PROCEDURE CODE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW PROCEDURE STATUS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW PROCESSLIST Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW PROFILE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW PROFILES Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW SCHEDULER STATUS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW SLAVE HOSTS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW SLAVE STATUS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW STATUS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW TABLE STATUS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW TABLES Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW TRIGGERS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW VARIABLES Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW WARNINGS Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SHOW Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "MySQL Compound-Statement Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "SQL Statement Syntax", 5000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "START SLAVE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "STOP SLAVE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "Stored Routine Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "Subquery Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "Trigger Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "TRUNCATE TABLE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "UNINSTALL PLUGIN Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "UNION Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "UPDATE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "USE Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "View Syntax", 1000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "XA Transaction SQL Syntax", 1000);

-- Rating for datatype pages
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "Data Types", 10000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "Numeric Types", 15000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "Date and Time Types", 10000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "The DATETIME, DATE, and TIMESTAMP Types", 15000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "The TIME Type", 10000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "The YEAR Type", 10000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "String Types", 10000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "The CHAR and VARCHAR Types", 15000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "The BINARY and VARBINARY Types", 15000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "The BLOB and TEXT Types", 15000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "The ENUM Type", 10000);
INSERT INTO page_rating (id_manual_type, title, rating) VALUES (10, "The SET Type", 10000);



