/* 
 * Copyright (c) 2007, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */


#include "stdafx.h"

#include "recordset_text_storage.h"
#include "recordset_be.h"
#include "string_utilities.h"
#include "file_functions.h"

#include <sqlite/query.hpp>
#include <boost/foreach.hpp>
#include <fstream>
#include <memory>
#include <errno.h>

#define WIN32 // required by ctemplate to compile on win

#include <ctemplate/template.h>

using namespace bec;
using namespace grt;
using namespace base;

using ctemplate::Template;
using ctemplate::TemplateDictionary;

typedef std::map<std::string, std::string> Templates;
Templates _templates; // data format name -> template name

#define APPEND(literal)  out->Emit("" literal "", sizeof(literal)-1)

// string escaper for CSV tokens, encloses fields with " if needed, depending on the separator
class CSVTokenQuote : public ctemplate::TemplateModifier {
  void Modify(const char* in, size_t inlen,
              const ctemplate::PerExpandData* per_expand_data,
              ctemplate::ExpandEmitter* out, const std::string& arg) const 
  {
    int ch;
    if (arg == "=comma")
      ch = ',';
    else if (arg == "=tab")
      ch = '\t';
    else if (arg == "=semicolon")
      ch = ';';
    else
      ch = ';';
    // check if quotation needed
    if (memchr(in, ch, inlen) || memchr(in, ' ', inlen) || memchr(in, '"', inlen)
        || memchr(in, '\t', inlen) || memchr(in, '\r', inlen) || memchr(in, '\n', inlen))
    {
      out->Emit(std::string("\""));
      for (size_t i = 0; i < inlen; ++i) {
        if (in[i] == '"') 
          APPEND("\"\"");
        else 
          out->Emit(in[i]);
      } 
      out->Emit(std::string("\""));
    }
    else
      out->Emit(std::string(in, inlen));
  }
};
CSVTokenQuote csv_quote;


Recordset_text_storage::Recordset_text_storage(GRTManager *grtm)
:
Recordset_data_storage(grtm)
{
  static bool registered_csvquote = false;
  if (!registered_csvquote)
  {
    registered_csvquote = true;
    ctemplate::AddModifier("x-csv_quote=", &csv_quote);
    //XXX ctemplate::AddModifier("x-sql_quote", &sql_quote);
  }
  
  {
    class Template_names_initializer
    {
    public:
      Template_names_initializer()
      {
        _templates["CSV"]= "CSV.tpl";
        _templates["CSV_semicolon"]= "CSV_semicolon.tpl";
        _templates["tab"]= "tab.tpl";
        _templates["HTML"]= "HTML.tpl";
        _templates["XML"]= "XML.tpl";
        _templates["SQL_inserts"]= "SQL_inserts.tpl";
      }
    };
    static Template_names_initializer template_names_initializer;
  }
}


Recordset_text_storage::~Recordset_text_storage()
{
}


ColumnId Recordset_text_storage::aux_column_count()
{
  throw std::runtime_error("Recordset_text_storage::aux_column_count is not implemented");
}


void Recordset_text_storage::do_apply_changes(const Recordset *recordset, sqlite::connection *data_swap_db)
{
  throw std::runtime_error("Recordset_text_storage::apply_changes is not implemented");
}


void Recordset_text_storage::do_serialize(const Recordset *recordset, sqlite::connection *data_swap_db)
{
  std::string template_name= this->template_name(_data_format);
  if (template_name.empty())
    throw std::runtime_error(strfmt("Unknown data format: %s", _data_format.c_str()));

  bool strings_are_pre_quoted = _data_format == "SQL_inserts";  
  
  std::string tpl_path= _grtm->get_data_file_path("modules/data/sqlide/" + template_name);
  Template *pre_tpl = 0;
  Template *post_tpl = 0;
  Template *tpl= Template::GetTemplate(tpl_path, ctemplate::DO_NOT_STRIP);
  if (!tpl)
  {
    Template::ClearCache();
    throw std::runtime_error(strfmt("Failed to open template file: `%s`", tpl_path.c_str()));
  }
  
  // templates can be all in a single file or be divided in 3 files (pre, body and post)
  // to save memory when exporting large resultsets
  std::string pre_tpl_path;
  std::string post_tpl_path;
  if (g_str_has_suffix(tpl_path.c_str(), ".tpl"))
  {
    std::string name = tpl_path.substr(0, tpl_path.size()-4);
    if (g_file_test((name+".pre.tpl").c_str(), G_FILE_TEST_EXISTS))
    {
      pre_tpl_path = name+".pre.tpl";
      pre_tpl= Template::GetTemplate(pre_tpl_path, ctemplate::DO_NOT_STRIP);
      if (!pre_tpl)
        g_warning("Failed to open template file: `%s`", pre_tpl_path.c_str()); 
      else
        pre_tpl->ReloadIfChanged();
    }      
    if (g_file_test((name+".post.tpl").c_str(), G_FILE_TEST_EXISTS))
    {
      post_tpl_path = name+".post.tpl";
      post_tpl= Template::GetTemplate(post_tpl_path, ctemplate::DO_NOT_STRIP);
      if (!post_tpl)
        g_warning("Failed to open template file: `%s`", post_tpl_path.c_str());
      else
        post_tpl->ReloadIfChanged();
    }
  }    
  
  {
    if (!g_file_set_contents(_file_path.c_str(), "", 1, NULL))
      throw std::runtime_error(strfmt("Failed to open output file: `%s`", _file_path.c_str()));
  }

  tpl->ReloadIfChanged();

  std::auto_ptr<TemplateDictionary> dict(new TemplateDictionary("/"));
  BOOST_FOREACH (const Parameters::value_type &param, _parameters)
    dict->SetValue(param.first, param.second);

  const Recordset::Column_names *column_names= recordset->column_names();
  const Recordset::Column_types &column_types= get_column_types(recordset);
  const Recordset::Column_quoting &column_quoting= get_column_quoting(recordset);
  
  ColumnId visible_col_count= recordset->get_column_count();
  sqlide::QuoteVar qv;
  {
    qv.escape_string= sigc::ptr_fun(sqlide::QuoteVar::escape_ansi_sql_string);
    // swap db (sqlite) stores unknown values as quoted strings
    qv.store_unknown_as_string= true;
    qv.allow_func_escaping= false;
    qv.blob_to_string= (true) ? sqlide::QuoteVar::Blob_to_string() : sigc::ptr_fun(sqlide::QuoteVar::blob_to_hex_string);
  }

  // misc subst variables
  dict->SetValue("INDENT", "\t");

  // headers
  for (ColumnId col= 0; col < visible_col_count; ++col)
  {
    TemplateDictionary* col_dict = dict->AddSectionDictionary("COLUMN");
    col_dict->SetValueWithoutCopy("COLUMN_NAME", (*column_names)[col]);
  }

  // if at least one of pre or post templates exist, then we process the recordset as
  // 1. dump pre
  // 2. for each row, dump the row
  // 3. dump post
  // otherwise, the whole thing is dumped at once
  if (pre_tpl || post_tpl)
  {
    FILE *outf = base_fopen(_file_path.c_str(), "w+");
    if (!outf)
      throw std::runtime_error(strfmt("Could not create file `%s`: %s", _file_path.c_str(), g_strerror(errno)));

    if (pre_tpl)
    {
      std::string result_text;
      pre_tpl->Expand(&result_text, dict.get());

      fprintf(outf, "%s", result_text.c_str());
    }
    
    // data
    {
      const size_t partition_count= recordset->data_swap_db_partition_count();
      std::list<boost::shared_ptr<sqlite::query> > data_queries(partition_count);
      Recordset::prepare_partition_queries(data_swap_db, "select * from `data%s`", data_queries);
      std::vector<boost::shared_ptr<sqlite::result> > data_results(data_queries.size());
      std::vector<TemplateDictionary*> field_dicts;

      TemplateDictionary* row_dict = dict->AddSectionDictionary("ROW");
      
      if (Recordset::emit_partition_queries(data_swap_db, data_queries, data_results))
      {
        bool first_row = true;
        bool next_row_exists= true;
        sqlite::Variant v;
        do
        {
          int field_index = 0;

          // process a single row
          for (size_t partition= 0; partition < partition_count; ++partition)
          {
            boost::shared_ptr<sqlite::result> &data_rs= data_results[partition];
            for (ColumnId col_begin= partition * Recordset::DATA_SWAP_DB_TABLE_MAX_COL_COUNT, col= col_begin,
                 col_end= std::min<ColumnId>(visible_col_count, (partition + 1) * Recordset::DATA_SWAP_DB_TABLE_MAX_COL_COUNT); col < col_end; ++col)
            {
              TemplateDictionary* field_dict;
              ColumnId partition_column= col - col_begin;
              data_rs->get_variant(partition_column, v);

              if (first_row)
              {
                field_dict = row_dict->AddSectionDictionary("FIELD");
                field_dicts.push_back(field_dict);
              }
              else
                field_dict = field_dicts[field_index++];
              field_dict->SetValueWithoutCopy("FIELD_NAME", (*column_names)[col]);
              
              std::string field_value;
              sqlide::VarToStr var_to_str;
              if (sqlide::is_var_null(v)) // for some reason, the apply_visitor stuff isnt handling NULL
                field_value = "NULL";
              else if (strings_are_pre_quoted)
                field_value= column_quoting[col] || sqlide::is_var_null(v)?
                     boost::apply_visitor(qv, column_types[col], v):
                     boost::apply_visitor(var_to_str, v);
              else
                field_value= boost::apply_visitor(var_to_str, v);
              field_dict->SetValue("FIELD_VALUE", field_value);                
            }
          }

          // expand template & flush row
          {
            std::string result_text;
            tpl->Expand(&result_text, dict.get());

            fprintf(outf, "%s", result_text.c_str());
          }          
          
          first_row = false;
          
          BOOST_FOREACH (boost::shared_ptr<sqlite::result> &data_rs, data_results)
            next_row_exists= data_rs->next_row();
        }
        while (next_row_exists);
      }
    }

    if (post_tpl)
    {
      std::string result_text;
      post_tpl->Expand(&result_text, dict.get());
      
      fprintf(outf, "%s", result_text.c_str());
    }

    fclose(outf);
  }
  else // no pre/post separation
  {
    // data
    {
      const size_t partition_count= recordset->data_swap_db_partition_count();
      std::list<boost::shared_ptr<sqlite::query> > data_queries(partition_count);
      Recordset::prepare_partition_queries(data_swap_db, "select * from `data%s`", data_queries);
      std::vector<boost::shared_ptr<sqlite::result> > data_results(data_queries.size());
      if (Recordset::emit_partition_queries(data_swap_db, data_queries, data_results))
      {
        bool next_row_exists= true;
        sqlite::Variant v;
        do
        {
          TemplateDictionary* row_dict = dict->AddSectionDictionary("ROW");
          for (size_t partition= 0; partition < partition_count; ++partition)
          {
            boost::shared_ptr<sqlite::result> &data_rs= data_results[partition];
            for (ColumnId col_begin= partition * Recordset::DATA_SWAP_DB_TABLE_MAX_COL_COUNT, col= col_begin,
                 col_end= std::min<ColumnId>(visible_col_count, (partition + 1) * Recordset::DATA_SWAP_DB_TABLE_MAX_COL_COUNT); col < col_end; ++col)
            {
              ColumnId partition_column= col - col_begin;
              data_rs->get_variant(partition_column, v);
              TemplateDictionary* field_dict = row_dict->AddSectionDictionary("FIELD");
              field_dict->SetValueWithoutCopy("FIELD_NAME", (*column_names)[col]);
              std::string field_value;
              sqlide::VarToStr var_to_str;
              
              if (strings_are_pre_quoted)
                field_value= column_quoting[col] || sqlide::is_var_null(v)?
                    boost::apply_visitor(qv, column_types[col], v):
                    boost::apply_visitor(var_to_str, v);
              else
                field_value= boost::apply_visitor(var_to_str, v);
              field_dict->SetValue("FIELD_VALUE", field_value);                
            }
          }
          BOOST_FOREACH (boost::shared_ptr<sqlite::result> &data_rs, data_results)
          next_row_exists= data_rs->next_row();
        }
        while (next_row_exists);
      }
    }
    
    // expand tempalte & flush result
    {
      std::string result_text;
      tpl->Expand(&result_text, dict.get());
      GError *error = 0;
      
      // use g_file_set because it can handle utf8 filenames
      if (!g_file_set_contents(_file_path.c_str(), result_text.data(), result_text.size(), &error))
      {
        std::string message = error->message;
        g_free(error);
        throw std::runtime_error(strfmt("Failed to write to output file `%s`: %s", _file_path.c_str(), message.c_str()));
      }
    }
  }
}


void Recordset_text_storage::do_unserialize(Recordset *recordset, sqlite::connection *data_swap_db)
{
  throw std::runtime_error("Recordset_text_storage::unserialize is not implemented");
}


void Recordset_text_storage::do_fetch_blob_value(Recordset *recordset, sqlite::connection *data_swap_db, RowId rowid, ColumnId column, sqlite::Variant &blob_value)
{
}


std::string Recordset_text_storage::template_name(const std::string &format) const
{
  Templates::const_iterator i= _templates.find(format);
  return (_templates.end() != i) ? i->second : std::string();
}


std::string Recordset_text_storage::parameter_value(const std::string &name) const
{
  Parameters::const_iterator i= _parameters.find(name);
  return (_parameters.end() != i) ? i->second : std::string();
}
