/* 
 * Copyright (c) 2007, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

using System.Drawing;
using System.Windows.Forms;
using System.Collections.Generic;
using System.Drawing.Drawing2D;

namespace MySQL.Controls
{
  public class FlatMainToolStripRenderer : ToolStripRenderer
	{
    protected override void OnRenderToolStripBackground(ToolStripRenderEventArgs e)
    {
      base.OnRenderToolStripBackground(e); // Erase background with our back color.

      Brush innerBrush = new SolidBrush(Color.FromArgb(188, 199, 216));
      GraphicsPath fillPath = new GraphicsPath();
      GraphicsPath borderPath = new GraphicsPath();
      Pen borderPen = new Pen(Color.FromArgb(206, 211, 231));
      try
      {
        RectangleF bounds = e.ToolStrip.ClientRectangle;
        bounds.X += e.ToolStrip.Padding.Left;
        bounds.Width -= e.ToolStrip.Padding.Horizontal;
        bounds.Y += e.ToolStrip.Padding.Top;
        bounds.Height -= e.ToolStrip.Padding.Vertical;

        float cornerSize = 6;

        borderPen.Width = 1;
        bounds.Width -= 1;
        bounds.Height -= 1;

        // We only need to define the corners as their endpoints are automatically connected.
        borderPath.AddArc(bounds.Right - cornerSize, bounds.Bottom - cornerSize, cornerSize, cornerSize, 0, 90);
        borderPath.AddArc(bounds.Left, bounds.Bottom - cornerSize, cornerSize, cornerSize, 90, 90);
        borderPath.AddArc(bounds.Left, bounds.Top, cornerSize, cornerSize, 180, 90);
        borderPath.AddArc(bounds.Right - cornerSize, bounds.Top, cornerSize, cornerSize, -90, 90);
        borderPath.CloseAllFigures();

        bounds.X -= 0.5f;
        bounds.Y -= 0.5f;
        bounds.Width += 1;
        bounds.Height += 1;

        fillPath.AddArc(bounds.Right - cornerSize, bounds.Bottom - cornerSize, cornerSize, cornerSize, 0, 90);
        fillPath.AddArc(bounds.Left, bounds.Bottom - cornerSize, cornerSize, cornerSize, 90, 90);
        fillPath.AddArc(bounds.Left, bounds.Top, cornerSize, cornerSize, 180, 90);
        fillPath.AddArc(bounds.Right - cornerSize, bounds.Top, cornerSize, cornerSize, -90, 90);
        fillPath.CloseAllFigures();

        e.Graphics.SmoothingMode = SmoothingMode.HighQuality;
        e.Graphics.FillPath(innerBrush, fillPath); // The interior.

        e.Graphics.DrawPath(borderPen, borderPath); // The border.
      }
      finally
      {
        innerBrush.Dispose();
        fillPath.Dispose();
        borderPath.Dispose();
        borderPen.Dispose();
      }
    }

    /// <summary>
    /// Used to render the toolstrip's grip handle.
    /// </summary>
    protected override void OnRenderGrip(ToolStripGripRenderEventArgs e)
    {
      using (Pen linePen = new Pen(Color.FromArgb(98, 113, 140), 2))
      {
        linePen.DashPattern = new float[] { 1f, 1f };

        // Top position is determined by the toolstrip's padding, the inner border and the grip's size,
        // which we shorten by 8 pixels to make it more appealing.
        Point gripStart = new Point(e.GripBounds.X, e.GripBounds.Y + e.ToolStrip.GripMargin.Top);
        Point gripEnd = new Point(gripStart.X, e.GripBounds.Y + e.GripBounds.Height - e.ToolStrip.GripMargin.Bottom);
        e.Graphics.DrawLine(linePen, gripStart, gripEnd);
      }
    }

    protected override void OnRenderSeparator(ToolStripSeparatorRenderEventArgs e)
    {
      using (Pen linePen = new Pen(Color.FromArgb(131, 146, 165)))
      {
        Point lineStart = new Point(e.Item.Bounds.Width / 2, e.ToolStrip.Margin.Top);
        Point lineEnd = new Point(e.Item.Bounds.Width / 2, e.Item.Bounds.Height - e.ToolStrip.Margin.Bottom);
        e.Graphics.DrawLine(linePen, lineStart, lineEnd);
      }
    }
  }

  //------------------------------------------------------------------------------------------------

  public class FlatSubToolStripRenderer : ToolStripRenderer
  {
    protected override void OnRenderToolStripBackground(ToolStripRenderEventArgs e)
    {
      base.OnRenderToolStripBackground(e);
    }

    /// <summary>
    /// Used to render the toolstrip's grip handle.
    /// </summary>
    protected override void OnRenderGrip(ToolStripGripRenderEventArgs e)
    {
      using (Pen linePen = new Pen(Color.FromArgb(98, 113, 140), 2))
      {
        linePen.DashPattern = new float[] { 1f, 1f };

        // Top position is determined by the toolstrip's padding, the inner border and the grip's size,
        // which we shorten by 8 pixels to make it more appealing.
        Point gripStart = new Point(e.GripBounds.X, e.GripBounds.Y + e.ToolStrip.GripMargin.Top);
        Point gripEnd = new Point(gripStart.X, e.GripBounds.Y + e.GripBounds.Height - e.ToolStrip.GripMargin.Bottom);
        e.Graphics.DrawLine(linePen, gripStart, gripEnd);
      }
    }

    protected override void OnRenderSeparator(ToolStripSeparatorRenderEventArgs e)
    {
      using (Pen linePen = new Pen(Color.FromArgb(131, 146, 165)))
      {
        Point lineStart = new Point(e.Item.Bounds.Width / 2, e.ToolStrip.GripMargin.Top);
        Point lineEnd = new Point(e.Item.Bounds.Width / 2, e.Item.Bounds.Height - e.ToolStrip.GripMargin.Bottom);
        e.Graphics.DrawLine(linePen, lineStart, lineEnd);
      }
    }
  }
}
