using System;
using System.IO;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

using MySQL.Utilities;
using MySQL.Grt;
using MySQL.Grt.Db;
using MySQL.Grt.Db.Sql;
using MySQL.Workbench;
using MySQL.Controls;
using ScintillaNet;
using MySQL.Utilities.SysUtils;

namespace MySQL.GUI.Workbench
{
  public partial class DbSqlEditor : Plugins.DockablePlugin
  {
    private List<RecordsetView> pendingRelayouts = new List<RecordsetView>();

    private Db_sql_editor dbSqlEditorBE;
    public Db_sql_editor Backend { get { return dbSqlEditorBE; } }

    protected WbContext wbContext;

    #region Initialization
    public DbSqlEditor(WbContext wbContext, UIForm uiForm)
    {
      this.wbContext = wbContext;
      dbSqlEditorBE = uiForm as Db_sql_editor;
      Initialize();
    }

    protected void Initialize()
    {
      InitializeComponent();

      SizeChanged +=new EventHandler(onSizeChanged);
      grtManager = dbSqlEditorBE.grt_manager();

      sqlSnippetsForm = new SqlSnippetsForm(wbContext);
      sideBar = new SqlIdeSideBar(wbContext);

      TabText = dbSqlEditorBE.caption();

      sqlEditors = new List<SqlEditor>();
      editorTabControl.TabClosing += new EventHandler<MySQL.Controls.TabClosingEventArgs>(editorTabControl_TabClosing);
      editorTabControl.TabClosed += new EventHandler<MySQL.Controls.TabClosedEventArgs>(editorTabControl_TabClosed);
      editorTabControl.SelectedIndexChanged += new EventHandler(editorTabControl_SelectedIndexChanged);

      SqlEditor sqlEditor = AddSqlEditor("", dbSqlEditorBE.active_sql_editor_index());
      ActiveControl = sqlEditor;
      dbSqlEditorBE.sql_editor_new_ui_cb(OnSqlEditorNew);

      logView = new GridView(dbSqlEditorBE.log());
      logView.AutoScroll = true;
      logView.RowHeadersVisible = false;
      logView.Parent = logPanel;
      logView.AllowAutoResizeColumns = false;
      dbSqlEditorBE.log().refresh_ui_cb(logView.ProcessModelRowsChange);
      logView.ProcessModelChange();
      logView.Columns[0].DefaultCellStyle.Alignment = DataGridViewContentAlignment.TopCenter;
      logView.Columns[0].AutoSizeMode = DataGridViewAutoSizeColumnMode.DisplayedCells;
      logView.Columns[1].AutoSizeMode = DataGridViewAutoSizeColumnMode.DisplayedCells;
      logView.Columns[2].AutoSizeMode = DataGridViewAutoSizeColumnMode.DisplayedCells;
      logView.Columns[3].AutoSizeMode = DataGridViewAutoSizeColumnMode.Fill;
      logView.Columns[4].AutoSizeMode = DataGridViewAutoSizeColumnMode.Fill;
      logView.Columns[5].AutoSizeMode = DataGridViewAutoSizeColumnMode.DisplayedCells;
      logView.SelectionMode = DataGridViewSelectionMode.FullRowSelect;
      wrapButton.Checked = logView.WrapMode;
      logView.SelectionChanged += new EventHandler(logView_SelectionChanged);
      logView_SelectionChanged(null, null); // reset log event details
      logView.CellFormatting += new DataGridViewCellFormattingEventHandler(logView_CellFormatting);

      historyEntriesView = new GridView(dbSqlEditorBE.history().entries_model());
      historyEntriesView.MultiSelect = false;
      historyEntriesView.RowHeadersVisible = false;
      historyEntriesView.Parent = historySplitContainer.Panel1;
      dbSqlEditorBE.history().entries_model().refresh_ui_cb(historyEntriesView.ProcessModelRowsChange);
      historyEntriesView.ProcessModelChange();
      historyEntriesView.AutoSizeColumnsMode = DataGridViewAutoSizeColumnsMode.Fill;
      historyEntriesView.SelectionMode = DataGridViewSelectionMode.FullRowSelect;
      historyEntriesView.RowEnter += new DataGridViewCellEventHandler(historyEntriesView_RowEnter);
      historyEntriesView.CellContextMenuStripNeeded += historyEntriesView_CellContextMenuStripNeeded;
      {
        SqlIdeMenuManager.MenuContext popupMenuContext = new SqlIdeMenuManager.MenuContext();
        //popupMenuContext.ActionList = model.action_list;
        popupMenuContext.GetNodesMenuItems = historyEntriesView.Model.get_popup_items_for_nodes;
        popupMenuContext.GetSelectedNodes = historyEntriesView.SelectedNodes;
        popupMenuContext.TriggerNodesAction = historyEntriesView.Model.activate_popup_item_for_nodes;
        SqlIdeMenuManager.InitMenu(contextHistoryEntries_MenuStrip, popupMenuContext);
      }

      historyDetailsView = new GridView(dbSqlEditorBE.history().details_model());
      historyDetailsView.RowHeadersVisible = false;
      historyDetailsView.Parent = historySplitContainer.Panel2;
      dbSqlEditorBE.history().details_model().refresh_ui_cb(historyDetailsView.ProcessModelRowsChange); 
      historyDetailsView.ProcessModelChange();
      historyDetailsView.AutoSizeColumnsMode = DataGridViewAutoSizeColumnsMode.Fill;
      historyDetailsView.SelectionMode = DataGridViewSelectionMode.FullRowSelect;
      historyDetailsView.CellDoubleClick += new DataGridViewCellEventHandler(historyDetailsView_CellDoubleClick);
      wrapButton.Checked = historyDetailsView.WrapMode;

      dbSqlEditorBE.exec_sql_task.finish_cb(AfterExecSql);
      dbSqlEditorBE.exec_sql_task.progress_cb(OnExecSqlProgress);
      dbSqlEditorBE.close_recordset_ui_cb(CloseRecordset);
      dbSqlEditorBE.output_text_ui_cb(RecordsetTextOutput);
      resultSetTextBox.Font = grtManager.get_font_option("workbench.general.Editor:Font");

      dbSqlEditorBE.sql_editor_text_insert_cb(OnSqlEditorTextInsert);

      dbSqlEditorBE.refresh_ui().set_refresh_slot(OnRefresh);
      dbSqlEditorBE.refresh_ui().set_partial_refresh_slot(OnPartialRefresh);

      MySQL.Grt.ActionList actionList = dbSqlEditorBE.action_list;
      actionList.register_action("exec_sql", ExecuteSqlScript);
      actionList.register_action("exec_current_sql_statement", ExecuteCurrentSqlStatement);

      InitLivePhysicalOverviewPage();

      sideBar.TopLevel = false;
      sideArea.Controls.Add(sideBar);
      sideBar.Dock = DockStyle.Fill;
      sideBar.Show();

      outputPageToolStrip.Renderer = new FlatSubToolStripRenderer();
      toolStrip4.Renderer = new FlatSubToolStripRenderer();

      sqlSnippetsForm.Init();
      snippetsPage.Controls.Add(sqlSnippetsForm.contentPanel);
      sqlSnippetsForm.contentPanel.Dock = DockStyle.Fill;
    }
    #endregion

    #region Finalization

    private void Destroy()
    {
      sideBar.Dispose();

      List<long> recordsetKeyList = new List<long>(recordset2placeholder.Count);
      foreach (KeyValuePair<long, RecordsetView> pair in recordset2placeholder)
        recordsetKeyList.Add(pair.Key);
      foreach (long key in recordsetKeyList)
        CloseRecordset(key);

      foreach (SqlEditor sqlEditor in sqlEditors)
        sqlEditor.Dispose();
      sqlEditors.Clear();

      historyDetailsView.Dispose();
      historyEntriesView.Dispose();

      dbSqlEditorBE.Dispose();
    }

    delegate void CollectGarbageDelegate();

    static void CollectGarbage()
    {
      GC.Collect();
    }

    #endregion

    #region Refresh Messages
    public override void RefreshGUI(RefreshType refresh, String str, IntPtr ptr)
    {
      switch (refresh)
      {
        case RefreshType.RefreshOverviewNodeChildren:
          LivePhysicalOverview.RefreshNodeInfo(new NodeId(str));
          break;

      }
    }

    protected void OnRefresh()
    {
      UpdateTooltip();
      livePhysicalOverview.RebuildModelContents();
      sideBar.RefreshEx(this, true);
    }

    protected void OnPartialRefresh(int what)
    {
      Db_sql_editor.PartialRefreshType refresh_type = (Db_sql_editor.PartialRefreshType)what;
      switch (refresh_type)
      {
        case Db_sql_editor.PartialRefreshType.RefreshSidebar:
          UpdateTooltip();
          sideBar.Refresh(this);
          break;
        case Db_sql_editor.PartialRefreshType.RefreshActiveSchemaSelector:
          sideBar.RefreshActiveSchemaSelector(false);
          break;
        case Db_sql_editor.PartialRefreshType.RefreshSchemaTree:
          sideBar.Refresh(this);
          break;
        case Db_sql_editor.PartialRefreshType.RefreshEditor:
          SqlEditor sqlEditor = SqlEditor();
          sqlEditor.SqlText = dbSqlEditorBE.sql_editor().sql();
          sqlEditor.Focus();
          break;
        case Db_sql_editor.PartialRefreshType.RunCurrentScript:
          ExecuteSqlScript();
          break;
        case Db_sql_editor.PartialRefreshType.RefreshSnippets:
          sqlSnippetsForm.SqlSnippetsModel.RefreshModel();
          break;
        case Db_sql_editor.PartialRefreshType.RefreshEditorTitle:
          OnSqlEditorCaptionChange();
          break;
        case Db_sql_editor.PartialRefreshType.RefreshRecordsetTitle:
          OnRecordsetCaptionChange();
          break;
      }
    }
    #endregion

    #region SQL Script Execution
    private void ExecuteSqlScript()
    {
      DoExecuteSqlScript(false);
    }

    private void ExecuteCurrentSqlStatement()
    {
      DoExecuteSqlScript(true);
    }

    private int DoExecuteSqlScript(bool currentStatementOnly)
    {
      SqlEditor sqlEditor = SqlEditor();
      String sql;
      if (currentStatementOnly)
      {
        sqlEditor.RunBackgroundAction(true);
        sql = sqlEditor.CurrentSqlStatement;
      }
      else
      {
        sql = (sqlEditor.Selection.Length > 0) ? sqlEditor.Selection.Text : sqlEditor.Text;
      }

      if (sql.Length == 0)
        return 0;

      outdatedLogEventCount = logView.RowCount;
      resultTabControl.SelectedTab = outputPage;
      if (dbSqlEditorBE.exec_sql_error_count() > 0)
        outputSelector.SelectedIndex = 0; // The actions page on the output page.
      dbSqlEditorBE.exec_sql(sql, currentStatementOnly);

      return 0;
    }

    private int OnExecSqlProgress(float progress, String msg)
    {
      logView.ProcessModelRowsChange();
      logView_SelectionChanged(null, null);
      return 0;
    }

    private int AfterExecSql()
    {
      {
        TabPage firstNewPage = null;
        resultTabControl.SelectedTab = outputPage;
        resultTabControl.Update();

        // close recordsets that have gone
        {
          Dictionary<long, long> recordsets = new Dictionary<long, long>();
          for (int n = 0, rsCount = dbSqlEditorBE.recordset_count(); n < rsCount; ++n)
          {
            MySQL.Grt.Db.Recordset rs = dbSqlEditorBE.recordset(n);
            recordsets.Add(rs.key(), 1);
          }
          List<long> recordsetsToClose = new List<long>();
          foreach (KeyValuePair<long, RecordsetView> pair in recordset2placeholder)
            if (!recordsets.ContainsKey(pair.Key))
              recordsetsToClose.Add(pair.Key);
          foreach (long key in recordsetsToClose)
            CloseRecordset(key);
        }

        // add new recordsets
        for (int n = 0, rsCount = dbSqlEditorBE.recordset_count(); n < rsCount; ++n)
        {
          MySQL.Grt.Db.Recordset rs = dbSqlEditorBE.recordset(n);
          if (recordset2page.ContainsKey(rs.key()))
            continue;

          TabPage page = new TabPage(rs.caption());
          page.BackColor = SystemColors.ButtonFace;
          if (null == firstNewPage)
            firstNewPage = page;

          RecordsetView recordsetView = new RecordsetView();
          components.Add(recordsetView);
          pendingRelayouts.Add(recordsetView);

          recordsetView.Embed(page, rs);
          resultTabControl.TabPages.Add(page);
          resultTabControl.SetCloseButtonVisibility(resultTabControl.TabCount - 1,
            FlatTabControl.CloseButtonVisiblity.ShowButton);
          recordset2page.Add(rs.key(), page);
          recordset2placeholder.Add(rs.key(), recordsetView);
          page2recordset.Add(page, rs);
          recordsetView.ProcessModelChange();
        }

        resultTabControl.SelectedTab = (null == firstNewPage) ? outputPage : firstNewPage;
      }

      logView.ProcessModelRowsChange();
      logView_SelectionChanged(null, null);
      logView.AutoResizeColumn(1);
      logView.AutoResizeColumn(2);
      logView.AutoResizeColumn(5);

      historyEntriesView.ProcessModelRowsChange();
      historyDetailsView.ProcessModelRowsChange();

      if (logView.Rows.Count > 0)
        logView.SetRowSelected(logView.Rows.Count - 1);

      sqlEditors[dbSqlEditorBE.active_sql_editor_index()].Focus();

      return 0;
    }

    void onSizeChanged(object sender, EventArgs e)
    {
      foreach (RecordsetView view in pendingRelayouts)
      {
        // Workaround for data grids added while the form was minimized.
        // They refuse to compute their own layout until a new resize event occurs.
        view.GridView.Dock = DockStyle.None;
        view.GridView.Dock = DockStyle.Fill;
      }
      pendingRelayouts.Clear();

      // Workaround for the anchored clear button. Position is completely wrong if we don't adjust it (at least once).
      clearButton.Location = new Point(outputPage.Width - clearButton.Width - 6, 2);
    }
    #endregion

    #region SQL Editors

    private List<SqlEditor> sqlEditors;

    private SqlEditor SqlEditor()
    {
      return sqlEditors[dbSqlEditorBE.active_sql_editor_index()];
    }

    private int OnSqlEditorNew(int index)
    {
      AddSqlEditor("", index);
      return 0;
    }

    private void OnSqlEditorCaptionChange()
    {
      editorTabControl.TabPages[dbSqlEditorBE.active_sql_editor_index()].Text = dbSqlEditorBE.sql_editor_caption(dbSqlEditorBE.active_sql_editor_index());
    }

    private SqlEditor AddSqlEditor(String filepath, int index)
    {
      if (-1 == index)
        index = dbSqlEditorBE.add_sql_editor();

      SqlEditor sqlEditor = new SqlEditor(grtManager);
      sqlEditor.BE = dbSqlEditorBE.sql_editor(index);
      String caption = dbSqlEditorBE.sql_editor_caption(index);
      TabPage page = new TabPage(caption);
      page.Controls.Add(sqlEditor);
      editorTabControl.TabPages.Add(page);
      editorTabControl.SelectedIndex = index;

      sqlEditor.Selection.Start = sqlEditor.Text.Length;
      sqlEditor.Selection.End = sqlEditor.Text.Length;
      sqlEditor.KeyDown += new KeyEventHandler(sqlEditor_KeyDown);

      sqlEditors.Add(sqlEditor);

      return sqlEditor;
    }

    private void RemoveSqlEditor(int index)
    {
      if (-1 == index)
        index = dbSqlEditorBE.active_sql_editor_index();
      dbSqlEditorBE.remove_sql_editor(index);
      SqlEditor sqlEditor = sqlEditors[index];
      sqlEditors.RemoveAt(index);
      sqlEditor.Dispose();
    }

    private void sqlEditor_KeyDown(object sender, KeyEventArgs e)
    {
      switch (e.KeyCode)
      {
        case Keys.Enter:
          if (e.Control && !e.Shift && !e.Alt)
            ExecuteSqlScript();
          break;
      }
    }

    private int OnSqlEditorTextInsert(String text)
    {
      SqlEditor sqlEditor = SqlEditor();
      sqlEditor.InsertText(sqlEditor.Caret.Position, text);
      sqlEditor.Selection.Start = sqlEditor.Caret.Position + text.Length;
      sqlEditor.Selection.End = sqlEditor.Selection.Start;
      if (sqlEditor.CanFocus)
        sqlEditor.Focus();
      return 0;
    }

    void editorTabControl_SelectedIndexChanged(object sender, EventArgs e)
    {
      dbSqlEditorBE.active_sql_editor_index(editorTabControl.SelectedIndex);
    }

    void editorTabControl_TabClosing(object sender, MySQL.Controls.TabClosingEventArgs e)
    {
      if (!dbSqlEditorBE.sql_editor_will_close(dbSqlEditorBE.active_sql_editor_index()))
        e.canClose = false;
      else
        // Temporarily disconnect our tab change handler to avoid unnecessary changes.
        editorTabControl.SelectedIndexChanged -= editorTabControl_SelectedIndexChanged;
    }

    void editorTabControl_TabClosed(object sender, MySQL.Controls.TabClosedEventArgs e)
    {
      RemoveSqlEditor(dbSqlEditorBE.active_sql_editor_index());

      // Reconnect the tab change handler we disconnected in eidotrTabControl_TabClosing.
      editorTabControl.SelectedIndexChanged += new EventHandler(editorTabControl_SelectedIndexChanged);

      // If the last editor was closed create a new, empty one.
      if (editorTabControl.TabPages.Count > 0)
        dbSqlEditorBE.active_sql_editor_index(editorTabControl.SelectedIndex);
      else
      {
        SqlEditor sqlEditor = AddSqlEditor("", -1);
        dbSqlEditorBE.active_sql_editor_index(0);
        ActiveControl = sqlEditor;
      }
    }

    public void SearchText(String text)
    {
      TabPage page = editorTabControl.TabPages[dbSqlEditorBE.active_sql_editor_index()];
      if (page.Controls[0] is SqlEditor)
      {
        Range textPosition = (page.Controls[0] as SqlEditor).FindReplace.FindNext(text);
        if (textPosition != null)
          textPosition.Select();
      }
    }

    // IWorkbenchDocument override.
    public override UIForm BackendClass
    {
      get { return dbSqlEditorBE; }
    }

    #endregion

    #region Recordsets

    private Dictionary<long, TabPage> recordset2page = new Dictionary<long, TabPage>();
    private Dictionary<long, RecordsetView> recordset2placeholder = new Dictionary<long, RecordsetView>();
    private Dictionary<TabPage, MySQL.Grt.Db.Recordset> page2recordset = new Dictionary<TabPage, MySQL.Grt.Db.Recordset>();

    public MySQL.Grt.Db.Recordset ActiveRecordset
    {
      get
      {
        if (resultTabControl.SelectedTab == null)
          return null;
        return page2recordset.ContainsKey(resultTabControl.SelectedTab) ? page2recordset[resultTabControl.SelectedTab] : null;
      }
    }

    private void OnRecordsetCaptionChange()
    {
      TabPage page = resultTabControl.SelectedTab;
      if (null == page)
        return;
      if (!page2recordset.ContainsKey(page))
        return;

      MySQL.Grt.Db.Recordset rs = page2recordset[page];

      page.Text = rs.caption();
    }

    private int CloseRecordset(long key)
    {
      if (recordset2page.ContainsKey(key))
      {
        TabPage page = recordset2page[key];
        resultTabControl.TabPages.Remove(page);

        recordset2page.Remove(key);

        RecordsetView recordsetView = recordset2placeholder[key];
        recordset2placeholder.Remove(key);
        components.Remove(recordsetView);
        recordsetView.Dispose();

        page2recordset.Remove(page);
        page.Dispose();

        if (IsHandleCreated)
          BeginInvoke(new CollectGarbageDelegate(CollectGarbage));
      }
      return 0;
    }

    private delegate void RecordsetTextOutputDelegate(String text, bool bringToFront);

    private void RecordsetTextOutput(String text, bool bringToFront)
    {
      if (InvokeRequired)
      {
        RecordsetTextOutputDelegate f = new RecordsetTextOutputDelegate(RecordsetTextOutput);
        Invoke(f, new Object[] {text, bringToFront} );
      }
      else
      {
        resultSetTextBox.AppendText(text);
        if (bringToFront)
        {
          resultTabControl.SelectedTab = outputPage;
          outputSelector.SelectedIndex = 1;
        }
      }
    }

    private void resultTabControl_SelectedIndexChanged(object sender, EventArgs e)
    {
      dbSqlEditorBE.active_recordset(ActiveRecordset);
    }

    private void resultTabControl_TabClosing(object sender, TabClosingEventArgs e)
    {
      e.canClose = false;
      MySQL.Grt.Db.Recordset rs = page2recordset.ContainsKey(e.page) ? page2recordset[e.page] : null;
      if (null != rs)
        e.canClose = rs.can_close();
    }

    private void resultTabControl_TabClosed(object sender, TabClosedEventArgs e)
    {
      TabPage page = e.page;
      MySQL.Grt.Db.Recordset rs = page2recordset.ContainsKey(e.page) ? page2recordset[e.page] : null;
      if (null != rs)
        rs.close();
    }

    private void renamePage_ToolStripMenuItem_Click(object sender, EventArgs e)
    {
      TabPage page = resultTabControl.SelectedTab;
      if (null == page)
        return;
      if (!page2recordset.ContainsKey(page))
        return;

      MySQL.Grt.Db.Recordset rs = page2recordset[page];
      rs.caption();
      
      String newName = page.Text;
      if (DialogResult.OK == StringInputForm.ShowModal("New name of the page", "", "", ref newName))
      {
        rs.caption(newName);
        page.Text = rs.caption();
      }
    }

    private void tabControl_MouseClick(object sender, MouseEventArgs e)
    {
      switch (e.Button)
      {
        case MouseButtons.Right:
          {
            TabPage page = resultTabControl.SelectedTab;
            if (null == page)
              break;
            renamePage_ToolStripMenuItem.Enabled = page2recordset.ContainsKey(page);
            Point p = resultTabControl.PointToScreen((Point)e.Location);
            tabControlContextMenuStrip.Show(p);
          }
          break;
      }
    }
    #endregion

    #region Log Panel

    private GridView logView;
    private int outdatedLogEventCount = 0; //! to be taken from db sql editor back-end

    void logView_CellFormatting(object sender, DataGridViewCellFormattingEventArgs e)
    {
      if (e.RowIndex < outdatedLogEventCount)
        e.CellStyle.ForeColor = Color.Gray;
    }

    void logView_SelectionChanged(object sender, EventArgs e)
    {
      String logEventAction = "";
      String logEventMessage = "";

      if (logView.SelectedRows.Count == 0)
      {
        logEventDetailHeaderLabel.Text = "";
      }
      else
      {
        int logEventNo = logView.SelectedRows[0].Index;
        int logEventTypeCode = 0;
        String logEventTime = "";
        dbSqlEditorBE.get_log_event_details(logEventNo, out logEventTypeCode, out logEventTime, out logEventAction, out logEventMessage);
        String logEventType;
        switch (logEventTypeCode)
        {
          case (int)Msg_type.MT_error:
            logEventType = "Error";
            break;
          case (int)Msg_type.MT_warning:
            logEventType = "Warning";
            break;
          case (int)Msg_type.MT_info:
            logEventType = "Info";
            break;
          default:
            logEventType = "Unknown";
            break;
        }
        logEventDetailHeaderLabel.Text = String.Format("Event {0} of type {1} at {2}", logEventNo + 1, logEventType, logEventTime);
      }
      logEventDetailActionTextBox.Text = logEventAction;
      logEventDetailMessageTextBox.Text = logEventMessage;
    }
    #endregion

    #region History Panel

    private GridView historyEntriesView;
    private GridView historyDetailsView;

    ContextMenuStrip contextHistoryEntries_MenuStrip = new ContextMenuStrip();

    private void wrapButton_Click(object sender, EventArgs e)
    {
      logView.WrapMode = !logView.WrapMode;
      wrapButton.Checked = logView.WrapMode;

      historyDetailsView.WrapMode = wrapButton.Checked;
      resultSetTextBox.WordWrap = wrapButton.Checked;
    }

    void historyEntriesView_RowEnter(Object sender, DataGridViewCellEventArgs e)
    {
      dbSqlEditorBE.history().current_entry(e.RowIndex);
    }

    private void historyEntriesView_CellContextMenuStripNeeded(object sender, DataGridViewCellContextMenuStripNeededEventArgs e)
    {
      e.ContextMenuStrip = contextHistoryEntries_MenuStrip;
    }

    void loadSelectedHistoryItems(bool overwrite)
    {
      if (null == historyEntriesView.CurrentRow)
        return;

      List<int> sel_indexes = new List<int>();
      foreach (DataGridViewRow row in historyDetailsView.SelectedRows)
        sel_indexes.Add(row.Index);
      sel_indexes.Sort();
      String sql = dbSqlEditorBE.restore_sql_from_history(historyEntriesView.CurrentRow.Index, sel_indexes);

      SqlEditor sqlEditor = SqlEditor();
      sqlEditor.IsRefreshEnabled = true;
      if (overwrite)
        sqlEditor.SqlText = sql;
      else
        sqlEditor.SqlText += sql;
    }

    void historyDetailsView_CellDoubleClick(Object sender, DataGridViewCellEventArgs e)
    {
      loadSelectedHistoryItems(false);
    }

    private void appendToCurrentSqlMenuItem_Click(object sender, EventArgs e)
    {
      loadSelectedHistoryItems(false);
    }

    private void replaceCurrentSqlMenuItemMenuItem_Click(object sender, EventArgs e)
    {
      loadSelectedHistoryItems(true);
    }

    #endregion

    #region Sidebar

    private SqlIdeSideBar sideBar;

    #endregion

    #region Live Physical Overview

    ModelOverviewForm livePhysicalOverview;

    private void InitLivePhysicalOverviewPage()
    {
      schemaOverviewPlaceholder.SuspendLayout();
      try
      {
        livePhysicalOverview = new ModelOverviewForm(wbContext, dbSqlEditorBE.live_physical_overview());
        livePhysicalOverview.skipHeader = true;
        livePhysicalOverview.contentHeaderPanel.Parent = schemaOverviewPlaceholder;
        livePhysicalOverview.contentHeaderPanel.Dock = DockStyle.Fill;
        livePhysicalOverview.contentHeaderPanel.Padding = new Padding();
        livePhysicalOverview.contentHeaderPanel.Show();
        livePhysicalOverview.contentHeaderPanel.BackColor = Color.White;
        livePhysicalOverview.RebuildModelContents();
        livePhysicalOverview.shadowPicture.Visible = false;
        Panel p = new Panel();
        p.Dock = DockStyle.Fill;
        p.BackColor = Color.White;
        p.Parent = livePhysicalOverview.contentHeaderPanel.Parent;

        WorkbenchToolbarManager.ToolbarCallback cb = delegate (String name)
        {
          return dbSqlEditorBE.live_physical_overview().activate_toolbar_item(new NodeId(0), name);
        };

        new WorkbenchToolbarManager(wbContext, toolStrip4).UpdateToolbar(toolStrip4,
          dbSqlEditorBE.live_physical_overview().get_toolbar_items(new NodeId(0)),
          cb);

        if (!dbSqlEditorBE.is_physical_overview_enabled())
          resultTabControl.TabPages.Remove(tpSchemaOverview);
      }
      finally
      {
        schemaOverviewPlaceholder.ResumeLayout();
      }
    }

    public Overview LivePhysicalOverviewBE
    {
      get { return dbSqlEditorBE.live_physical_overview(); }
    }

    public ModelOverviewForm LivePhysicalOverview
    {
      get { return livePhysicalOverview; }
    }
    #endregion

    #region SQL Snippets
    private SqlSnippetsForm sqlSnippetsForm;
    #endregion

    #region Form Layout
    private void LoadFormState()
    {
      int sidebarWidth = wbContext.read_state("sidebar_width", "query_editor", 200);
      bool left_aligned = wbContext.read_option_value("", "Sidebar:RightAligned", "0") == "0";

      if (left_aligned)
      {
        if (mainSplitContainer.Width > sidebarWidth)
          mainSplitContainer.SplitterDistance = sidebarWidth;
      }
      else
      {
        mainSplitContainer.FixedPanel = FixedPanel.Panel2;
        contentSplitContainer.Parent = mainSplitContainer.Panel1;
        sideArea.Parent = mainSplitContainer.Panel2;
        if (mainSplitContainer.Width > sidebarWidth)
          mainSplitContainer.SplitterDistance = mainSplitContainer.Width - mainSplitContainer.SplitterWidth - sidebarWidth;
      }
    }

    private void DbSqlEditor_Shown(object sender, EventArgs e)
    {
      LoadFormState();
    }

    private void mainSplitContainer_SplitterMoved(object sender, SplitterEventArgs e)
    {
      int sidebarWidth;
      bool left_aligned = wbContext.read_option_value("", "Sidebar:RightAligned", "0") == "0";
      if (left_aligned)
        sidebarWidth = mainSplitContainer.SplitterDistance;
      else
        sidebarWidth = mainSplitContainer.Width - mainSplitContainer.SplitterWidth - mainSplitContainer.SplitterDistance;
      wbContext.save_state("sidebar_width", "query_editor", sidebarWidth);
    }

    /// <summary>
    /// Updates the editor's tab tooltip to contain the current connection info.
    /// </summary>
    public void UpdateTooltip()
    {
      ToolTipText = dbSqlEditorBE.connection_info();
    }

    void outputPaneIndexChanged(object sender, System.EventArgs e)
    {
      ToolStripComboBox selector = sender as ToolStripComboBox;
      if (outputPageContent.Controls.Count > 0)
        outputPageContent.Controls.RemoveAt(0);
      switch (selector.SelectedIndex)
      {
        case 0:
          outputPageContent.Controls.Add(actionPage);
          actionPage.Dock = DockStyle.Fill;
          clearButton.Enabled = false;
          break;
        case 1:
          outputPageContent.Controls.Add(textOutputPage);
          textOutputPage.Dock = DockStyle.Fill;
          clearButton.Enabled = true;
          break;
        case 2:
          outputPageContent.Controls.Add(historyPage);
          historyPage.Dock = DockStyle.Fill;
          clearButton.Enabled = false;
          break;
      }
    }

    void clearButton_Click(object sender, System.EventArgs e)
    {
      resultSetTextBox.Clear();
    }

    #endregion
  }
}
