﻿/* 
 * Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Text;
using System.Text.RegularExpressions;
using System.Windows.Forms;

using MySQL.Utilities;

using ScintillaNet;

namespace MySQL.Grt.Db.Sql
{
  public class SqlEditor : Scintilla
  {
    private Sql_editor sqlEditorBE;
    private GrtManager grtManager;

    private Timer timer;
    private List<ManagedRange> messagesRanges = new List<ManagedRange>();
    private List<String> messages = new List<String>();

    private MenuManager menuManager = new MenuManager();

    public delegate void BackgroundActionDelegate(bool sync);
    public BackgroundActionDelegate BackgroundAction;

    public SqlEditor(GrtManager grtManager)
    {
      GrtManager = grtManager;
      Initialize();
    }

    protected override void Dispose(bool disposing)
    {
      if (disposing && (sqlEditorBE != null))
      {
        BackgroundAction = null;
        sqlEditorBE.Dispose();
        sqlEditorBE = null;
      }
      base.Dispose(disposing);
    }

    public GrtManager GrtManager
    {
      get { return grtManager; }
      set { grtManager = value; }
    }

    public Sql_editor BE
    {
      get { return sqlEditorBE; }
      set
      {
        if (null != sqlEditorBE)
          sqlEditorBE.reset_fe_callbacks();
        sqlEditorBE = value;
        if (sqlEditorBE == null)
          return;

        sqlEditorBE.set_report_sql_statement_border_delegate(ProcessSqlStatementBorder);
        sqlEditorBE.sql_parser_err_cb(ProcessSqlError);
        sqlEditorBE.set_insert_text_delegate(DoInsertText);
        sqlEditorBE.set_replace_selected_text_delegate(DoReplaceSelectedText);
        sqlEditorBE.set_do_search_delegate(DoTextSeach);
        sqlEditorBE.set_current_statement_delegate(GetCurrentSqlStatement);
        sqlEditorBE.set_change_selected_range_delegate(ChangeSelectedRange);
        sqlEditorBE.set_change_cursor_pos_delegate(ChangeCursorPos);

        SelectionChanged += new EventHandler(TextSelectionChanged);

        SuspendLayout();

        // customized parameters
        ConfigurationManager.Language = sqlEditorBE.string_option("ConfigurationManager.Language");
        Indentation.IndentWidth = sqlEditorBE.int_option("Indentation.IndentWidth");
        Indentation.TabWidth = sqlEditorBE.int_option("Indentation.TabWidth");
        Indentation.UseTabs = sqlEditorBE.int_option("Indentation.UseTabs") != 0;
        //! todo: virtualize the rest params

        //EndOfLine.IsVisible = true;

        ResumeLayout();
      }
    }

    void TextSelectionChanged(object sender, EventArgs e)
    {
      int start = Selection.Start;
      int end = Selection.End;
      sqlEditorBE.set_selected_range(start, end);

      sqlEditorBE.set_cursor_pos(CurrentPos);
    }

    protected virtual void Initialize()
    {
      BeginInit();

      Caret.IsSticky = true;
      Dock = DockStyle.Fill;
      Printing.PageSettings.Color = false;
      UseFont = true;
      //!DocumentChange += new System.EventHandler<NativeScintillaEventArgs>(sqlEditorControl_Change);
      Font = grtManager.get_font_option("workbench.general.Editor:Font");
      Margins.Margin0.Width = 35; // line numbers
      Margins.Margin1.Width = 16; // markers
      Margins.Margin2.Width = 16; // indicators
      Indentation.BackspaceUnindents = true;
      Indentation.SmartIndentType = SmartIndent.Simple;

      // errors indication
      Indicator errInd = Indicators[0];
      errInd.Color = Color.Red;
      errInd.Style = IndicatorStyle.Squiggle;
      errInd.IsDrawnUnder = true;

      errInd = Indicators[1];
      errInd.Color = Color.Red;
      errInd.Style = IndicatorStyle.RoundBox;
      errInd.IsDrawnUnder = true;

      Marker marker = Markers[1];
      String path = "images/ui/editor_error.xpm";
      if (File.Exists(path))
      {
        StreamReader reader = File.OpenText(path);
        String icon = reader.ReadToEnd();
        marker.SetImage(icon);
      }
      else
        marker.BackColor = Color.Red;

      marker = Markers[0];
      path = "images/ui/editor_statement.xpm";
      if (File.Exists(path))
      {
        StreamReader reader = File.OpenText(path);
        String icon = reader.ReadToEnd();
        marker.SetImage(icon);
      }
      else
        marker.BackColor = Color.Blue;

      // fixes improper ViewportSize from start.
      // once row is stretched enough to make thumb appear this thumb becomes irreversible.
      Scrolling.HorizontalWidth = 1;

      NativeInterface.SetCodePage((int)Constants.SC_CP_UTF8);
      EndOfLine.Mode = EndOfLineMode.Crlf;

      timer = new Timer();
      timer.Interval = 500;
      timer.Tick += new EventHandler(BackgroundActionTimer);

      BackgroundAction = CheckSql;

      TextDeleted += new System.EventHandler<TextModifiedEventArgs>(TextModified);
      TextInserted += new System.EventHandler<TextModifiedEventArgs>(TextModified);

      DwellStart += new EventHandler<ScintillaMouseEventArgs>(OnDwellStart);
      DwellEnd += new EventHandler(OnDwellEnd);
      NativeInterface.SetMouseDwellTime(200);

      NativeInterface.UsePopUp(false);
      MouseUp += new MouseEventHandler(OnMouseUp);

      EndInit();
    }

    void OnMouseUp(object sender, MouseEventArgs e)
    {
      if (e.Button == MouseButtons.Right)
      {
        List<MySQL.Grt.MenuItem> items = sqlEditorBE.get_context_menu();
    
        if (items.Count > 0)
        {      
          foreach (MySQL.Grt.MenuItem item in items)
          {        
            bool enabled = item.get_enabled();
            switch (item.get_name())
            {
              case "undo":
                enabled = UndoRedo.CanUndo;
                break;
              case "redo":
                enabled = UndoRedo.CanRedo;
                break;
              case "cut":
              case "copy":
                enabled = Selection.Length > 0;
                break;
              case "paste":
              {
                IDataObject iData = System.Windows.Forms.Clipboard.GetDataObject();
                enabled = iData.GetDataPresent(DataFormats.Text);
                break;
              }
            }
            item.set_enabled(enabled);
          }

          menuManager.ShowContextMenu(this, items, e.X, e.Y, new EventHandler(PopupActionHandler));
        }
      }
    }

    void PopupActionHandler(object sender, EventArgs e)
    {
      ToolStripItem item = sender as ToolStripItem;

      if (item != null)
      {
        switch (item.Name)
        {
          case "undo":
            UndoRedo.Undo();
            break;
          case "redo":
            UndoRedo.Redo();
            break;
          case "cut":
            Clipboard.Cut();
            break;
          case "copy":
            Clipboard.Copy();
            break;
          case "paste":
            Clipboard.Paste();
            break;
          case "delete":
            Selection.Text = "";
            break;
          case "select_all":
            Selection.SelectAll();
            break;
          default:
            sqlEditorBE.activate_context_menu_item(item.Name);
            break;
        }
      }
    }

    public String SqlText
    {
      get { return Text; }
      set
      {
        if (!IsRefreshEnabled)
          return;

        IsRefreshEnabled = false;

        BackgroundActionDelegate backgroundAction = BackgroundAction;
        BackgroundAction = null;

        int caretPosition = Caret.Position;

        if (null != sqlEditorBE)
        {
          switch (sqlEditorBE.eol())
          {
            case "\n": EndOfLine.Mode = EndOfLineMode.LF; break;
            case "\r": EndOfLine.Mode = EndOfLineMode.CR; break;
            case "\r\n": EndOfLine.Mode = EndOfLineMode.Crlf; break;
          }
        }

        // Split text loading into smaller pieces to avoid big memory impacts for large
        // content while Scintilla.NET prepares it for the native control.
        // Does only partially solve the trouble with big files as the content is copied around
        // on various situations and the backend control also holds a copy.
        Text = "";
        int position = 0;
        while (position < value.Length)
        {
          int remaining = value.Length - position;
          String part = value.Substring(position, (remaining >= 100000) ? 100000 : remaining);
          AppendText(part);
          position += part.Length;
        }
        //Text = value;
        IsDirty = false;

        if (caretPosition > Text.Length)
          caretPosition = Text.Length;
        Selection.Start = Selection.End = caretPosition;

        BackgroundAction = backgroundAction;

        CheckSql(false);
      }
    }

    private void TextModified(object sender, TextModifiedEventArgs e)
    {
      if (null != BackgroundAction)
      {
        timer.Stop();
        timer.Start();
      }
      sqlEditorBE.sql(Text);
    }

    public void RunBackgroundAction(bool sync)
    {
      if (!IsDirty)
        return;

      IsDirty = false;
      if (null != BackgroundAction)
        BackgroundAction(sync);
    }

    private void BackgroundActionTimer(Object obj, EventArgs args)
    {
      RunBackgroundAction(false);
    }

    public void CheckSql(bool sync)
    {
      if (!IsSqlCheckEnabled)
        return;
      if (CheckSql != BackgroundAction)
        IsSqlCheckEnabled = false;

      ControlUtilities.SuspendDrawing(this);
      ResetSqlCheckState();
      if (null == sqlEditorBE)
      {
        ControlUtilities.ResumeDrawing(this);
        return;
      }
      sqlEditorBE.sql(Text);
      sqlEditorBE.check_sql(sync);
      ControlUtilities.ResumeDrawing(this);
    }

    public int ProcessSqlStatementBorder(int beginLineNo, int beginLinePos, int endLineNo, int endLinePos)
    {
      beginLineNo -= 1;
      endLineNo -= 1;
      Lines[beginLineNo].AddMarker(0);
      return 0;
    }

    public int ProcessSqlError(int errTokLine, int errTokLinePos, int errTokLen, String msg)
    {
      errTokLine -= 1;
      try
      {
        errTokLinePos += Lines[errTokLine].Range.Start;
        ManagedRange range = new ManagedRange(errTokLinePos, errTokLinePos + errTokLen, this);
        
        SuspendLayout();
        range.SetIndicator(0);
        range.SetIndicator(1);
        Lines[errTokLine].AddMarker(1);
        ResumeLayout();

        messages.Add(msg);
        messagesRanges.Add(range);
        ManagedRanges.Add(range);
      }
      catch (Exception) { }
      return 0;
    }

    public void ResetSqlCheckState()
    {
      // errors
      {
        timer.Stop();
        foreach (ManagedRange managedRange in messagesRanges)
          if (!managedRange.IsDisposed)
            managedRange.Dispose();
        messagesRanges.Clear();
        messages.Clear();
      }

      Range range = new Range(0, RawText.Length, this);
      range.ClearIndicator(0);
      range.ClearIndicator(1);
      Markers.DeleteAll(Markers[1]);
      Markers.DeleteAll(Markers[0]);
    }

    public bool IsRefreshEnabled
    {
      get { return (null == BE) ? true : BE.is_refresh_enabled(); }
      set { if (null != BE) BE.is_refresh_enabled(value); }
    }

    public bool IsSqlCheckEnabled
    {
      get { return (null == BE) ? true : BE.is_sql_check_enabled(); }
      set { if (null != BE) BE.is_sql_check_enabled(value); }
    }

    private String GetCurrentSqlStatement()
    {
      return CurrentSqlStatement;
    }

    public String CurrentSqlStatement
    {
      get
      {
        int stmtBeginLineNo = -1;
        int stmtBeginLinePos = -1;
        int stmtEndLineNo = -1;
        int stmtEndLinePos = -1;
        sqlEditorBE.get_sql_statement_border_by_line_pos(
          (Selection.Range.StartingLine.Number)+1, (Selection.Range.Start - Selection.Range.StartingLine.StartPosition),
          ref stmtBeginLineNo, ref stmtBeginLinePos, ref stmtEndLineNo, ref stmtEndLinePos);
        if (stmtBeginLineNo == -1)
          return "";
        stmtBeginLineNo -= 1;
        stmtEndLineNo -= 1;
        return GetRange(Lines[stmtBeginLineNo].StartPosition + stmtBeginLinePos, Lines[stmtEndLineNo].StartPosition + stmtEndLinePos).Text;
      }
    }

    /// <summary>
    /// Called by the backend if the selection in the editor must be changed.
    /// </summary>
    /// <param name="start"></param>
    /// <param name="stop"></param>
    private void ChangeSelectedRange(int start, int stop)
    {
      Selection.Start = start;
      Selection.End = stop;
    }

    /// <summary>
    /// Called by the backend if the caret position must be changed.
    /// </summary>
    /// <param name="position"></param>
    private void ChangeCursorPos(int position)
    {
      Caret.Position = position;
    }

    void OnDwellStart(object sender, ScintillaMouseEventArgs e)
    {
      if (!Visible || !ClientRectangle.Contains(PointToClient(MousePosition)))
        return;

      int pos = PositionFromPoint(e.X, e.Y);
      if (pos == -1)
        return;
      int messageIndex = -1;
      if (NativeInterface.IndicatorValueAt(0, pos) == 1)
      {
        for (int n = 0; n < messagesRanges.Count; ++n)
        {
          if (messagesRanges[n].PositionInRange(pos))
          {
            messageIndex = n;
            break;
          }
        }
      }
      if ((messageIndex >= 0) && (messageIndex < messages.Count))
        CallTip.Show(messages[messageIndex], pos);
    }

    void OnDwellEnd(object sender, EventArgs e)
    {
      CallTip.Cancel();
    }

    protected override void OnLeave(EventArgs e)
    {
      base.OnLeave(e);
      CallTip.Cancel();
    }

    int DoReplaceSelectedText(String newText)
    {
      int oldStart = Selection.Start;
      Selection.Text = newText;
      Selection.Start = oldStart;
      Selection.End = Selection.Start + newText.Length;
      return 0;
    }

    int DoInsertText(String text)
    {
      NativeInterface.AddText(Encoding.UTF8.GetByteCount(text), text);
      return 0;
    }

    public void ShowFindReplaceDialog()
    {
      // Scintilla.NET already has an implementation of the search/replace dialog and all the
      // usual features (including search/replace in the current selection etc.). So we use that
      // instead of our own dialog.
      //sqlEditorBE.show_find_replace_dialog();
      Commands.Execute(BindableCommand.ShowReplace);
    }

    bool DoTextSeach(String searchText, String replaceText, int flags)
    {
      if (searchText.Length == 0)
        return false;

      Sql_editor.SearchFlags searchFlags = (Sql_editor.SearchFlags) flags;
      SearchFlags scintillaFlags = SearchFlags.Empty;

      bool previous = (searchFlags & Sql_editor.SearchFlags.SearchPrevious) != 0;
      bool useRegEx = (searchFlags & Sql_editor.SearchFlags.SearchUseRegularExpression) != 0;
      bool replace = (searchFlags & Sql_editor.SearchFlags.SearchDoReplace) != 0;
      bool all = (searchFlags & Sql_editor.SearchFlags.SearchAll) != 0;

      Range searchRange = new Range(0, RawText.Length, this);
      if ((searchFlags & Sql_editor.SearchFlags.SearchMatchCase) != 0)
        scintillaFlags |= SearchFlags.MatchCase;
      if ((searchFlags & Sql_editor.SearchFlags.SearchMatchWholeWord) != 0)
        scintillaFlags |= SearchFlags.WholeWord;

      Range result = null;
      bool canClose = false;
      if (useRegEx)
      {
        Regex expression = new Regex(searchText);
        if (replace)
        {
          if (all)
          {
            FindReplace.ReplaceAll(expression, replaceText);
            canClose = true;
          }
          else
          {
            // There is no overload to replace with RE search, so we have to handle that ourselves.
            if (previous)
              result = FindReplace.FindPrevious(expression, true);
            else
              result = FindReplace.FindNext(expression, true);

            if (result != null)
            {
              result.Text = replaceText;
              result.End = result.Start + replaceText.Length;
            }
          }
        }
        else
          if (previous)
            result = FindReplace.FindPrevious(expression, true);
          else
            result = FindReplace.FindNext(expression, true);
      }
      else
      {
        if (replace)
        {
          if (all)
          {
            FindReplace.ReplaceAll(searchText, replaceText, scintillaFlags);
            canClose = true;
          }
          else
          {
            if (previous)
              result = FindReplace.ReplacePrevious(searchText, replaceText, true, scintillaFlags);
            else
              result = FindReplace.ReplaceNext(searchText, replaceText, true, scintillaFlags);
          }
        }
        else
        {
          if (previous)
            result = FindReplace.FindPrevious(searchText, true, scintillaFlags);
          else
            result = FindReplace.FindNext(searchText, true, scintillaFlags);
        }
      }
      if (result != null)
      {
        canClose = true;
        result.Select();
      }

      return canClose;
    }
  }
}
