/* 
 * Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#import "MySQLViewEditor.h"
#import "MCPPUtilities.h"
#include "grtdb/db_object_helpers.h" // get_rdbms_for_db_object()

#import "DbPrivilegeEditorTab.h"
#import "string_utilities.h"

@implementation DbMysqlViewEditor

static void call_refresh(DbMysqlViewEditor *self)
{
  [self performSelectorOnMainThread:@selector(refresh) withObject:nil waitUntilDone:YES];
}

- (id)initWithModule:(grt::Module*)module GRTManager:(bec::GRTManager*)grtm arguments:(const grt::BaseListRef&)args
{
  self= [super initWithModule:module GRTManager:grtm arguments:args];
  if (self != nil)
  {
    // load GUI. Top level view in the nib is the NSTabView that will be docked to the main window
    if (![[NSBundle bundleForClass:[self class]] loadNibFile:@"MySQLViewEditor"
                                           externalNameTable:[NSDictionary dictionaryWithObject:self forKey:NSNibOwner] 
                                                    withZone:nil])
      NSLog(@"Could not load MySQLViewEditor.xib");
    
    // take the minimum size of the view from the initial size in the nib.
    // Therefore the nib should be designed as small as possible
    // note: the honouring of the min size is not yet implemented
    [self setMinimumSize: [tabView frame].size];
    
    [self reinitWithArguments: args];
    
    if (mBackEnd && mBackEnd->is_editing_live_object())
      [tabView removeTabViewItem: [tabView tabViewItemAtIndex: [tabView indexOfTabViewItemWithIdentifier:@"comment"]]];
  }
  return self;
}


- (void)reinitWithArguments: (const grt::BaseListRef&)args
{
  [super reinitWithArguments: args];
  
  delete mBackEnd;
  
  // setup the editor backend with the schema object (args[0])
  mBackEnd= new MySQLViewEditorBE(_grtm, db_mysql_ViewRef::cast_from(args[0]), get_rdbms_for_db_object(args[0]));
  
  // register a callback that will make [self refresh] get called
  // whenever the backend thinks its needed to refresh the UI from the backend data (ie, the
  // edited object was changed from somewhere else in the application)
  mBackEnd->set_refresh_ui_slot(sigc::bind(sigc::ptr_fun(call_refresh), self));
  
  NSUInteger index= [tabView indexOfTabViewItemWithIdentifier: @"privileges"];
  if (index != NSNotFound)
    [tabView removeTabViewItem: [tabView tabViewItemAtIndex: index]];
  [mPrivileges release];
  
  if (!mBackEnd->is_editing_live_object())
  {
    NSTabViewItem *tabItem= [[NSTabViewItem alloc] initWithIdentifier:@"privileges"];
    mPrivileges= [[DbPrivilegeEditorTab alloc] initWithObjectEditor: mBackEnd];
    [tabItem setView: [mPrivileges view]];
    [tabItem setLabel: @"Privileges"];
    [tabView addTabViewItem: tabItem];
  }
  [self setupEditor]; 
  
  // update the UI
  [self refresh];
}


- (void) dealloc
{
  [mPrivileges release];
  [mSyntaxCheckTimer invalidate];
  
  delete mBackEnd;
  [super dealloc];
}


/** Fetches object info from the backend and update the UI
 */
- (void) refresh
{
  if (mBackEnd)
  {
    std::string sql;
    std::string code;
    
    [nameText setStringValue: [NSString stringWithCPPString: mBackEnd->get_name()]];
    [self updateTitle: [self title]];
    
    code= mBackEnd->get_sql();
    
    if (code.empty())
    {
      code= mBackEnd->get_query();
      sql.append(code);
    }
    else
      sql.append(code);
    
    if (mBackEnd->get_sql_editor()->is_refresh_enabled())
    {
      NSString *newstr = [NSString stringWithCPPString: sql];
      mBackEnd->get_sql_editor()->is_refresh_enabled(false);
      if (![[codeText string] isEqual: newstr])
        [codeText setString: newstr];
    }
    
    [commentText setString: [NSString stringWithCPPString: mBackEnd->get_comment()]];    
  }
}


- (id)identifier
{
  // an identifier for this editor (just take the object id)
  return [NSString stringWithCPPString:mBackEnd->get_object().id()];
}


- (NSString*)title
{
  // the title for the editor
  return [NSString stringWithCPPString: base::strfmt("%s - View", mBackEnd->get_name().c_str())];
}


- (NSView*)dockableView
{
  // the view to be docked to the main window
  return tabView;
}


- (BOOL)matchesIdentifierForClosingEditor:(NSString*)identifier
{
  return mBackEnd->should_close_on_delete_of([identifier UTF8String]);
}

//--------------------------------------------------------------------------------------------------

- (void) updateBackend
{
  mBackEnd->set_query([[codeText string] UTF8String], false);
  
  // Name field could change.
  if (!self->mBackEnd->get_name().compare([[self->nameText stringValue] UTF8String]))
    [self->nameText setStringValue: [NSString stringWithCPPString:self->mBackEnd->get_name()]];
}

//--------------------------------------------------------------------------------------------------

- (void) textDidEndEditing: (NSNotification *) aNotification
{
  if ([aNotification object] == codeText)
  {    
    [self updateBackend];
  }
}

//--------------------------------------------------------------------------------------------------

/**
 * Callback from the backend telling us about syntax errors it founds.
 */
static int process_syntax_error(const int line, const int err_tok_line_pos, const int err_tok_len,
                                const std::string& err_msg, DbMysqlViewEditor* self)
{
  Scintilla::ScintillaCocoa *backend= self->codeText.backend;
  if (backend)
  {
    if (self->mErrorCount == 0)
      [self resetSyntaxErrors];
    
    int line_start_pos= backend->WndProc(SCI_POSITIONFROMLINE, line - 1, 0);
    
    backend->WndProc(SCI_SETINDICATORCURRENT, 0, 0);
    backend->WndProc(SCI_INDICATORFILLRANGE, line_start_pos + err_tok_line_pos, err_tok_len);
    
    backend->WndProc(SCI_MARKERADD, line - 1, 1);
    
    ++self->mErrorCount;
    
    [self->codeText setStatusText: [NSString stringWithFormat: @"%d error(s) found.", self->mErrorCount]];
  }
  return 0;
}


static void process_syntax_done(DbMysqlViewEditor *self)
{
  if (self->mErrorCount == 0)
    [self resetSyntaxErrors];
}

//--------------------------------------------------------------------------------------------------

/**
 * Prepares the code editor and the backend to have proper syntax highlighting, error parsing etc.
 */
- (void) setupEditor
{
  [[NSNotificationCenter defaultCenter] removeObserver: self
                                                  name: nil
                                                object: codeText];
  
  [[NSNotificationCenter defaultCenter] addObserver: self
                                           selector: @selector(textDidChange:)
                                               name: NSTextDidChangeNotification
                                             object: codeText];
  [[NSNotificationCenter defaultCenter] addObserver: self
                                           selector: @selector(textDidEndEditing:)
                                               name: NSTextDidEndEditingNotification
                                             object: codeText];
  [WBPluginEditorBase setupCodeEditor: codeText backend: mBackEnd->get_sql_editor() withStatus: YES];
  [codeText setStatusText: @""/*"No errors found"*/];
  
  // Connect the parser with our callback to get notifications about syntax errors.
  mBackEnd->get_sql_editor()->sql_parser_err_cb(sigc::bind(sigc::ptr_fun(process_syntax_error), self));
  mBackEnd->get_sql_editor()->sql_check_finish_signal.connect(sigc::bind(sigc::ptr_fun(process_syntax_done), self));
}

//--------------------------------------------------------------------------------------------------

/**
 * Called when text was added or removed in the editor.
 */
- (void) textDidChange: (NSNotification*) aNotification
{
  if ([aNotification object] == commentText)
  {
    // Set comment for the view.
    mBackEnd->set_comment([[commentText string] UTF8String]);
  }
  else
  {
    // Stop the timer in case it is about to trigger.
    [mSyntaxCheckTimer invalidate];
    
    // Set up a new timer.
    mSyntaxCheckTimer= [NSTimer scheduledTimerWithTimeInterval: 0.5
                                                        target: self
                                                      selector: @selector(checkSyntax:)
                                                      userInfo: nil
                                                       repeats: NO];
  }
};

//--------------------------------------------------------------------------------------------------

/**
 * Remove all markers we set for previous syntax errors.
 */
- (void) resetSyntaxErrors
{
  [codeText setStatusText: @""/*"No errors found"*/];
  int length= codeText.backend->WndProc(SCI_GETLENGTH, 0, 0);
  
  [codeText setGeneralProperty: SCI_SETINDICATORCURRENT parameter: 0 value: 0];
  [codeText setGeneralProperty: SCI_INDICATORCLEARRANGE parameter: 0 value: length];
  
  [codeText setGeneralProperty: SCI_MARKERDELETEALL parameter: -1 value: 0];
}

//--------------------------------------------------------------------------------------------------

/**
 * Triggered once a change in the editor content happend and there was a pause of at least 500 ms.
 */
- (void) checkSyntax:(NSTimer*) timer
{  
  mSyntaxCheckTimer = nil;
  
  // Before syntax checking store all changes.
  [self updateBackend];
  
  NSString *text = [codeText string];
  if (text)
  {
    mErrorCount = 0;
    mBackEnd->get_sql_editor()->sql([text UTF8String]);
    mBackEnd->get_sql_editor()->check_sql(true);
    
    if (mErrorCount == 0)
      [self resetSyntaxErrors];
  }
  else
    [self resetSyntaxErrors];
}

- (bec::BaseEditor*)editorBE
{
  return mBackEnd;
}

@end
