﻿using System;
using System.Diagnostics;
using System.Runtime.Remoting;
using System.Runtime.Remoting.Channels;
using System.Runtime.Remoting.Channels.Ipc;
using System.Threading;

namespace MySQL.Workbench
{
	/// <summary>
	/// Application Instance Manager
	/// </summary>
	public static class ApplicationInstanceManager
	{
		/// <summary>
		/// Creates the single instance.
		/// </summary>
		/// <param name="name">The name.</param>
		/// <param name="callback">The callback.</param>
		/// <returns></returns>
		public static bool CreateSingleInstance(string name, String[] args, 
      EventHandler<InstanceCallbackEventArgs> callback)
		{
			EventWaitHandle eventWaitHandle = null;
			string eventName = String.Format("{0}.{1}.{2}", Environment.MachineName, Environment.UserName, name);

			InstanceProxy.IsFirstInstance = false;
			InstanceProxy.CommandLineArgs = args;

			try
			{
				// Try opening existing wait handle.
				eventWaitHandle = EventWaitHandle.OpenExisting(eventName);
			}
			catch
			{
				// Got exception => handle wasn't created yet.
				InstanceProxy.IsFirstInstance = true;
			}

			if (InstanceProxy.IsFirstInstance)
			{
				// Since this is the first instance we need to set up our communication infrastructure.
				eventWaitHandle = new EventWaitHandle(false, EventResetMode.AutoReset, eventName);
				ThreadPool.RegisterWaitForSingleObject(eventWaitHandle, WaitOrTimerCallback, callback,
          Timeout.Infinite, false);
				eventWaitHandle.Close();

				// Register shared type (used to pass data between processes).
				RegisterRemoteType(name);
			}
			else
			{
				// We are second in a row, so pass application arguments to the shared object and quit.
				UpdateRemoteObject(name);

				if (eventWaitHandle != null)
          eventWaitHandle.Set();

				Environment.Exit(0);
			}

			return InstanceProxy.IsFirstInstance;
		}

		/// <summary>
		/// Sends the stored command line arguments over to the first instance.
		/// </summary>
    /// <param name="uri">The name used to identify the application.</param>
		private static void UpdateRemoteObject(string uri)
		{
			// Open an inter-process-communication channel to the target application.
			var clientChannel = new IpcClientChannel();
			ChannelServices.RegisterChannel(clientChannel, true);

			// Get shared object from other process.
			var proxy = Activator.GetObject(typeof(InstanceProxy),
        String.Format("ipc://{0}.{1}.{2}/{2}", Environment.MachineName, Environment.UserName, uri)) as InstanceProxy;

			// If we got a proxy object then pass the current command line arguments on.
			if (proxy != null)
				proxy.SetCommandLineArgs(InstanceProxy.IsFirstInstance, InstanceProxy.CommandLineArgs);

			// Finally clean up. We are done.
			ChannelServices.UnregisterChannel(clientChannel);
		}

		/// <summary>
		/// Registers our instance proxy so we can use it in IPC.
		/// </summary>
		/// <param name="uri">The name used to identify the application.</param>
		private static void RegisterRemoteType(string uri)
		{
			// Create and register the IPC channel for communication.
      var serverChannel = new IpcServerChannel(
        String.Format("{0}.{1}.{2}", Environment.MachineName, Environment.UserName, uri));
			ChannelServices.RegisterChannel(serverChannel, true);

			// Register the proxy type...
			RemotingConfiguration.RegisterWellKnownServiceType(
				typeof(InstanceProxy), uri, WellKnownObjectMode.Singleton);

			// ... and take care that things are cleaned up when the process goes down.
			Process process = Process.GetCurrentProcess();
			process.Exited += delegate { ChannelServices.UnregisterChannel(serverChannel); };
		}

		/// <summary>
		/// Callback triggered if the wait handle is set. This means a new process was started and
    /// we have to pass on its command line parameters to the first instance.
		/// </summary>
		/// <param name="state">The application callback.</param>
		/// <param name="timedOut">Can never be true as we have an infinite timeout.</param>
		private static void WaitOrTimerCallback(object state, bool timedOut)
		{
			var callback = state as EventHandler<InstanceCallbackEventArgs>;
			if (callback == null)
        return;

			// Invoke the first instance's application callback so it can do what it needs to do
      // with the given parameters.
			callback(state,
        new InstanceCallbackEventArgs(InstanceProxy.IsFirstInstance, InstanceProxy.CommandLineArgs)
      );
		}
	}
}