/* 
 * Copyright (c) 2007, 2012, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;

using MySQL.Controls;
using MySQL.Forms;
using MySQL.Grt;
using MySQL.Grt.Db;

namespace MySQL.GUI.Workbench.Plugins
{
  /// <summary>
  /// Generic GRT Object Editor
  /// </summary>
  public partial class ObjectEditorPlugin : DockablePlugin
  {
    #region Member Variables

    /// <summary>
    /// Specifies if the controls on the form are currently getting initialized
    /// </summary>
    private bool initializingControls = false;

    private int refreshBlocked = 0;

    private BaseEditor editorBE;

    private SplitContainer mainSplitContainer = null;
    private bool isEdictingLiveObjectCancelled = false;
    //Button closeLiveEditorButton;
    Button revertLiveObjectButton;
    Button applyLiveObjectButton;
    public ObjectEditorView ContainerForm;

    #endregion

    #region SqlEditorHelper

    protected class SqlEditorHelper
    {
      private BaseEditor editorBE;
      private DBObjectEditorBE dbObjEditorBE;
      public Control parent;
      public MySQL.Grt.Db.Sql.SqlEditor sqlEditor;
      public delegate void ProcessParserLogDelegate(List<String> messages);
      private ProcessParserLogDelegate processParserLog;
      public delegate void SetSqlDelegate(string sql, bool sync);
      private SetSqlDelegate setSql;

      class Token
      {
        public int line;
        public int errTokLinePos;
        public int errTokLen;
        public string msg;
      }
      List<Token> errTokens;

      public SqlEditorHelper(GrtManager grtManager, Control parent, ProcessParserLogDelegate processParserLog)
      {
        errTokens = new List<Token>();

        sqlEditor = new MySQL.Grt.Db.Sql.SqlEditor(grtManager);
        sqlEditor.BackgroundAction = Parse;

        this.parent = parent;
        parent.Controls.Add(sqlEditor);
        sqlEditor.Dock = DockStyle.Fill;
        sqlEditor.Scrolling.ScrollBars = ScrollBars.Both;
        sqlEditor.Scrolling.HorizontalWidth = 20000;

        this.processParserLog = processParserLog;
      }

      public void SetEditorBackend(BaseEditor editorBE, SetSqlDelegate setSql)
      {
        if (this.editorBE != editorBE)
        {
          if (this.editorBE != null)
          {
            this.editorBE.Dispose();
            this.editorBE = null;
          }
          this.editorBE = editorBE;
          dbObjEditorBE = editorBE as DBObjectEditorBE;
          if (sqlEditor.BE != null)
          {
            MySQL.Grt.Db.Sql.SqlEditorWrapper sqlEditorBE = sqlEditor.BE;
            sqlEditor.BE = null;
            sqlEditorBE.Dispose();
          }
          sqlEditor.BE = editorBE.get_sql_editor();
          if (null != dbObjEditorBE)
          {
            //!dbObjEditorBE.set_sql_parser_log_cb(new DBObjectEditorBE.VoidStringListDelegate(processParserLog));
            dbObjEditorBE.set_sql_parser_log_cb(ProcessParserLog);
            dbObjEditorBE.set_sql_parser_err_cb(ProcessSyntaxError);
          }
        }

        this.setSql = setSql;
      }

      public void UnregisterCallbacks()
      {
        if (null != dbObjEditorBE)
        {
          dbObjEditorBE.set_sql_parser_log_cb(null);
          dbObjEditorBE.set_sql_parser_err_cb(null);
        }
      }

      public BaseEditor EditorBE
      {
        get { return editorBE; }
      }

      public void Parse(bool sync)
      {
        errTokens.Clear();
        if (null != sqlEditor)
        {
          sqlEditor.ResetSqlCheckState();
          if (null != setSql)
            setSql(sqlEditor.Text, sync);
        }
      }

      public int ProcessSyntaxError(int line, int errTokLinePos, int errTokLen, String msg)
      {
        Token tok = new Token();
        tok.line = line;
        tok.errTokLinePos = errTokLinePos;
        tok.errTokLen = errTokLen;
        tok.msg = msg;

        errTokens.Add(tok);

        return 0;
      }

      private void ProcessParserLog(List<String> messages)
      {
        if (null != sqlEditor)
          foreach (Token tok in errTokens)
            sqlEditor.ProcessSqlError(tok.line, tok.errTokLinePos, tok.errTokLen, tok.msg);

        if (null != processParserLog)
          processParserLog(messages);
      }
    };

    protected SqlEditorHelper sqlEd;

    #endregion

    #region Constructors

    /// <summary>
    /// Standard constructor
    /// </summary>
    protected ObjectEditorPlugin() : base()
    {
    }

    /// <summary>
    /// Overloaded constructor taking the GRT Manager and GRT object to edit
    /// </summary>
    /// <param name="GrtManager">The GRT Manager</param>
    /// <param name="GrtObject">The object to edit</param>
    public ObjectEditorPlugin(GrtManager GrtManager, GrtValue GrtList)
      : base(GrtManager, GrtList)
    {
      CreateHandle();
    }

    public ObjectEditorPlugin(BaseEditor editor)
      : base(editor.get_grt_manager(), editor.get_object())
    {
      CreateHandle();
      editorBE = editor;
    }

    #endregion

    #region Properties

    protected bool InitializingControls
    {
      get { return initializingControls; }
      set { initializingControls = value; }
    }

    public bool InsideInitializationOrRefresh()
    {
      return InitializingControls || insideRefreshFormData;
    }

    public new Control ActiveControl
    {
      get
      {
        if (null != ContainerForm)
          return ContainerForm.ActiveControl;
        else
          return ActiveControl;
      }
      set
      {
        if (null != ContainerForm)
          ContainerForm.ActiveControl = value;
        else
          ActiveControl = value;
      }
    }

    protected BaseEditor Backend
    {
      get { return editorBE; }
      set
      {
        if (editorBE != null)
          editorBE.Dispose();
        editorBE = value;
      }
    }

    #endregion

    #region IWorkbenchDocument Interface

    public override UIForm BackendForm
    {
      get { return editorBE; }
    }

    public override void RefreshGUI(RefreshType refresh, String str, IntPtr ptr)
    {
    }

    public override void PerformCommand(String command)
    {
    }

    public override DockablePlugin FindPluginOfType(Type type)
    {
      if (GetType() == type)
        return this;
      return null;
    }

    public override bool ClosePluginOfType(Type type)
    {
      if (GetType() == type)
      {
        Close();
        return true;
      }
      return false;
    }

    #endregion

    #region Form Implementation

    public bool IsEditingLiveObject
    {
      get { return (null == editorBE) ? false : editorBE.is_editing_live_object(); }
    }

    public void ApplyChangesToLiveObject()
    {
      editorBE.apply_changes_to_live_object();
    }

    private void AdjustEditModeControls(Control mainTabControl)
    {
      if (null == mainTabControl)
        return;

      bool isEditingLiveObject = IsEditingLiveObject;
      bool isEditingLiveObjectUI = ((null != mainSplitContainer) && (mainTabControl.Parent == mainSplitContainer.Panel1));
      if (isEditingLiveObject == isEditingLiveObjectUI)
        return;

      Control mainContainer = (isEditingLiveObjectUI) ? mainSplitContainer.Parent : mainTabControl.Parent;
      mainContainer.SuspendLayout();

      try
      {
        if (isEditingLiveObject)
        {
          if (null == mainSplitContainer)
          {
            mainSplitContainer = new SplitContainer();
            mainSplitContainer.Dock = DockStyle.Fill;
            mainSplitContainer.Orientation = Orientation.Horizontal;
            mainSplitContainer.SplitterWidth = 2;
            mainSplitContainer.FixedPanel = FixedPanel.Panel2;

            Panel liveObjectControlsPanel = new Panel();
            liveObjectControlsPanel.Parent = mainSplitContainer.Panel2;
            liveObjectControlsPanel.Dock = DockStyle.Fill;
            /*
            {
              closeLiveEditorButton = new Button();
              closeLiveEditorButton.UseVisualStyleBackColor = true;
              closeLiveEditorButton.Parent = liveObjectControlsPanel;
              closeLiveEditorButton.Text = "Close";
              closeLiveEditorButton.Location = new Point(
                mainSplitContainer.Panel2.ClientSize.Width - closeLiveEditorButton.Width - closeLiveEditorButton.Margin.Right,
                closeLiveEditorButton.Margin.Top);
              closeLiveEditorButton.Click += new EventHandler(closeLiveObjectEditorButton_Click);
            }
             * */
            {
              revertLiveObjectButton = new Button();
              revertLiveObjectButton.UseVisualStyleBackColor = true;
              revertLiveObjectButton.Parent = liveObjectControlsPanel;
              revertLiveObjectButton.FlatStyle = FlatStyle.System;
              revertLiveObjectButton.Text = "Revert";
              revertLiveObjectButton.Location = new Point(
                mainSplitContainer.Panel2.ClientSize.Width - revertLiveObjectButton.Width - revertLiveObjectButton.Margin.Right,
                revertLiveObjectButton.Margin.Top);
              revertLiveObjectButton.Click += new EventHandler(revertChangesToLiveObjectButton_Click);
            }
            {
              applyLiveObjectButton = new Button();
              applyLiveObjectButton.UseVisualStyleBackColor = true;
              applyLiveObjectButton.Parent = liveObjectControlsPanel;
              applyLiveObjectButton.FlatStyle = FlatStyle.System;
              applyLiveObjectButton.Text = "Apply";
              applyLiveObjectButton.Location = new Point(
                revertLiveObjectButton.Location.X - revertLiveObjectButton.Margin.Left - applyLiveObjectButton.Width - applyLiveObjectButton.Margin.Right,
                revertLiveObjectButton.Location.Y);
              applyLiveObjectButton.Click += new EventHandler(applyChangesToLiveObjectButton_Click);
            }

            mainSplitContainer.Panel2MinSize = revertLiveObjectButton.Height + revertLiveObjectButton.Margin.Vertical;
            mainSplitContainer.SplitterDistance = mainContainer.ClientSize.Height - mainSplitContainer.Panel2MinSize - mainSplitContainer.SplitterWidth;

            //closeLiveEditorButton.Anchor =
            revertLiveObjectButton.Anchor =   AnchorStyles.Right | AnchorStyles.Bottom;
            applyLiveObjectButton.Anchor = AnchorStyles.Right | AnchorStyles.Bottom;
            /*applyingChangesProgressBar.Anchor =*/
          }

          mainSplitContainer.Parent = mainTabControl.Parent;
          mainTabControl.Parent = mainSplitContainer.Panel1;

       }
        else
        {
          mainSplitContainer.Parent = null;
          mainTabControl.Parent = mainContainer;
        }
      }
      finally
      {
        mainContainer.ResumeLayout();
      }
    }

    private void applyChangesToLiveObjectButton_Click(object sender, EventArgs e)
    {
      if (null != sqlEd)
        sqlEd.sqlEditor.RunBackgroundAction(true);

      ApplyChangesToLiveObject();
    }

    private void revertChangesToLiveObjectButton_Click(object sender, EventArgs e)
    {
      if (null != sqlEd)
        sqlEd.sqlEditor.IsDirty = false;

      editorBE.refresh_live_object();
    }

    private void closeLiveObjectEditorButton_Click(object sender, EventArgs e)
    {
      if (sqlEd != null)
        sqlEd.sqlEditor.RunBackgroundAction(true);
      Close(false);
    }

    public override string TabText
    {
      set
      {
        base.TabText = value;
        if (null != ContainerForm)
          ContainerForm.TabText = value;
      }
    }

    public bool ShouldCloseOnObjectDelete(String oid)
    {
      return editorBE.should_close_on_delete_of(oid);
    }

    public virtual bool ChangeGrtList(GrtManager GrtManager, GrtValue GrtList)
    {
      return false;
    }

    public virtual void Close(bool force)
    {
      if (null != ContainerForm)
        ContainerForm.Close(force);
      else
        Close();
    }

    protected override void OnFormClosing(FormClosingEventArgs e)
    {
      base.OnFormClosing(e);

      if (e.Cancel)
        return;

      if (null != sqlEd)
        sqlEd.sqlEditor.RunBackgroundAction(true);

      if (IsEditingLiveObject && !isEdictingLiveObjectCancelled)
      {
        bool can_close = editorBE.can_close();
        if (!can_close)
        {
          e.Cancel = true;
          return;
        }
      }

      if (editorBE != null)
        editorBE.disable_auto_refresh();
    }

    protected void BlockUIRefreshes()
    {
      refreshBlocked++;
    }

    protected void UnblockUIRefreshes()
    {
      refreshBlocked--;
      if (refreshBlocked == 0)
        CallRefreshFormData();
      else if (refreshBlocked < 0)
        throw new Exception("too many UnblockUIRefreshes");
    }

    protected bool insideRefreshFormData= false;
    protected delegate void RefreshFormDataCallback();
    protected virtual void RefreshFormData() {}

    protected bool insideRefreshPartialFormData= false;
    protected delegate void RefreshPartialFormDataCallback(int where);
    protected virtual void RefreshPartialFormData(int where) { }

    protected virtual void CallRefreshFormData() 
    {
      if (insideRefreshFormData || refreshBlocked > 0)
        return;

      insideRefreshFormData = true;
      try
      {
        //This is not right to hack inside derived class, pick part of it containg tabs 
        //and adjust something there leaving rest of controls unadjusted.
        //There is no guariantee that derived editor will have anything named "mainTabControl" at all
        Control[] controls = Controls.Find("mainTabControl", true);
        if (controls.Length > 0)
        {
          FlatTabControl mainTabControl = controls[0] as FlatTabControl;
          AdjustEditModeControls(mainTabControl);
        }

        RefreshFormData();
      }
      finally 
      {
        insideRefreshFormData = false;
      }
    }

    protected virtual void CallRefreshPartialFormData(int where) 
    {
      if (insideRefreshPartialFormData || refreshBlocked > 0)
        return;

      insideRefreshPartialFormData = true;
      try
      {
        RefreshPartialFormData(where);
      }
      finally 
      {
        insideRefreshPartialFormData = false;
      }
    }

    // this method might be called from non-UI thread
    protected void RefreshFormDataInvoke() 
    {
      if (InvokeRequired)
        this.Invoke(new RefreshFormDataCallback(CallRefreshFormData));
      else
        CallRefreshFormData();
    }

    protected void RefreshPartialFormDataInvoke(int arg)
    {
      if (InvokeRequired)
        this.Invoke(new RefreshPartialFormDataCallback(CallRefreshPartialFormData), new Object[] { arg });
      else
        CallRefreshPartialFormData(arg);
    }
    #endregion
  }
}