/* 
 * Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */


#import "MGridView.h"
#include "recordset_be.h"

#import "NSMenu_extras.h"
#import "MCPPUtilities.h"

extern NSString* NSMenuActionNotification;


@implementation MGridView

- (void)setRecordset:(Recordset*)rset
{
  mRecordset = rset;
}


- (BOOL)becomeFirstResponder
{
  [self setNeedsDisplay: YES];
  return [super becomeFirstResponder];
}


- (BOOL)resignFirstResponder
{
  [self setNeedsDisplay: YES];
  return [super resignFirstResponder];
}


- (void) mouseDown: (NSEvent*) event;
{
  NSPoint localPoint = [self convertPoint: [event locationInWindow]
                                 fromView: nil];
  NSInteger column= [self columnAtPoint: localPoint];
  
  mOSelectedRowIndex = mSelectedRowIndex;
  mOSelectedColumnIndex = mSelectedColumnIndex;
  
  if (column < 0)
    mSelectedColumnIndex = [self numberOfColumns] - 1;
  else
    mSelectedColumnIndex= column;
  mSelectedRowIndex = [self rowAtPoint: localPoint];
  
  if (selectionChangedActionTarget)
    [selectionChangedActionTarget performSelector:selectionChangedAction];

  [self display];
}


- (void)rightMouseDown: (NSEvent*)event
{
  [self mouseDown: event];
  [super rightMouseDown: event];
}


- (void) mouseUp: (NSEvent*) event;
{
  NSPoint localPoint = [self convertPoint: [event locationInWindow]
                                 fromView: nil];
  NSInteger column= [self columnAtPoint: localPoint];
  if (column < 0)
    mSelectedColumnIndex = [self numberOfColumns] - 1;
  else
    mSelectedColumnIndex= column;
  mSelectedRowIndex = [self rowAtPoint: localPoint];
  
  if (selectionChangedActionTarget)
    [selectionChangedActionTarget performSelector:selectionChangedAction];
  
  [self display];
  
  if (column >= 0 && mSelectedRowIndex == mOSelectedRowIndex && mSelectedColumnIndex == mOSelectedColumnIndex)
  {
    if (mSelectedRowIndex >= 0 && mSelectedColumnIndex >= 0 &&
        [[self delegate] respondsToSelector: @selector(tableView:shouldEditTableColumn:row:)] &&      
        [[self delegate] tableView: self
             shouldEditTableColumn: [[self tableColumns] objectAtIndex: mSelectedColumnIndex]
                               row: mSelectedRowIndex])
      [self editColumn: mSelectedColumnIndex
                   row: mSelectedRowIndex
             withEvent: event
                select: YES];
  }
}


- (void)selectCellAtRow:(int)row column:(int)column
{
  mSelectedRowIndex = row;
  mSelectedColumnIndex = column;
  [self setNeedsDisplay:YES];
}


- (void) keyDown: (NSEvent*) event;
{
  unsigned short key = [event keyCode];
 
  switch (key)
  {
    case 36: // Return
    case 76: // Enter
      if (mSelectedRowIndex >= 0 && mSelectedColumnIndex > 0 &&
        [[self delegate] tableView: self
             shouldEditTableColumn: [[self tableColumns] objectAtIndex: mSelectedColumnIndex]
                               row: mSelectedRowIndex])
      {
        // Start edit.
        [self editColumn: mSelectedColumnIndex
                     row: mSelectedRowIndex
               withEvent: event
                  select: YES];
        return;
      }
      break;
    case 48: // Tab
    {
      NSUInteger modifiers = [event modifierFlags];
      bool shift = (modifiers & NSShiftKeyMask) != 0;

      if (shift)
      {
        // Move backward.
        mSelectedColumnIndex--;
        if (mSelectedColumnIndex < 1) // Column 0 is only the current-row-indicator.
        {
          // Continue on the previous line if we reached the beginning of the current line.
          if (mSelectedRowIndex > 0)
          {
            mSelectedRowIndex--;
            mSelectedColumnIndex = [self numberOfColumns] - 1;
          }
          else
            mSelectedColumnIndex++; // Restore previous column index. We cannot move further.
        }
        mSelectedColumnIndex = MIN(mSelectedColumnIndex, [self numberOfColumns] - 1);
      }
      else
      {
        // Move forward.
        mSelectedColumnIndex++;
        if (mSelectedColumnIndex == [self numberOfColumns])
        {
          // Continue on the next line if we reached the end of the current line.
          if (mSelectedRowIndex < [self numberOfRows] - 1)
          {
            mSelectedRowIndex++;
            mSelectedColumnIndex = 1;
          }
          else
            mSelectedColumnIndex--; // Restore previous column index. We cannot move further.
        }
        mSelectedColumnIndex = MIN(mSelectedColumnIndex, [self numberOfColumns] - 1);
      }
      
      if (selectionChangedActionTarget)
        [selectionChangedActionTarget performSelector:selectionChangedAction];
      
      break;
    }
    case 123: // Left
      mSelectedColumnIndex -= 1;
      mSelectedColumnIndex = MAX(mSelectedColumnIndex, 1);
      if (selectionChangedActionTarget)
        [selectionChangedActionTarget performSelector:selectionChangedAction];
      break;
    case 124: // Right
      mSelectedColumnIndex += 1;
      mSelectedColumnIndex = MIN(mSelectedColumnIndex, [self numberOfColumns] - 1);
      if (selectionChangedActionTarget)
        [selectionChangedActionTarget performSelector:selectionChangedAction];
      break;
    case 125: // Down
      mSelectedRowIndex += 1;
      mSelectedRowIndex = MIN(mSelectedRowIndex, [self numberOfRows] - 1);
      if (selectionChangedActionTarget)
        [selectionChangedActionTarget performSelector:selectionChangedAction];
      break;
    case 121: // PgDown
      if (selectionChangedActionTarget)
        [selectionChangedActionTarget performSelector:selectionChangedAction];
      [super keyDown: event]; // let original handler do the page down
      mSelectedRowIndex = [self rowAtPoint: NSMakePoint(0, NSMaxY([self visibleRect])-[self rowHeight]/3)];
      if (mSelectedRowIndex < 0)
        mSelectedRowIndex = [self numberOfRows] - 1;
      break;
    case 126: // Up
      mSelectedRowIndex -= 1;
      mSelectedRowIndex = MAX(mSelectedRowIndex, 0);
      if (selectionChangedActionTarget)
        [selectionChangedActionTarget performSelector:selectionChangedAction];
      break;
    case 116: // PgUp
      if (selectionChangedActionTarget)
        [selectionChangedActionTarget performSelector:selectionChangedAction];
      [super keyDown: event]; // let original handler do the page up
      mSelectedRowIndex = [self rowAtPoint: NSMakePoint(0, NSMinY([self visibleRect])+[self rowHeight]/3)];
      break;
    case 51: // Backspace
      if (mSelectedRowIndex >= 0 && mSelectedColumnIndex > 0 &&
          [[self delegate] tableView: self
               shouldEditTableColumn: [[self tableColumns] objectAtIndex: mSelectedColumnIndex]
                                 row: mSelectedRowIndex])
      {
        [[self delegate] tableView: self
                    setObjectValue: nil
                    forTableColumn: [[self tableColumns] objectAtIndex: mSelectedColumnIndex]
                               row: mSelectedRowIndex];
      }
      break;
    default:
      [super keyDown: event];
      break;
  }
  
  if (key != 121 && key != 116) // except for page keys
  {
    [self scrollRowToVisible: mSelectedRowIndex];
    [self scrollColumnToVisible: mSelectedColumnIndex];
  }
  [self setNeedsDisplay: YES];
}


- (void)cancelOperation:(id)sender
{
  // Abort editing if Escape pressed
  if ([self currentEditor])
  {
    [self abortEditing];
    [[self window] makeFirstResponder: self];
  }
}


- (void)textDidEndEditing:(NSNotification *)notification
{
  // Let the ancestor do its job first (e.g. write the value back if needed).
  [super textDidEndEditing: notification];

  int textMovement = [[[notification userInfo] objectForKey: @"NSTextMovement"] intValue];
  if (textMovement == NSTabTextMovement)
  {
    [[self window] makeFirstResponder: self];
    mSelectedColumnIndex++;
    if (mSelectedColumnIndex == [self numberOfColumns])
    {
      mSelectedColumnIndex = 1;
      mSelectedRowIndex++;
      if (mSelectedRowIndex > [self numberOfRows]-1)
        mSelectedRowIndex = [self numberOfRows]-1;
    }
    if (mSelectedRowIndex >= 0 && mSelectedColumnIndex >= 0 &&
        [[self delegate] tableView: self
             shouldEditTableColumn: [[self tableColumns] objectAtIndex: mSelectedColumnIndex]
                               row: mSelectedRowIndex])
    {
      [self editColumn: mSelectedColumnIndex row: mSelectedRowIndex withEvent: [NSApp currentEvent] select: YES];
    }
  }
  else if (textMovement == NSBacktabTextMovement)
  {
    [[self window] makeFirstResponder: self];
    mSelectedColumnIndex--;
    if (mSelectedColumnIndex == 0)
    {
      mSelectedColumnIndex = [self numberOfColumns]-1;
      mSelectedRowIndex--;
      if (mSelectedRowIndex < 0)
        mSelectedRowIndex = 0;
    }
    if (mSelectedRowIndex >= 0 && mSelectedColumnIndex >= 0 &&
        [[self delegate] tableView: self
             shouldEditTableColumn: [[self tableColumns] objectAtIndex: mSelectedColumnIndex]
                               row: mSelectedRowIndex])
    {
      [self editColumn: mSelectedColumnIndex row: mSelectedRowIndex withEvent: [NSApp currentEvent] select: YES];
    }
  }
}

- (void)deleteBackward:(id)sender
{
  [[self delegate] tableView: self
              setObjectValue: nil
              forTableColumn: nil
                         row: mSelectedRowIndex];
}


- (int) selectedColumnIndex;
{
  return mSelectedColumnIndex;
}



- (int) selectedRowIndex;
{
  return mSelectedRowIndex;
}


- (void)setSelectionChangedAction:(SEL)aSelector
{
  selectionChangedAction= aSelector;
}

- (void)selectionChangedActionTarget:(id)target;
{
  selectionChangedActionTarget = target;
}


static std::vector<int> get_indexes(NSIndexSet *iset, NSInteger clickedRow)
{
  std::vector<int> indexes;
  NSUInteger index = [iset firstIndex];
  while (index != NSNotFound)
  {
    indexes.push_back(index);
    index = [iset indexGreaterThanIndex: index];
  }
  std::reverse(indexes.begin(), indexes.end());
  
  if (indexes.empty() && clickedRow >= 0)
    indexes.push_back(clickedRow);
  
  return indexes;
}


- (NSMenu *)menuForEvent:(NSEvent *)theEvent
{
  if (mRecordset)
  {      
    std::vector<int> rows = get_indexes([self selectedRowIndexes], [self selectedRowIndex]);
    bec::MenuItemList items= mRecordset->get_popup_menu_items(rows, [self selectedColumnIndex]-1);
    
    return [NSMenu menuFromMenuItems:items
                              action: @selector(activateMenuItem:)
                              target:self];
  }
  return [super menuForEvent: theEvent];
}



- (void)activateMenuItem:(id)sender
{
  if (mRecordset)
  {
    std::vector<int> rows = get_indexes([self selectedRowIndexes], [self selectedRowIndex]);
    if (mRecordset->activate_popup_menu_item([[sender representedObject] UTF8String], 
                                             rows, [self selectedColumnIndex]-1))
      [self reloadData];
    else
      [[NSNotificationCenter defaultCenter] postNotificationName: NSMenuActionNotification object: sender];
  }
}


@end


