/* 
 * Copyright (c) 2009, 2012, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#import "WBSQLQueryPanel.h"
#import "GRTIconCache.h"
#import "MTabSwitcher.h"
#import "MVerticalLayoutView.h"
#import "WBTabView.h"
#import "WBSplitView.h"
#import "MCPPUtilities.h"
#import "MContainerView.h"
#import "WBSplitViewUnbrokenizerDelegate.h"
#import "MScintillaView.h"
#import "WBMiniToolbar.h"
#import "GRTListDataSource.h"
#import "WBQueryTab.h"

#import "WBPluginPanel.h"
#import "WBPluginEditorBase.h"
#import "MSpinProgressCell.h"
#import "MSQLEditorController.h"
#include "sqlide/query_side_palette.h"
#include "mforms/toolbar.h"

#include "mforms/../cocoa/MFView.h"
#include "mforms/toolbar.h"

@interface ColorBallTextCell : NSTextFieldCell
{
}
@end

@implementation ColorBallTextCell

- (void)drawWithFrame:(NSRect)cellFrame inView:(NSView *)controlView 
{
  [[self backgroundColor] set];
  
  NSAttributedString *text = [self attributedStringValue];
  NSSize textSize = [text size];
  
  cellFrame.origin.y += (cellFrame.size.height - cellFrame.size.width) / 2;
  cellFrame.size.height = cellFrame.size.width;
  
  cellFrame = NSInsetRect(cellFrame, 2, 2);
  
  [[NSBezierPath bezierPathWithOvalInRect:cellFrame] fill];
  
  cellFrame.origin.y += (cellFrame.size.height - textSize.height) / 2;
  [text drawInRect:cellFrame];
}

@end


@implementation WBSQLQueryPanel

@synthesize backEnd = mBackEnd;

#pragma mark Table View support

- (NSInteger) numberOfRowsInTableView: (NSTableView*) tableView;
{
  if (tableView == mMessagesTable)
  {
    if (mBackEnd)
      return mBackEnd->log()->count();
  }
  else if (tableView == mHistoryTable)
  {
    if (mBackEnd)
      return mBackEnd->history()->entries_model()->count();
  }
  else if (tableView == mHistoryDetailsTable)
  {
    if (mBackEnd && [mHistoryTable selectedRow] >= 0)
      return mBackEnd->history()->details_model()->count();
  }

  return 0;
}

- (void)tableView:(NSTableView *)tableView
willDisplayCell:(id)cell
 forTableColumn:(NSTableColumn *)tableColumn
            row:(NSInteger)row
{
  if (tableView == mMessagesTable)
  {
    if ([[tableColumn identifier] isEqual:@"0"])
    {
      int msgtype;
      mBackEnd->log()->get_field(row, 0, msgtype);
      if (msgtype == DbSqlEditorLog::BusyMsg)
      {
        [[cell progressIndicator] startAnimation: nil];
        [[cell progressIndicator] setHidden: NO];
      }
      else
      {
        [[cell progressIndicator] stopAnimation: nil];
        [[cell progressIndicator] setHidden: YES];
      }
    }
  }
}

- (id) tableView: (NSTableView*) aTableView
objectValueForTableColumn: (NSTableColumn*) aTableColumn
             row: (NSInteger) rowIndex;
{
  if (aTableView == mMessagesTable)
  {
    std::string text;
    
    if ([[aTableColumn identifier] isEqual:@"0"])
    {
      int msgtype;
      mBackEnd->log()->get_field(rowIndex, 0, msgtype);
      if (msgtype != DbSqlEditorLog::BusyMsg)
      {
        bec::IconId icon_id= mBackEnd->log()->get_field_icon(rowIndex, 0, bec::Icon16);
      
        if (icon_id != 0)
          return [[GRTIconCache sharedIconCache] imageForIconId:icon_id];
      }
      return nil;
    }
    else
    {
      mBackEnd->log()->get_field(rowIndex, [[aTableColumn identifier] intValue], text);
      
      return [NSString stringWithCPPString: text];
    }
  }
  else if (aTableView == mHistoryTable)
  {
    std::string text;
    
    mBackEnd->history()->entries_model()->get_field(rowIndex, [[aTableColumn identifier] intValue], text);
    
    return [NSString stringWithCPPString: text];
  }
  else if (aTableView == mHistoryDetailsTable)
  {
    std::string text;
    
    mBackEnd->history()->details_model()->get_field(rowIndex, [[aTableColumn identifier] intValue], text);
    
    return [NSString stringWithCPPString: text];
  }

  return @"foo";
}

- (NSString *)tableView:(NSTableView *)aTableView 
         toolTipForCell:(NSCell *)aCell
                   rect:(NSRectPointer)rect 
            tableColumn:(NSTableColumn *)aTableColumn 
                    row:(NSInteger)row
          mouseLocation:(NSPoint)mouseLocation
{
  int column;
  if (aTableView == mMessagesTable && ((column = [[aTableColumn identifier] intValue]) == 3 || column == 4))
  {
    std::string text;
    mBackEnd->log()->get_field_description(row, column, text);
    return [NSString stringWithCPPString: text];
  }     
  return nil;
}

- (void)tableViewSelectionDidChange:(NSNotification *)aNotification
{
  NSTableView *sender= [aNotification object];
  
  if (sender == mHistoryTable)
  {
    if ([mHistoryTable selectedRow] >= 0)
      mBackEnd->history()->current_entry([mHistoryTable selectedRow]);
    [mHistoryDetailsTable reloadData];
  }
  else if (sender == mHistoryDetailsTable)
  {
  }
  else if (sender == mMessagesTable)
  {
    //std::string text;
    //mBackEnd->log()->get_field([mMessagesTable selectedRow], 3, text);

    //mBackEnd->log()->get_field([mMessagesTable selectedRow], 4, text);
  }
}


- (CGFloat)tableView:(NSTableView *)tableView heightOfRow:(NSInteger)row
{
  if (tableView == mHistoryDetailsTable)
  {
    std::string text;
    mBackEnd->history()->details_model()->get_field(row, 0, text);
    size_t lines= 0;
    const char *ptr= text.c_str();
    
    do
    {
      ptr= strchr(ptr, '\n');
      lines++;
    } while (ptr++);

    return [tableView rowHeight] * lines;
  }
  else
    return [tableView rowHeight];
}


- (NSString*)selectedOutputItemsAsString:(BOOL)queryOnly
{
  std::list<int> sel_indexes;
  NSIndexSet *iset = [mMessagesTable selectedRowIndexes];
  
  if ([iset count] > 0)
    for (int row = [iset firstIndex]; row <= [iset lastIndex]; row = [iset indexGreaterThanIndex: row])
      sel_indexes.push_back(row);
  
  if (sel_indexes.empty())
    return nil;
  
  std::string sql= mBackEnd->get_text_for_actions(sel_indexes, queryOnly);
  
  return [NSString stringWithCPPString: sql];
}


- (NSString*)selectedHistoryItemsAsString
{
  std::list<int> sel_indexes;
  NSIndexSet *iset = [mHistoryDetailsTable selectedRowIndexes];
  
  if ([iset count] > 0)
    for (int row = [iset firstIndex]; row <= [iset lastIndex]; row = [iset indexGreaterThanIndex: row])
      sel_indexes.push_back(row);

  if (sel_indexes.empty() || mBackEnd->history()->current_entry() < 0)
    return nil;
  
  std::string sql= mBackEnd->restore_sql_from_history(mBackEnd->history()->current_entry(), sel_indexes);

  return [NSString stringWithCPPString: sql];
}


- (bec::ListModel*)listModelForTableView:(NSTableView*)table
{
  if (table == mHistoryTable)
    return mBackEnd->history()->entries_model().get();
  return 0;
}


- (void)activateHistoryDetailEntry:(id)sender
{
  std::string text, query;
  NSIndexSet *selection= [mHistoryDetailsTable selectedRowIndexes];
  
  for (NSUInteger row= [selection firstIndex]; row != NSNotFound; row= [selection indexGreaterThanIndex:row])
  {
    mBackEnd->history()->details_model()->get_field(row, 1, text);
    query.append("\n").append(text);
  }

  MSQLEditorController *editor = [self activeEditor];
  
  [editor setString: [[editor string] stringByAppendingString: [NSString stringWithCPPString: query]]];
}


static int processTaskFinish(WBSQLQueryPanel *self)
{
  [self->mMessagesTable reloadData];
  [[self activeQueryTab] updateResultsetTabs];
  [self->mUpperTabSwitcher setBusyTab: nil];
  
  if (self->mBackEnd->exec_sql_error_count() > 0)
  {
    [self->mOutputTabView selectTabViewItemWithIdentifier: @"actions"];
    [self->mOutputSelector selectItemAtIndex: 0];
  }
  else
  {
    // re-select the top editor to force re-selection of its resultset tab
    [self->mUpperTabView selectTabViewItemWithIdentifier: [[self->mUpperTabView selectedTabViewItem] identifier]];
  }
  
  return 0;
}

static int processTaskProgress(float progress, const std::string &message, WBSQLQueryPanel *self)
{
  [self->mMessagesTable reloadData];
  [self->mMessagesTable scrollRowToVisible: [self->mMessagesTable numberOfRows]-1];
  
  return 0;
}


- (void)refreshTable:(NSTableView*)table
{
  [table reloadData];
  [[table delegate] tableViewSelectionDidChange:[NSNotification notificationWithName: NSTableViewSelectionDidChangeNotification
                                                                              object: table]];
  if ([table selectedRow] >= 0)
    [table scrollRowToVisible: [table selectedRow]];
  else if ([table numberOfRows] > 0)
    [table scrollRowToVisible: [table numberOfRows]-1];
}


static int reloadTable(NSTableView *table, WBSQLQueryPanel *self)
{
  if ([NSThread mainThread] == [NSThread currentThread])
  {
    [self refreshTable: table];
  }
  else
  {
    [self performSelectorOnMainThread: @selector(refreshTable:)
                           withObject: table
                        waitUntilDone: NO];
  }
  return 0;
}

static void recordsetListChanged(int editor_index, Recordset::Ref rs, bool added, WBSQLQueryPanel *self)
{
  if (!added)
  {
    if (![NSThread isMainThread])
    {
      NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
      [[self queryTabForBackEndIndex: editor_index]
            performSelectorOnMainThread: @selector(removeRecordsetWithIdentifier:) 
                             withObject: [NSString stringWithFormat:@"rset%li", rs->key()]
                          waitUntilDone: NO];
      [pool release];
    }
    else
      [[self queryTabForBackEndIndex: editor_index]
           removeRecordsetWithIdentifier: [NSString stringWithFormat:@"rset%li", rs->key()]];
  }
  else
    [[self queryTabForBackEndIndex: editor_index]
          performSelectorOnMainThread:@selector(updateResultsetTabs) withObject:nil waitUntilDone:NO];
}

static void addTextToOutput(const std::string &text, bool bring_to_front, WBSQLQueryPanel *self)
{
  [self->mTextOutputLock lock];
  self->mTextOutputBuffer.append(text);
  [self->mTextOutputLock unlock];

  if ([NSThread isMainThread])
  {
    [self flushOutputBuffer];
    if (bring_to_front)
    {
      [self->mOutputSelector selectItemAtIndex: 1];
      [self->mOutputTabView selectTabViewItemWithIdentifier: @"text"];
    }
  }
  else
    [self performSelectorOnMainThread:@selector(flushOutputBuffer) withObject:nil waitUntilDone:NO];
}

#pragma mark User actions



#define QUERY_AREA_EXPANDED_MIN_HEIGHT (100)
#define QUERY_AREA_COLLAPSED_MIN_HEIGHT (0)

#define OUTPUT_AREA_EXPANDED_MIN_HEIGHT (22)
#define OUTPUT_AREA_COLLAPSED_MIN_HEIGHT (22)


- (void)executeQuery:(id)sender currentStatementOnly: (bool) currentStatementOnly
{
  std::string text;
  WBQueryTab *qtab = [self activeQueryTab];
  
  if (qtab)
  {
    if (currentStatementOnly)
      text= [[[qtab editorController] currentSqlStatement] UTF8String];
    else
      text= [[[qtab editorController] stringOrSelection] UTF8String] ? : "";
    if (text.empty())
      return;

    [mUpperTabSwitcher setBusyTab: [mUpperTabView selectedTabViewItem]];
    mBackEnd->exec_sql(text, [[qtab editorController] backEnd], false, currentStatementOnly);  
  }
}


- (void)addEditor:(WBBasePanel*)editor
{
  if ([editor isKindOfClass: [WBPluginPanel class]])
  {
    WBPluginEditorBase *peditor = [(WBPluginPanel*)editor pluginEditor];
    [peditor enableLiveChangeButtons];
    [peditor setCompactMode: NO];
  }
  
  [mEditors addObject: editor];
  NSTabViewItem *item = [[[NSTabViewItem alloc] initWithIdentifier: editor] autorelease];
  [item setView: [editor topView]];
  [item setLabel: [editor title]];
  [mUpperTabView addTabViewItem: item];
  [mUpperTabView selectLastTabViewItem: nil];
  
  if ([editor respondsToSelector: @selector(didShow)])
    [editor performSelector: @selector(didShow)];
}


- (void)addSQLEditorTabWithBackEndIndex:(int)editor_index
{
  WBQueryTab *qtab = [[[WBQueryTab alloc] initWithOwner: self
                                                backEnd: mBackEnd->sql_editor(editor_index)] autorelease];  
  [self addEditor: qtab];

  if (mBackEnd->sql_editor_start_collapsed(editor_index))
    [qtab setQueryCollapsed: YES];
  else
    [qtab activateQueryArea: nil];
}


- (void)closeActiveEditorTab
{
  NSTabViewItem *item = [mUpperTabView selectedTabViewItem];
  if (item)
    [mUpperTabSwitcher closeTabViewItem: item];
}


- (WBQueryTab*)activeQueryTab
{
  boost::shared_ptr<Sql_editor> be(mBackEnd->active_sql_editor());
  if (!be) return nil;
  
  for (id qtab in mEditors)
  {
    if ([qtab isKindOfClass: [WBQueryTab class]] && [[qtab editorController] backEnd] == be)
      return qtab;
  }
  return nil;  
}

- (MSQLEditorController*)activeEditor
{
  boost::shared_ptr<Sql_editor> be(mBackEnd->active_sql_editor());
  if (!be) return nil;
  
  for (id qtab in mEditors)
  {
    if ([qtab isKindOfClass: [WBQueryTab class]] && [[qtab editorController] backEnd] == be)
      return [qtab editorController];
  }
  return nil;
  
//  return [[[mUpperTabView tabViewItemAtIndex: mBackEnd->active_sql_editor_index()] identifier] editorController];
}

- (WBQueryTab*)queryTabForBackEndIndex:(int)index
{
  if (index >= 0)
  {
    Sql_editor::Ref editor(mBackEnd->sql_editor(index));
    for (id tab in mEditors)
    {
      if ([tab isKindOfClass: [WBQueryTab class]] && [[tab editorController] backEnd] == editor)
        return tab;
    }
  }
  return nil;
}

#pragma mark Output

- (IBAction)activateCollectionItem:(id)sender
{
  if (sender == mOutputSelector)
    [mOutputTabView selectTabViewItemAtIndex: [sender indexOfSelectedItem]];
}


- (IBAction)copyOutputEntry:(id)sender
{
  NSString *sql = nil;
  
  if ([[[mOutputTabView selectedTabViewItem] identifier] isEqualTo: @"actions"])
    sql = [self selectedOutputItemsAsString: [sender tag] != 3];
  else if ([[[mOutputTabView selectedTabViewItem] identifier] isEqualTo: @"history"])
    sql = [self selectedHistoryItemsAsString];
  
  if (sql)
  {
    switch ([sender tag])
    {
      case 1:
        [[self activeEditor] setString: [[[self activeEditor] string] stringByAppendingString: sql]];
        break;
        
      case 2:
        [[self activeEditor] setString: sql];
        break;
        
      case 3:
        [[NSPasteboard generalPasteboard] declareTypes: [NSArray arrayWithObject:NSStringPboardType] owner:nil];
        [[NSPasteboard generalPasteboard] setString: sql forType: NSStringPboardType];
        break;
    }
  }
}

- (IBAction)clearOutput:(id)sender
{
  if ([[[mOutputTabView selectedTabViewItem] identifier] isEqualTo: @"actions"])
  {
    mBackEnd->log()->reset();
    [mMessagesTable reloadData];
  }
  else if ([[[mOutputTabView selectedTabViewItem] identifier] isEqualTo: @"text"])
  {
    [mTextOutput setString: @""];
  }
  else
  {
    std::vector<int> sel;
    if ([mHistoryTable selectedRow] >= 0)
    {
      sel.push_back([mHistoryTable selectedRow]);
      mBackEnd->history()->entries_model()->delete_entries(sel);
      [mHistoryTable reloadData];
      if ([mHistoryTable numberOfRows] > 0)
      {
        [mHistoryTable selectRowIndexes: [NSIndexSet indexSetWithIndex: [mHistoryTable numberOfRows]-1]
                   byExtendingSelection: NO];
      }
    }
  }
}


- (void)flushOutputBuffer
{
  [mTextOutputLock lock];

  if (!mTextOutputBuffer.empty())
  {
    NSRange range;
    range = NSMakeRange([[mTextOutput string] length], 0);
    [mTextOutput replaceCharactersInRange: range withString: [NSString stringWithUTF8String: mTextOutputBuffer.c_str()]];
  
    range = NSMakeRange([[mTextOutput string] length], 0);
    [mTextOutput scrollRangeToVisible: range];  
  
    mTextOutputBuffer.clear();
  }

  [mTextOutputLock unlock];
}

#pragma mark Public getters + setters

- (NSView*) topView;
{
  return mView;
}

- (NSString*)title
{
  return [NSString stringWithUTF8String: mBackEnd->caption().c_str()];
}


- (id)identifier
{
  return [NSString stringWithFormat:@"dbquery%p", mBackEnd.get()];
}

- (NSImage*)tabIcon
{
  return [NSImage imageNamed: @"tab.sqlquery.16x16"];
}


- (bec::UIForm*)formBE
{
  return boost::get_pointer(mBackEnd);
}


- (void)setActiveEditorTitle:(NSString*)title
{
  [[mUpperTabView selectedTabViewItem] setLabel: title];
  [mUpperTabSwitcher setNeedsDisplay: YES];
}


static void refreshUIPartial(const int what, WBSQLQueryPanel *panel)
{
  switch (what)
  {
    case SqlEditorForm::RefreshEditor:
      [[panel activeEditor] setString: [NSString stringWithCPPString: panel->mBackEnd->active_sql_editor()->sql()]];
      break;
    case SqlEditorForm::RefreshEditorBackend:
      panel->mBackEnd->active_sql_editor()->sql([[[panel activeEditor] string] UTF8String]);
      break;
    case SqlEditorForm::RefreshEditorTitle:
      [panel setActiveEditorTitle: [NSString stringWithCPPString: panel->mBackEnd->sql_editor_caption()]];
      break;
    case SqlEditorForm::RunCurrentScript:
      [panel executeQuery: panel currentStatementOnly: false];
      break;
    case SqlEditorForm::RunCurrentStatement:
      [panel executeQuery: panel currentStatementOnly: true];
      break;
    case SqlEditorForm::RefreshRecordsetTitle:
      [[panel activeQueryTab] updateActiveRecordsetTitle];
      break;
    case SqlEditorForm::ShowFindPanel:
      [[panel activeQueryTab] showFindPanel: [[panel activeQueryTab] editorView]];
      break;
    case SqlEditorForm::ShowSpecialCharacters:
      [[[panel activeQueryTab] editorView] setShowsSpecialCharacters: YES];
      break;
    case SqlEditorForm::HideSpecialCharacters:
      [[[panel activeQueryTab] editorView] setShowsSpecialCharacters: NO];
      break;
  }
}


static int insertText(const std::string &text, WBSQLQueryPanel *panel)
{
  ScintillaView *editor = [[panel activeEditor] view];
  
  [editor insertText: [NSString stringWithCPPString: text]];

  [[editor window] makeFirstResponder: [editor content]];
  
  return 0;
}

static int editorCreated(int editor_index, WBSQLQueryPanel *panel)
{
  [panel addSQLEditorTabWithBackEndIndex: editor_index];
  return 0;
}

#pragma mark Other Delegates

- (NSRect)splitView:(NSSplitView *)splitView
      effectiveRect:(NSRect)proposedEffectiveRect 
       forDrawnRect:(NSRect)drawnRect
   ofDividerAtIndex:(NSInteger)dividerIndex
{
  // if the divider is too thin, increase effective rect by 2px to make it less impossible to drag
  if ([splitView isVertical])
  {
    if (proposedEffectiveRect.size.width < 2)
    {
      proposedEffectiveRect.origin.x -= 1;
      proposedEffectiveRect.size.width += 2;
    }
  }
  else
  {
    if (proposedEffectiveRect.size.height < 2)
    {
      proposedEffectiveRect.origin.y -= 1;
      proposedEffectiveRect.size.height += 2;
    }
  }
  return proposedEffectiveRect;
}


- (void)splitViewDidResizeSubviews:(NSNotification *)notification
{
  if ([notification object] == mView)
  {
    if (!mBackEnd->grt_manager()->get_app_option_int("DbSqlEditor:SchemaSidebarHidden", 0))
      mBackEnd->grt_manager()->set_app_option("DbSqlEditor:SchemaSidebarWidth", 
                                              grt::IntegerRef(mBackEnd->get_sidebar()->get_width()));
  }
  else if ([notification object] == mSidebarSplit)
  {
    if (!mBackEnd->grt_manager()->get_app_option_int("DbSqlEditor:SidebarHidden", 0))
      mBackEnd->grt_manager()->set_app_option("DbSqlEditor:SidebarWidth", 
                                            grt::IntegerRef(mBackEnd->get_side_palette()->get_width()));
  }  
  else if ([notification object] == mWorkView)
  {
    if (!mBackEnd->grt_manager()->get_app_option_int("DbSqlEditor:OutputAreaHidden", 0))
      mBackEnd->grt_manager()->set_app_option("DbSqlEditor:OutputAreaHeight", 
                                            grt::IntegerRef((int)NSHeight([[mOutputTabView superview] frame])));
  }  
}


- (BOOL)splitView:(NSSplitView *)splitView shouldAdjustSizeOfSubview:(NSView *)subview
{
  if (splitView == mView)
  {
    if (subview != mWorkView) // the sidebar
      return NO;
  }
  else if (splitView == mWorkView)
  {
    if (subview == [mOutputTabView superview])
      return NO;
  }
  else if (splitView == mSidebarSplit)
  {
    if (subview == nsviewForView(mBackEnd->get_side_palette()))
      return NO;
  }
  return YES;
}


- (void)tabViewDraggerClicked: (NSTabView*) tabView 
{
  if (mLastClick > 0 && [NSDate timeIntervalSinceReferenceDate] - mLastClick < 0.3)
  {  
    if (mQueryAreaOpen && mResultsAreaOpen)
    {
      mResultsAreaOpen= NO;
      [mWorkView setPosition: NSHeight([mWorkView frame]) - OUTPUT_AREA_COLLAPSED_MIN_HEIGHT
            ofDividerAtIndex: 0];
    }
    else if (mQueryAreaOpen && !mResultsAreaOpen)
    {
      mResultsAreaOpen= YES;
      mQueryAreaOpen= NO;
      [mWorkView setPosition: 0
            ofDividerAtIndex: 0];
    }
    else
    {
      mResultsAreaOpen= YES;
      mQueryAreaOpen= YES;
      
      [mWorkView setPosition: 200
            ofDividerAtIndex: 0];
    }
    mLastClick= 0;
  }
  else
    mLastClick= [NSDate timeIntervalSinceReferenceDate];
}

- (void)tabView:(NSTabView *)tabView didSelectTabViewItem:(NSTabViewItem *)tabViewItem
{
  if (tabView == mUpperTabView)
  {
    if ([[tabViewItem identifier] isKindOfClass: [WBQueryTab class]])
    {
      MSQLEditorController *editor = [[tabViewItem identifier] editorController];
      Sql_editor::Ref ed([editor backEnd]);
      
      for (int i= 0; i < mBackEnd->sql_editor_count(); i++)
      {
        if (mBackEnd->sql_editor(i) == ed)
        {
          mBackEnd->active_sql_editor_index(i);
          break;
        }
      }
    }
    else
      mBackEnd->active_sql_editor_index(-1);
  }
}

- (void)tabViewDidChangeNumberOfTabViewItems:(NSTabView*)tabView
{
  if (tabView == mUpperTabView)
  {
    if ([tabView numberOfTabViewItems] == 0)
      mBackEnd->new_sql_script_file();
  }
}


- (BOOL)tabView:(NSTabView *)tabView itemHasCloseButton:(NSTabViewItem *)item
{
  if (tabView == mUpperTabView)
    return YES;
  return NO;
}


- (BOOL)tabView:(NSTabView *)tabView willReorderTabViewItem:(NSTabViewItem *)item toIndex:(NSInteger)index
{
  if (tabView == mUpperTabView)
  {
    if ([[item identifier] isKindOfClass: [WBQueryTab class]])
    {
      WBQueryTab *qtab = [item identifier];
      if (!mBackEnd->sql_editor_reorder([[qtab editorController] backEnd], index))
        return NO;
    }
  }
  return YES;
}


- (BOOL)tabView:(NSTabView *)tabView
willCloseTabViewItem:(NSTabViewItem*)tabViewItem
{
  if (tabView == mUpperTabView)
  {
    if ([[tabViewItem identifier] isKindOfClass: [WBQueryTab class]])
    {
      MSQLEditorController *editor = [[tabViewItem identifier] editorController];
      
      Sql_editor::Ref ed = [editor backEnd];
      int idx = mBackEnd->sql_editor_index(ed);
      if (idx < 0)
        return NO;
      
      if (!mBackEnd->sql_editor_will_close(idx))
        return NO;
      
      mBackEnd->remove_sql_editor(idx);
      
      [mEditors removeObject: editor];
    }
    return YES;
  }
  else
  {
    id ident = [tabViewItem identifier];
    if ([ident isEqual: @"history"] ||
        [ident isEqual: @"messages"])
      return NO;    
    return YES;
  }
}


- (NSImage*)tabView:(NSTabView *)tabView iconForItem:(NSTabViewItem *)tabViewItem
{
  if (tabView == mUpperTabView)
  {
    id tab = [tabViewItem identifier];
    if ([tab respondsToSelector: @selector(tabIcon)])
      return [tab tabIcon];
  }
  return nil;
}


- (NSString*)tabView:(NSTabView *)tabView toolTipForItem:(NSTabViewItem *)item
{
  if (tabView == mUpperTabView)
  {
    if ([[item identifier] isKindOfClass: [WBQueryTab class]])
    {
      MSQLEditorController *editor = [[item identifier] editorController];
      
      Sql_editor::Ref ed = [editor backEnd];
      int idx = mBackEnd->sql_editor_index(ed);
      if (idx >= 0)
        return [NSString stringWithCPPString: mBackEnd->sql_editor_path(idx)];
    }    
  }
  return nil;
}


- (void)tabView:(NSTabView *)tabView willDisplayMenu:(NSMenu *)menu forTabViewItem:(NSTabViewItem *)item
{
  if (tabView == mUpperTabView)
  {
    MSQLEditorController *editor = [[item identifier] editorController];
    if (editor)
    {
      Sql_editor::Ref ed([editor backEnd]);
      int i = mBackEnd->sql_editor_index(ed);
      if (i >= 0 && !mBackEnd->sql_editor_path(i).empty())
        [[menu itemWithTag: 60] setEnabled: YES];
      else
        [[menu itemWithTag: 60] setEnabled: NO];
    }
    else
      [[menu itemWithTag: 60] setEnabled: NO];
  }
}


- (IBAction)handleMenuAction:(id)sender
{
  switch ([sender tag])
  {
    case 50: // new tab
      mBackEnd->new_sql_script_file();
      break;
    case 51: // save tab
    {
      NSTabViewItem *item = [mUpperTabSwitcher clickedItem];
      MSQLEditorController *editor = [[item identifier] editorController];
      if (editor)
      {
        Sql_editor::Ref ed = [editor backEnd];
        int i = mBackEnd->sql_editor_index(ed);
        if (i >= 0)
          mBackEnd->save_sql_script_file(mBackEnd->sql_editor_path(i), i);
      }
      break;
    }
    case 60: // copy path to clipboard
    {
      NSTabViewItem *item = [mUpperTabSwitcher clickedItem];
      MSQLEditorController *editor = [[item identifier] editorController];
      if (editor)
      {        
        Sql_editor::Ref ed = [editor backEnd];
        int i = mBackEnd->sql_editor_index(ed);
        if (i >= 0)
        {
          NSPasteboard *pasteBoard= [NSPasteboard generalPasteboard];
          [pasteBoard declareTypes: [NSArray arrayWithObject:NSStringPboardType] owner:nil];
          [pasteBoard setString: [NSString stringWithUTF8String: mBackEnd->sql_editor_path(i).c_str()]
                        forType: NSStringPboardType]; 
        }
      }
      break;
    }
  }
}


#pragma mark Create + destroy

- (id)initWithBE:(const SqlEditorForm::Ref&)be
{
  self= [super init];
  if (self)
  {
    BOOL schemaSidebarHidden;
    BOOL sidebarHidden;
    BOOL outputAreaHidden;
    
    // restore state of toolbar
    {
      mforms::ToolBar *toolbar = be->get_toolbar();      
      toolbar->set_item_checked("wb.toggleSchemataBar", !(schemaSidebarHidden = be->grt_manager()->get_app_option_int("DbSqlEditor:SchemaSidebarHidden", 0))); 
      toolbar->set_item_checked("wb.toggleSideBar", !(sidebarHidden = be->grt_manager()->get_app_option_int("DbSqlEditor:SidebarHidden", 0))); 
      toolbar->set_item_checked("wb.toggleOutputArea", !(outputAreaHidden = be->grt_manager()->get_app_option_int("DbSqlEditor:OutputAreaHidden", 0))); 
    }
    lastSchematabarWidth = be->grt_manager()->get_app_option_int("DbSqlEditor:SchemaSidebarWidth", 220);
    lastSidebarWidth = be->grt_manager()->get_app_option_int("DbSqlEditor:SidebarWidth", 220);
    lastOutputAreaHeight = be->grt_manager()->get_app_option_int("DbSqlEditor:OutputAreaHeight", 135);

    [NSBundle loadNibNamed: @"WBSQLQueryPanel"
                     owner: self];
    
    mBackEnd= be;
    mBackEnd->log()->refresh_ui_cb= boost::bind(reloadTable, mMessagesTable, self);
    mBackEnd->history()->entries_model()->refresh_ui_cb= boost::bind(reloadTable, mHistoryTable, self);
    mBackEnd->history()->details_model()->refresh_ui_cb= boost::bind(reloadTable, mHistoryDetailsTable, self);

    mBackEnd->sql_editor_text_insert_signal.connect(boost::bind(insertText, _1, self));
    
    mBackEnd->set_partial_refresh_ui_slot(boost::bind(refreshUIPartial, _1, self));
    mBackEnd->output_text_slot= boost::bind(addTextToOutput, _1, _2, self);
    
    mBackEnd->exec_sql_task->progress_cb(boost::bind(processTaskProgress, _1, _2, self));
    mBackEnd->exec_sql_task->finish_cb(boost::bind(processTaskFinish, self));
    mBackEnd->recordset_list_changed.connect(boost::bind(recordsetListChanged, _1, _2, _3, self));
    
    mBackEnd->sql_editor_new_ui.connect(boost::bind(editorCreated, _1, self));
    
    mBackEnd->set_frontend_data(self);
    [mUpperTabSwitcher setTabStyle: MEditorTabSwitcher];
    [mUpperTabSwitcher setAllowTabReordering: YES];
    
    [mOutputToolbar setGradient: [[[NSGradient alloc] initWithColorsAndLocations: 
                                   [NSColor colorWithCalibratedWhite:0xd9/255.0 alpha: 1.0], (CGFloat)0.0,
                                   [NSColor colorWithCalibratedWhite:0xe2/255.0 alpha: 1.0], (CGFloat)0.5,
                                   [NSColor colorWithCalibratedWhite:0xef/255.0 alpha: 1.0], (CGFloat)0.87,
                                   [NSColor colorWithCalibratedWhite:0xe6/255.0 alpha: 1.0], (CGFloat)0.91,
                                   [NSColor colorWithCalibratedWhite:0xa9/255.0 alpha: 1.0], (CGFloat)1.0,
                                   nil] autorelease]];
    mTextOutputLock= [[NSLock alloc] init];
    
    NSFont *font = [NSFont fontWithName: @"Andale Mono" 
                                   size: [NSFont smallSystemFontSize]];
    if (!font)
      font = [NSFont fontWithName: @"Monaco" size: [NSFont smallSystemFontSize]];
    if (font)
      [mTextOutput setFont: font];
    
    [mSplitterDelegate setTopExpandedMinHeight: QUERY_AREA_EXPANDED_MIN_HEIGHT];
    [mSplitterDelegate setTopCollapsedMinHeight: QUERY_AREA_COLLAPSED_MIN_HEIGHT];
    [mSplitterDelegate setBottomExpandedMinHeight: OUTPUT_AREA_EXPANDED_MIN_HEIGHT];
    [mSplitterDelegate setBottomCollapsedMinHeight: OUTPUT_AREA_COLLAPSED_MIN_HEIGHT];

    [mWorkView setDividerThickness: 0];
    [mView setDividerThickness: 1];
    
    [mHistoryDetailsTable setTarget: self];
    [mHistoryDetailsTable setDoubleAction: @selector(activateHistoryDetailEntry:)];
    if ([mHistoryTable numberOfRows] > 0)
    {
      [mHistoryTable selectRowIndexes: [NSIndexSet indexSetWithIndex: [mHistoryTable numberOfRows]-1]
                 byExtendingSelection: NO];
    }
    
    // dock the backend provided schema sidebar and restore its width
    {
      mforms::View *sidebar = mBackEnd->get_sidebar();
      mSchemataBarView = nsviewForView(sidebar);
      if (schematabarAtRight)
        [mView addSubview: mSchemataBarView];
      else
        [mView addSubview: mSchemataBarView positioned: NSWindowBelow relativeTo: [[mView subviews] lastObject]];
      [mView adjustSubviews];

      if (schemaSidebarHidden)
      {
        if (schematabarAtRight)
          [mView setPosition: NSWidth([mView frame]) ofDividerAtIndex: 0];
        else
          [mView setPosition: 0 ofDividerAtIndex: 0];        
      }
      else
      {
        if (schematabarAtRight)
          [mView setPosition: NSWidth([mView frame]) - lastSchematabarWidth ofDividerAtIndex: 0];
        else
          [mView setPosition: lastSchematabarWidth ofDividerAtIndex: 0];
        
        // ugly hack to force the splitter position where we want it.. somehow, without this "adjustment"
        // the splitter would make the sidebar sized 10px narrower
        if (mBackEnd->get_sidebar()->get_width() < lastSchematabarWidth)
        {
          if (schematabarAtRight)
            [mView setPosition: NSWidth([mView frame]) - (lastSchematabarWidth - mBackEnd->get_sidebar()->get_width()) ofDividerAtIndex: 0];
          else
            [mView setPosition: lastSchematabarWidth + (lastSchematabarWidth - mBackEnd->get_sidebar()->get_width()) ofDividerAtIndex: 0];        
        }
      }
    }
    
    // dock the other sidebar
    {    
      mforms::View *view = mBackEnd->get_side_palette();
      if (view)
      {
        if (schematabarAtRight)
          [mSidebarSplit addSubview: nsviewForView(view) positioned: NSWindowBelow relativeTo: [[mSidebarSplit subviews] lastObject]];      
        else
          [mSidebarSplit addSubview: nsviewForView(view)];
      }
      [mSidebarSplit adjustSubviews];
      if (sidebarHidden)
      {
        if (schematabarAtRight)
          [mSidebarSplit setPosition: 0 ofDividerAtIndex: 0];        
        else
          [mSidebarSplit setPosition: NSWidth([mSidebarSplit frame]) ofDividerAtIndex: 0];
      }
      else
      {
        if (schematabarAtRight)
          [mSidebarSplit setPosition: lastSidebarWidth ofDividerAtIndex: 0];        
        else
          [mSidebarSplit setPosition: NSWidth([mSidebarSplit frame]) - lastSidebarWidth ofDividerAtIndex: 0];
      }
    }
    
    // restore height of the output area
    if (outputAreaHidden)
      [mWorkView setPosition: NSHeight([mWorkView frame]) ofDividerAtIndex: 0];
    else
      [mWorkView setPosition: NSHeight([mWorkView frame]) - lastOutputAreaHeight ofDividerAtIndex: 0];

    mEditors = [[NSMutableArray alloc] init];
    
    mQueryAreaOpen = YES;
    mResultsAreaOpen = YES;
    
    // realize pre-existing editors
    for (int i = 0; i < mBackEnd->sql_editor_count(); i++)
    {
      [self addSQLEditorTabWithBackEndIndex: [mUpperTabView numberOfTabViewItems]];
    }
    
    NSProgressIndicator *indicator = [[NSProgressIndicator alloc] initWithFrame: NSMakeRect(0, 0, 10, 10)];
    [indicator setControlSize: NSSmallControlSize];
    [indicator setStyle: NSProgressIndicatorSpinningStyle];
    [indicator setIndeterminate: YES];
    [((MSpinProgressCell*)[[mMessagesTable tableColumnWithIdentifier: @"0"] dataCell]) setProgressIndicator: indicator];
    [indicator release];
  }
  return self;
}


- (SqlEditorForm::Ref)backEnd
{
  return mBackEnd;
}

- (void)setRightSidebar:(BOOL)flag
{
  schematabarAtRight = flag;

  id view1 = [[mView subviews] objectAtIndex: 0];
  id view2 = [[mView subviews] objectAtIndex: 1];
  
  if (schematabarAtRight)
  {
    if (view2 != mSchemataBarView)
    {
      [[view1 retain] autorelease];
      [view1 removeFromSuperview];
      [mView addSubview: view1];
    }    
  }
  else
  {
    if (view1 != mSchemataBarView)
    {
      [[view1 retain] autorelease];
      [view1 removeFromSuperview];
      [mView addSubview: view1];
    }
  }
}

- (BOOL)willClose
{
  return mBackEnd->can_close();
}

- (void) dealloc
{ 
  mBackEnd->close();
  
  [mTextOutputLock release];
  
  [mEditors release];
  
  [mSplitterDelegate release];
  [mView release];
  [super dealloc]; 
}

/*possibly deprecated
- (void)searchString:(NSString*)text
{
  [[[self activeEditor] view] findAndHighlightText: text
                                         matchCase: NO
                                         wholeWord: NO
                                          scrollTo: YES
                                              wrap: YES
                                         backwards: NO];
}*/

//--------------------------------------------------------------------------------------------------

/**
 * Executes commands sent by the main form that should be handled here.
 */
- (void) performCommand: (const std::string) command
{
  if (command == "wb.toggleSchemataBar")
  {
    BOOL hidden = !mBackEnd->get_toolbar()->get_item_checked(command);
    mBackEnd->grt_manager()->set_app_option("DbSqlEditor:SchemaSidebarHidden", grt::IntegerRef(hidden));
    if (!hidden)
    {
      if (schematabarAtRight) 
        [mView setPosition: NSWidth([mView frame])-lastSchematabarWidth ofDividerAtIndex: 0];
      else
        [mView setPosition: lastSchematabarWidth ofDividerAtIndex: 0];
    }
    else
    {
      lastSchematabarWidth = NSWidth([mSchemataBarView frame]);
      if (schematabarAtRight)
        [mView setPosition: NSWidth([mView frame]) ofDividerAtIndex: 0];
      else
        [mView setPosition: 0 ofDividerAtIndex: 0];
    }
  }
  else if (command == "wb.toggleSideBar")
  {
    BOOL hidden = !mBackEnd->get_toolbar()->get_item_checked(command);
    mBackEnd->grt_manager()->set_app_option("DbSqlEditor:SidebarHidden", grt::IntegerRef(hidden));
    if (!hidden)
    {
      if (schematabarAtRight)
        [mSidebarSplit setPosition: lastSidebarWidth ofDividerAtIndex: 0];
      else
        [mSidebarSplit setPosition: NSWidth([mSidebarSplit frame])-lastSidebarWidth ofDividerAtIndex: 0];
    }
    else
    {
      lastSidebarWidth = mBackEnd->get_side_palette()->get_width();
      if (schematabarAtRight)
        [mSidebarSplit setPosition: 0 ofDividerAtIndex: 0];
      else
        [mSidebarSplit setPosition: NSWidth([mSidebarSplit frame]) ofDividerAtIndex: 0];
    }
  }
  else if (command == "wb.toggleOutputArea")
  {
    BOOL hidden = !mBackEnd->get_toolbar()->get_item_checked(command);
    mBackEnd->grt_manager()->set_app_option("DbSqlEditor:OutputAreaHidden", grt::IntegerRef(hidden));
    if (!hidden)
      [mWorkView setPosition: NSHeight([mWorkView frame])-lastOutputAreaHeight ofDividerAtIndex: 0];
    else
    {
      lastOutputAreaHeight = NSHeight([[mOutputTabView superview] frame]);
      [mWorkView setPosition: NSHeight([mWorkView frame]) ofDividerAtIndex: 0];
    }
  }
}

@end


