using System;
using System.ComponentModel;
using System.Windows.Forms;

using MySQL.Grt;
using MySQL.Grt.Db;

namespace MySQL.GUI.Workbench.Plugins
{
  public partial class DbMysqlSchemaEditor : ObjectEditorPlugin
  {
    #region Member Variables

    private MySQLSchemaEditorBE schemaEditorBE { get { return Backend as MySQLSchemaEditorBE; } } 
    protected string oldName;
    protected bool settingOptions = false;

    #endregion

    #region Constructors

    protected DbMysqlSchemaEditor()
    {
      InitializeComponent();
    }

    public DbMysqlSchemaEditor(GrtManager GrtManager, GrtValue GrtObject)
      : base(new MySQLSchemaEditorBE(GrtManager, GrtObject))
    {
      InitializeComponent();
      InitializingControls = true;
      try
      {
        CallRefreshFormData();
      }
      finally
      {
        InitializingControls = false;
      }

      schemaEditorBE.add_refresh_ui_handler(RefreshFormDataInvoke);
    }

    #endregion

    #region Form implemenation

    protected override void RefreshFormData()
    {
      if (settingOptions)
        return;

      if (schemaEditorBE.is_editing_live_object() && !schemaEditorBE.is_new_object())
        nameTextBox.Enabled = false;

      optComments.Visible = label7.Visible = !IsEditingLiveObject;

      try
      {
        nameTextBox.InitializingControl = true;
        InitializingControls = true;
        oldName = schemaEditorBE.get_name();
        nameTextBox.Text = oldName;

        System.Collections.Generic.List<String> collations_list = schemaEditorBE.get_charset_collation_list();

        optCollation.Items.Clear();
        optCollation.Items.Add("Server Default");
        optCollation.Items.AddRange(collations_list.ToArray());

        int idx = 0;
        String cscoll = schemaEditorBE.get_schema_option_by_name("CHARACTER SET - COLLATE");
        bool found = false;
        foreach (String next_cscoll in optCollation.Items)
        {
          if (next_cscoll == cscoll)
          {
            found = true;
            optCollation.SelectedIndex = idx;
            break;
          }
          idx++;
        }
        if (!found)
          optCollation.SelectedIndex = 0;

        TabText = nameTextBox.Text;

        optComments.Text = schemaEditorBE.get_comment();

        nameTextBox.DisplayDocActionIcon = !IsEditingLiveObject;
      }
      finally
      {
        nameTextBox.InitializingControl = false;
        InitializingControls = false;
      }
    }

    private void nameTextBox_TextChanged(object sender, EventArgs e)
    {
      if (!InitializingControls && !nameTextBox.Text.Equals(schemaEditorBE.get_name()))
        schemaEditorBE.set_name(nameTextBox.Text);

      TabText = nameTextBox.Text;
    }

    private void optCollation_SelectedIndexChanged(object sender, EventArgs e)
    {
      setSchemaOpt();
    }

    private void optComments_TextChanged(object sender, EventArgs e)
    {
      setSchemaOpt();
    }

    private void setSchemaOpt()
    {
      // When a UI value changes, update the schema options
      if (InitializingControls)
        return;

      settingOptions = true;
      try
      {
        // set charset/collation
        //If there is no "-" in optCollation.Text like in case of "Server default" collation and charset will be reset to ""
        schemaEditorBE.set_schema_option_by_name("CHARACTER SET - COLLATE", optCollation.Text);

        if (!optComments.Text.Equals(schemaEditorBE.get_comment()))
          schemaEditorBE.set_comment(optComments.Text);

      }
      finally
      {
        settingOptions = false;
      }
    }

    private void nameTextBox_DocAction(object sender, EventArgs eventArgs)
    {
      if (IsEditingLiveObject)
        return;

      schemaEditorBE.refactor_catalog_upon_schema_rename(oldName, nameTextBox.Text);

      // reset oldName for next refactoring
      oldName = nameTextBox.Text;
    }

    private void nameTextBox_Validating(object sender, CancelEventArgs e)
    {
      if (IsEditingLiveObject)
      {
        nameTextBox.HideDocActionPictureBox();
        return;
      }

      if (oldName == nameTextBox.Text)
        return;

      DialogResult result;
      result = MessageBox.Show("Rename all schema occurrences?", "Schema name changed",
        MessageBoxButtons.YesNoCancel, MessageBoxIcon.Question);
      if (result == DialogResult.Yes)
      {
        nameTextBox.HideDocActionPictureBox();
        schemaEditorBE.refactor_catalog_upon_schema_rename(oldName, nameTextBox.Text);
        // reset oldName for next refactoring
        oldName = nameTextBox.Text;
      }
      else if (result == DialogResult.Cancel)
        e.Cancel = true;
      else if (result == DialogResult.No)
        nameTextBox.HideDocActionPictureBox();
    }

    private void DbMysqlSchemaEditor_Shown(object sender, EventArgs e)
    {
      if (nameTextBox.Enabled)
        nameTextBox.Focus();
    }

    #endregion

  }
}
