/* 
 * Copyright (c) 2010, 2013, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

/**
 * Contains the implementation of the .NET wrapper for the mforms application class.
 */

#include "stdafx.h"

#include "mforms/mforms.h"
#include "wf_view.h"
#include "wf_box.h"
#include "wf_appview.h"
#include "wf_app.h"
#include "wf_dockingpoint.h"
#include "wf_utilities.h"

#include "base/log.h"
#include "GrtTemplates.h"

DEFAULT_LOG_DOMAIN(DOMAIN_MFORMS_WRAPPER)

using namespace System::Threading;
using namespace System::IO;

using namespace MySQL::Forms;
using namespace MySQL::Grt;

using namespace base;

//--------------------------------------------------------------------------------------------------

AppImpl::AppImpl(MySQL::Forms::Manager^ mgr,
  StringAppCommandAppViewStringDelegate^ app_command,
  DockingPointDelegateWrapper^ docking_delegate)
{
  Logger::log(Logger::LogDebug, DOMAIN_MFORMS_WRAPPER, "Creating application wrapper\n");

  // Store a reference to this class in the backend.
  _cb_handle = GCHandle::Alloc(this);
  IntPtr pointer= GCHandle::ToIntPtr(_cb_handle);

  _docking_delegate = docking_delegate;
  mforms::App::instantiate(docking_delegate->get_unmanaged_delegate(), false);

  mforms::App::get()->set_data(pointer.ToPointer());

  // This delegate is served by the mforms app impl structure, hence we don't need to create
  // an additional hook like we do for the other delegates.
  app_command_delegate= app_command;
}

//--------------------------------------------------------------------------------------------------

AppImpl::~AppImpl()
{
//  delete _dock_delegate;
  _cb_handle.Free();
  Logger::log(Logger::LogDebug, DOMAIN_MFORMS_WRAPPER, "Destroyed backend application wrapper\n");
}

//--------------------------------------------------------------------------------------------------

/**
 * Attempts to find the full path to the given image file. If the given path is already an absolute
 * one then it is returned as is. Otherwise the C# frontend is queried to find the file in any of the
 * (image) resource paths and if found a full path is constructed.
 * Note: this is a pure platform function and hence requires no string conversion like the (similar)
 *       get_resource_path function, which is used by the backend.
 */
String^ AppImpl::get_image_path(String ^path)
{
  if (Path::IsPathRooted(path))
    return path;

  AppImpl^ application= (AppImpl^) GetFromFixedId((IntPtr) mforms::App::get()->get_data_ptr());
  return  application->app_command_delegate(AppCommand::AppGetResourcePath, nullptr, path);
}

//--------------------------------------------------------------------------------------------------

Object^ AppImpl::GetFromFixedId(IntPtr ip)
{
  GCHandle gcHandle = GCHandle::FromIntPtr(ip);
  return (Object^) gcHandle.Target;
}

//--------------------------------------------------------------------------------------------------

std::string AppImpl::get_resource_path(mforms::App *app, const std::string &file)
{
  AppImpl^ application= (AppImpl^) GetFromFixedId((IntPtr) app->get_data_ptr());
  return application->app_command_wrapper(AppCommand::AppGetResourcePath, nullptr, file);
}

//--------------------------------------------------------------------------------------------------

void AppImpl::set_status_text(mforms::App *app, const std::string &text)
{
  AppImpl^ application= (AppImpl^) GetFromFixedId((IntPtr) app->get_data_ptr());
  application->app_command_wrapper(AppCommand::AppSetStatusText, nullptr, text);
}

//--------------------------------------------------------------------------------------------------

base::Rect AppImpl::get_application_bounds(mforms::App *app)
{
  AppImpl ^application = (AppImpl^) GetFromFixedId((IntPtr) app->get_data_ptr());
  Windows::Forms::Form ^form = UtilitiesImpl::get_mainform();
  Drawing::Rectangle bounds = form->Bounds;
  return base::Rect(bounds.Left, bounds.Top, bounds.Width, bounds.Height);
}

//--------------------------------------------------------------------------------------------------

static int message_loop_exit_code = MININT; // Can stay unguarded. We only use it from the main thread.

/**
 * Establishes a local event loop which is needed by the python debugger. This is potentially a very
 * dangerous and hard-to-debug thing so use it carefully.
 * Additionally, if you fail to call exit_event_loop the call will never return and block closing
 * the application.
 */
int AppImpl::enter_event_loop(mforms::App *app, float max_wait_time)
{
  message_loop_exit_code = -MININT;
  int remaining_milliseconds;
  if (max_wait_time <= 0)
    remaining_milliseconds = MAXINT;
  else
    remaining_milliseconds = (int) (1000 * max_wait_time);
  while (message_loop_exit_code == -MININT && remaining_milliseconds > 0)
  {
    Application::DoEvents();
    Thread::Sleep(100);
    remaining_milliseconds -= 100;
  }
  if (remaining_milliseconds == 0 || message_loop_exit_code == -MININT)
    return -1;
  return message_loop_exit_code;
}

//--------------------------------------------------------------------------------------------------

void AppImpl::exit_event_loop(mforms::App *app, int ret_code)
{
  message_loop_exit_code = ret_code;
}

//--------------------------------------------------------------------------------------------------

base::Color AppImpl::get_system_color(mforms::SystemColor type)
{
  System::Drawing::Color color;
  switch (type)
  {
  case mforms::SystemColorHighlight:
      color = System::Drawing::Color::FromKnownColor(KnownColor::Highlight);
      break;
  case mforms::SystemColorEditor:
      color = System::Drawing::Color::FromKnownColor(KnownColor::Window);
      break;
  case mforms::SystemColorDisabled:
      color = System::Drawing::Color::FromKnownColor(KnownColor::ButtonFace);
      break;
  case mforms::SystemColorContainer:
    color = System::Drawing::Color::FromKnownColor(KnownColor::Control);
    break;
  default:
#ifdef _DEBUG
    throw new std::runtime_error("mforms::App: Invalid system color enumeration given.");
#else
    log_error("App: Invalid system color enumeration given.\n");
    return base::Color::Black();
#endif
  }

  return Conversions::NativeToColor(color);
}

//--------------------------------------------------------------------------------------------------

std::string AppImpl::app_command_wrapper(AppCommand command, mforms::AppView* view, const std::string &str)
{
  AppViewImpl^ native_view= nullptr;
  if (view != NULL)
    native_view= (AppViewImpl^) ObjectImpl::FromUnmanaged(view);
  String^ result= app_command_delegate(command, native_view, CppStringToNative(str));
  return NativeToCppString(result);
}
