/* 
 * Copyright (c) 2009, 2013, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "base/geometry.h"
#include "base/string_utilities.h"

#import "MySQLRoutineEditor.h"
#import "MCPPUtilities.h"
#import "MVerticalLayoutView.h"
#include "grtdb/db_object_helpers.h" // get_rdbms_for_db_object()

#include "ScintillaView.h"

@implementation DbMysqlRoutineEditor

static void call_refresh(DbMysqlRoutineEditor *self)
{
  [self performSelectorOnMainThread:@selector(refresh) withObject:nil waitUntilDone:YES];
}


- (id)initWithModule:(grt::Module*)module GRTManager:(bec::GRTManager*)grtm arguments:(const grt::BaseListRef&)args
{
  self= [super initWithNibName: @"MySQLRoutineEditor"  bundle: [NSBundle bundleForClass:[self class]]];
  if (self != nil)
  {
    _grtm = grtm;
    // load GUI. Top level view in the nib is the NSTabView that will be docked to the main window
    [self loadView];

    // take the minimum size of the view from the initial size in the nib.
    // Therefore the nib should be designed as small as possible
    // note: the honouring of the min size is not yet implemented
    [self setMinimumSize: [[tabView superview] frame].size];

    [self reinitWithArguments: args];
  }
  return self;
}

- (void)reinitWithArguments: (const grt::BaseListRef&)args
{
  [super reinitWithArguments: args];
  
  delete mBackEnd;
  
    // setup the editor backend with the schema object (args[0])
  mBackEnd= new MySQLRoutineEditorBE(_grtm, db_mysql_RoutineRef::cast_from(args[0]), get_rdbms_for_db_object(args[0]));
  
  // register a callback that will make [self refresh] get called
  // whenever the backend thinks its needed to refresh the UI from the backend data (ie, the
  // edited object was changed from somewhere else in the application)
  mBackEnd->set_refresh_ui_slot(boost::bind(call_refresh, self));
  
  [self setupEditorOnHost: editorHost];
  mBackEnd->load_routine_sql();

  if (mBackEnd->is_editing_live_object())
  {
    int i = [tabView indexOfTabViewItemWithIdentifier: @"comments"];
    if (i > 0)
      [tabView removeTabViewItem: [tabView tabViewItemAtIndex: i]];
    commentText = nil;
  }
  // update the UI
  [self refresh];
}


- (void) dealloc
{
  delete mBackEnd;
  [super dealloc];
}


- (BOOL)matchesIdentifierForClosingEditor:(NSString*)identifier
{
  return mBackEnd->should_close_on_delete_of([identifier UTF8String]);
}

- (void) refresh
{
  if (mBackEnd)
  {
    [nameText setStringValue: [NSString stringWithCPPString: mBackEnd->get_name()]];
    [self updateTitle: [self title]];

    if (!mBackEnd->is_editing_live_object())
      [commentText setString: [NSString stringWithCPPString: mBackEnd->get_comment()]];
  }
}

- (id)identifier
{
  // an identifier for this editor (just take the object id)
  return [NSString stringWithCPPString:mBackEnd->get_object().id()];
}

//--------------------------------------------------------------------------------------------------

- (void) textDidEndEditing: (NSNotification *) aNotification
{
  if ([[aNotification object] isKindOfClass: [ScintillaView class]])
  {    
    mBackEnd->commit_changes();
  }
  else if ([aNotification object] == commentText)
  {
    [[aNotification object] breakUndoCoalescing];
    // Set comment for the view.
    mBackEnd->set_comment([[commentText string] UTF8String]);
  }
}

//--------------------------------------------------------------------------------------------------

- (bec::BaseEditor*)editorBE
{
  return mBackEnd;
}


- (NSImage*)titleIcon
{
  return [NSImage imageNamed: @"tab_icon_db.Routine.png"];
}
@end
