/* 
 * Copyright (c) 2009, 2013, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"

#include "mforms/mforms.h"
#include "wf_view.h"
#include "wf_drawbox.h"

using namespace MySQL;
using namespace MySQL::Forms;

using namespace System::Drawing;

//-------------------------------------------------------------------------------------------------

/**
 * Converts Windows specific mouse button identifiers to plain numbers for the back end.
 */
static int convert_mouse_button(MouseButtons button)
{
  switch (button)
  {
    case MouseButtons::Left:
      return 0;
    case MouseButtons::Right:
      return 1;
    case MouseButtons::Middle:
      return 2;
    default:
      return -1;
  }
}

//----------------- CanvasControl ------------------------------------------------------------------

CanvasControl::CanvasControl()
{
  // Enable custom draw and double buffering for flicker reduction.
  SetStyle(ControlStyles::UserPaint, true);
  SetStyle(ControlStyles::AllPaintingInWmPaint, true);
  SetStyle(ControlStyles::DoubleBuffer, true);
  //SetStyle(ControlStyles::SupportsTransparentBackColor, true);
  SetStyle(ControlStyles::OptimizedDoubleBuffer, true);
  UpdateStyles();
}

//-------------------------------------------------------------------------------------------------

System::Drawing::Size CanvasControl::GetPreferredSize(System::Drawing::Size proposedSize)
{
  int w, h;
  backend->get_layout_size(&w, &h);
  System::Drawing::Size minSize= MinimumSize;
  if (minSize.Width > w)
    w= minSize.Width;
  if (minSize.Height > h)
    h= minSize.Height;
  return System::Drawing::Size(w, h);
}

//-------------------------------------------------------------------------------------------------

void CanvasControl::set_backend(mforms::DrawBox* backend)
{
  this->backend= backend;
}

//-------------------------------------------------------------------------------------------------

void CanvasControl::DoRepaint()
{
  Invalidate();
}

//----------------- DrawBoxImpl -------------------------------------------------------------------

DrawBoxImpl::DrawBoxImpl(mforms::DrawBox *self)
  : ViewImpl(self)
{
}


//-------------------------------------------------------------------------------------------------

DrawBoxImpl::~DrawBoxImpl()
{
}

//-------------------------------------------------------------------------------------------------

bool DrawBoxImpl::create(mforms::DrawBox *self)
{
  DrawBoxImpl ^drawbox= gcnew DrawBoxImpl(self);

  if (drawbox != nullptr)
  {
    CanvasControl ^canvas= ViewImpl::create<CanvasControl>(self, drawbox);
    
    canvas->set_backend(self);
    canvas->Paint += gcnew PaintEventHandler(drawbox, &DrawBoxImpl::drawbox_repaint);
    canvas->MouseDown += gcnew MouseEventHandler(drawbox, &DrawBoxImpl::drawbox_mouse_down);
    canvas->MouseUp += gcnew MouseEventHandler(drawbox, &DrawBoxImpl::drawbox_mouse_up);
    canvas->MouseMove += gcnew MouseEventHandler(drawbox, &DrawBoxImpl::drawbox_mouse_move);
    canvas->MouseClick += gcnew MouseEventHandler(drawbox, &DrawBoxImpl::drawbox_mouse_click);
    canvas->MouseDoubleClick += gcnew MouseEventHandler(drawbox, &DrawBoxImpl::drawbox_mouse_double_click);
    canvas->MouseEnter += gcnew EventHandler(drawbox, &DrawBoxImpl::drawbox_mouse_enter);
    canvas->MouseLeave += gcnew EventHandler(drawbox, &DrawBoxImpl::drawbox_mouse_leave);
    canvas->KeyDown += gcnew KeyEventHandler(drawbox,  &DrawBoxImpl::drawbox_key_down);

    return true;
  }
  return false;
}

//-------------------------------------------------------------------------------------------------

void DrawBoxImpl::drawbox_mouse_down(System::Object ^sender, MouseEventArgs ^e)
{
  CanvasControl^ canvas = (CanvasControl^) sender;
  canvas->Focus();

  if (canvas->Tag != nullptr)
  {
    mforms::DrawBox* drawbox= ViewImpl::get_backend_control<mforms::DrawBox>(canvas);
    drawbox->mouse_down(convert_mouse_button(e->Button), e->X, e->Y);
  }
}

//-------------------------------------------------------------------------------------------------

void DrawBoxImpl::drawbox_mouse_up(System::Object ^sender, MouseEventArgs ^e)
{
  CanvasControl^ canvas= (CanvasControl^)sender;

  if (!canvas->IsDisposed && canvas->Tag != nullptr)
  {
    mforms::DrawBox* drawbox= ViewImpl::get_backend_control<mforms::DrawBox>(canvas);
    drawbox->mouse_up(convert_mouse_button(e->Button), e->X, e->Y);
  }
}

//-------------------------------------------------------------------------------------------------

void DrawBoxImpl::drawbox_mouse_click(System::Object ^sender, MouseEventArgs ^e)
{
  CanvasControl^ canvas= (CanvasControl^)sender;

  if (canvas->Tag != nullptr)
  {
    mforms::DrawBox* drawbox= ViewImpl::get_backend_control<mforms::DrawBox>(canvas);
    drawbox->mouse_click(convert_mouse_button(e->Button), e->X, e->Y);
  }
}

//-------------------------------------------------------------------------------------------------

void DrawBoxImpl::drawbox_mouse_double_click(System::Object ^sender, MouseEventArgs ^e)
{
  CanvasControl^ canvas= (CanvasControl^)sender;

  if (canvas->Tag != nullptr)
  {
    mforms::DrawBox* drawbox= ViewImpl::get_backend_control<mforms::DrawBox>(canvas);
    drawbox->mouse_double_click(convert_mouse_button(e->Button), e->X, e->Y);
  }
}

//-------------------------------------------------------------------------------------------------

void DrawBoxImpl::drawbox_mouse_move(System::Object ^sender, MouseEventArgs ^e)
{
  CanvasControl^ canvas= (CanvasControl^)sender;

  if (canvas->Tag != nullptr)
  {
    mforms::DrawBox* drawbox= ViewImpl::get_backend_control<mforms::DrawBox>(canvas);
    drawbox->mouse_move(e->X, e->Y);
  }
}

//-------------------------------------------------------------------------------------------------

void DrawBoxImpl::drawbox_mouse_leave(System::Object ^sender, EventArgs ^e)
{
  CanvasControl^ canvas= (CanvasControl^)sender;

  if (canvas->Tag != nullptr)
  {
    mforms::DrawBox* drawbox= ViewImpl::get_backend_control<mforms::DrawBox>(canvas);
    if (!drawbox->is_destroying())
      drawbox->mouse_leave();
  }
}

//-------------------------------------------------------------------------------------------------

void DrawBoxImpl::drawbox_mouse_enter(System::Object ^sender, EventArgs ^e)
{
  CanvasControl^ canvas= (CanvasControl^)sender;

  if (canvas->Tag != nullptr)
  {
    mforms::DrawBox* drawbox= ViewImpl::get_backend_control<mforms::DrawBox>(canvas);
    drawbox->mouse_enter();
  }
}

//-------------------------------------------------------------------------------------------------

void DrawBoxImpl::drawbox_repaint(System::Object ^sender, PaintEventArgs ^e)
{
  CanvasControl^ canvas= (CanvasControl^)sender;

  if (canvas->Tag != nullptr)
  {
    mforms::DrawBox* drawbox= ViewImpl::get_backend_control<mforms::DrawBox>(canvas);
    if (drawbox != 0)
    {
      IntPtr hdc= e->Graphics->GetHdc();

      cairo_surface_t* surface= cairo_win32_surface_create(static_cast<HDC>(hdc.ToPointer()));
      cairo_t* cr= cairo_create(surface);
      drawbox->repaint(cr, e->ClipRectangle.X, e->ClipRectangle.Y, e->ClipRectangle.Width, e->ClipRectangle.Height);
      
      cairo_destroy(cr);
      cairo_surface_destroy(surface);

      e->Graphics->ReleaseHdc(hdc);
    }
  }
}

//-------------------------------------------------------------------------------------------------

void DrawBoxImpl::set_needs_repaint(mforms::DrawBox *self)
{
  DrawBoxImpl^ drawbox= (DrawBoxImpl^)ObjectImpl::FromUnmanaged(self);

  CanvasControl^ canvas= drawbox->get_control<CanvasControl>();
  canvas->DoRepaint();
}

//-------------------------------------------------------------------------------------------------

void DrawBoxImpl::drawbox_key_down(System::Object ^sender, KeyEventArgs ^e)
{
  if (e->KeyCode == Keys::Escape)
  {
    CanvasControl^ canvas = (CanvasControl^) sender;
    if (canvas->Tag != nullptr)
    {
      mforms::DrawBox* drawbox= ViewImpl::get_backend_control<mforms::DrawBox>(canvas);
      drawbox->cancel_operation();
    }
  }
}

//-------------------------------------------------------------------------------------------------
