/* $Id: query.c,v 1.8 2005/06/30 07:58:59 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * imp/src/query.c,v 1.7 2004/11/19 19:13:12 flaw
 * if/src/query.c,v 1.8 2004/08/06 07:56:32 flaw
 *//*
 * Postgres Query Interface
 */
#include <postgres.h>
#include <miscadmin.h>
#include <access/heapam.h>
#include <catalog/pg_type.h>
#include <catalog/catversion.h>
#include <executor/executor.h>
#include <executor/execdesc.h>
#include <executor/tstoreReceiver.h>
#include <nodes/params.h>
#include <parser/analyze.h>
#include <tcop/tcopprot.h>
#include <tcop/dest.h>
#include <tcop/pquery.h>
#include <tcop/utility.h>
#include <utils/array.h>
#include <utils/palloc.h>
#include <utils/portal.h>
#include <utils/relcache.h>
#include <utils/tuplestore.h>
#include <pypg/postgres.h>
#include <pypg/environment.h>

#include <Python.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/tupledesc.h>
#include <pypg/heaptuple.h>
#include <pypg/object.h>
#include <pypg/type.h>
#include <pypg/relation.h>
#include <pypg/utils.h>
#include <pypg/error.h>
#include <pypg/conv.h>

#include <pypg/query.h>
#include <pypg/call.h>
#include <pypg/call/portal.h>

static PyMemberDef PyPgQuery_Members[] = {
	{"lastoid", T_UINT, offsetof(struct PyPgQuery, q_lastoid), RO,
		"oid returned from last utility statement"},
	{"arguments", T_OBJECT, offsetof(struct PyPgQuery, q_argtypes), RO,
		"a tuple of Postgres.Type objects specifying the expected types "
		"of the Postgres.Objects given as parameters when called"},
	{NULL}
};

/*
 * Initialize a query to ready it for execution
 */
static int
prepare(PyObj self)
{
	volatile PyObj argtypes = PyPgQuery_FetchArgTypes(self);
	volatile MemoryContext former = NULL, qcxt = NULL;
	volatile int rv = 0;
	List *rpt;
	TransactionId xid;
	SubTransactionId sid;
	CommandId cid;

	xid = PyPgQuery_FetchTransactionId(self);
	sid = PyPgQuery_FetchSubTransactionId(self);
	cid = PyPgQuery_FetchSubTransactionId(self);
	rpt = PyPgQuery_FetchRawParseTree(self);

	PyPgQuery_FixTransactionId(self, GetCurrentTransactionId());
	PyPgQuery_FixSubTransactionId(self, GetCurrentSubTransactionId());
	PyPgQuery_FixCommandId(self, GetCurrentCommandId());

	PG_TRY();
	{
		List * volatile ptl;
		ListCell * volatile pti;
		List * volatile ql = NIL;
		List * volatile pl = NIL;

		qcxt = PyPgQuery_FetchMemoryContext(self);
		if (!(TransactionIdIsCurrentTransactionId(xid) && qcxt != NULL))
		{
			qcxt = AllocSetContextCreate(TopTransactionContext,
				"Python Query Context",
				ALLOCSET_DEFAULT_MINSIZE,
				ALLOCSET_DEFAULT_INITSIZE,
				ALLOCSET_DEFAULT_MAXSIZE
			);
			PyPgQuery_FixMemoryContext(self, qcxt);
		}
		former = PyPgQuery_SwitchContext(self);

		if (rpt == NULL)
		{
			ptl = pg_parse_query(PyPgQuery_FetchSTRING(self));
			PythonMemoryContext(rpt = copyObject(ptl));
			PyPgQuery_FixRawParseTree(self, rpt);
		}
		ptl = rpt;

		foreach(pti, ptl)
		{
			Node * volatile ptree = (Node *) lfirst(pti);
			List * volatile qlist = NULL;
			ListCell * volatile qli = NULL;

			if (argtypes == NULL || PyTuple_GET_SIZE(argtypes) > 0)
			{
				int i, nargs = 0;
				Oid *argtype_oids = NULL;
				qlist = parse_analyze_varparams(ptree, &argtype_oids, &nargs);
				argtypes = PyTuple_New(nargs);
				for (i = 0; i < nargs; ++i)
				{
					PyObj ato;
					Oid at = argtype_oids[i];

					if (at == InvalidOid || at == UNKNOWNOID)
					{
						Py_DECREF(argtypes);
						ereport(ERROR, (
							errcode(ERRCODE_INDETERMINATE_DATATYPE),
							errmsg("could not determine data type of parameter $%d",
								i + 1)
						));
					}
					PythonMemoryContext(ato = PyPgType_FromOid(at));
					PyTuple_SET_ITEM(argtypes, i, ato);
				}
				PyPgQuery_FixArgTypes(self, argtypes);

				qlist = pg_rewrite_queries(qlist);
			}
			else
				qlist = pg_analyze_and_rewrite(ptree, NULL, 0);

			foreach(qli, qlist)
			{
				Query * volatile qtree = (Query *) lfirst(qli);
				Plan * volatile planTree;

				planTree = pg_plan_query(qtree, NULL);
				pl = lappend(pl, planTree);
				ql = lappend(ql, qtree);
			}
		}

		MemoryContextSwitchTo(TopTransactionContext);
		ql = copyObject(ql);
		pl = copyObject(pl);
		PyPgQuery_FixQueryList(self, ql);
		PyPgQuery_FixPlanList(self, pl);
	}
	PG_CATCH();
	{
		rv = -1;
		PyPgQuery_FixQueryList(self, NULL);
		PyPgQuery_FixPlanList(self, NULL);
		PyErr_SetPgError();
	}
	PG_END_TRY();

	if (former != NULL)
		MemoryContextSwitchTo(former);

	if (qcxt)
		MemoryContextReset(qcxt);

	return(rv);
}

/*
 * execute
 * 	A simple function for non-SELECT, non-Utility statements.
 * 	All SELECT statements are portalized.
 */
static PyObj
execute(PyObj self, PyObj args, Query *query, Plan *plan)
{
	long processed;
	Oid lastoid;
	PyObj rob = NULL;
	ParamListInfo pli;
	QueryDesc *qd = NULL;

	pli = ParamListInfo_FromPyTuple(args);
	qd = CreateQueryDesc(query, plan,
		GetLatestSnapshot(), GetLatestSnapshot(),
		None_Receiver, pli, false);
#if PGV_MM >= 80	
	ExecutorStart(qd, false);
#else
	ExecutorStart(qd, true, false);
#endif
	ExecutorRun(qd, ForwardScanDirection, 0);

	lastoid = qd->estate->es_lastoid;
	processed = qd->estate->es_processed;

	ExecutorEnd(qd);

	PyPgQuery_FixLastOid(self, lastoid);
	rob = PyLong_FromUnsignedLong(processed);
	return(rob);
}

static PyObj
utility(Node *utilstmt)
{
	TupleDesc utd;
	PyObj rob = NULL;

	utd = UtilityTupleDescriptor(utilstmt);
	if (utd)
	{
		Tuplestorestate *ts;
		DestReceiver *tsr;

		ts = tuplestore_begin_heap(false, false, work_mem);
		tsr = CreateTuplestoreDestReceiver(ts, CurrentMemoryContext);

		ProcessUtility(utilstmt, NULL, tsr, NULL);
		
		PythonMemoryContext(
			rob = PySequence_FromTupleDescAndTuplestore(utd, ts)
		);

		tsr->rDestroy(tsr);
		tuplestore_end(ts);
	}
	else
	{
		char completion[COMPLETION_TAG_BUFSIZE];
		ProcessUtility(utilstmt, NULL, None_Receiver, completion);
		rob = PyString_FromString(completion);
	}

	CommandCounterIncrement();

	return(rob);
}

static PyObj
query_prepare(PyObj self)
{
	if (prepare(self) < 0)
		return(NULL);

	RETURN_NONE;
}

static PyMethodDef PyPgQuery_Methods[] = {
	{"prepare", (PyCFunction) query_prepare, METH_NOARGS,
	"explicitly prepare the query"},
	{NULL}
};

static void
query_dealloc(PyObj self)
{
	PyObj ob;
	List *l;
	PyPgQuery_FixLastOid(self, InvalidOid);

	ob = PyPgQuery_FetchString(self);
	PyPgQuery_FixString(self, NULL);
	DECREF(ob);

	ob = PyPgQuery_FetchArgTypes(self);
	PyPgQuery_FixArgTypes(self, NULL);
	DECREF(ob);

	l = PyPgQuery_FetchRawParseTree(self);
	PyPgQuery_FixRawParseTree(self, NULL);
	list_free(l);

	if (PyPgQuery_IsPrepared(self) && PyPgQuery_IsCurrent(self))
	{
		l = PyPgQuery_FetchQueryList(self);
		PyPgQuery_FixQueryList(self, NULL);
		list_free(l);

		l = PyPgQuery_FetchPlanList(self);
		PyPgQuery_FixPlanList(self, NULL);
		list_free(l);

		MemoryContextDelete(PyPgQuery_FetchMemoryContext(self));
		PyPgQuery_FixMemoryContext(self, NULL);
	}

	self->ob_type->tp_free(self);
}

static PyObj
query_repr(PyObj self)
{
	PyObj argrepr, rob;
	argrepr = PyObject_Repr(PyPgQuery_FetchArgTypes(self));
	rob = PyString_FromFormat(
		"<Postgres.Query%s>",
		PyString_AS_STRING(argrepr)
	);
	Py_DECREF(argrepr);
	return(rob);
}

static PyObj
query_call(PyObj self, PyObj args, PyObj kw)
{
	int argc, argtypec;
	MemoryContext exec_context = PyPgQuery_FetchMemoryContext(self);
	PyObj argtypes, ob = NULL, rob = NULL;

	ListCell *queryc, *planc;
	List *queryl, *planl;

	if (!PyPgQuery_IsCurrent(self))
		if (prepare(self) < 0)
			return(NULL);

	argc = PyTuple_GET_SIZE(args);
	argtypes = PyPgQuery_FetchArgTypes(self);
	argtypec = PyTuple_GET_SIZE(argtypes);

	if (argc != argtypec)
	{
		PyErr_Format(PyExc_TypeError,
			"query requires %d arguments, given %d", argtypec, argc
		);
		return(NULL);
	}

	if (argc > 0)
	{
		int i;
		PyObj pgarg, pgargs;
		pgargs = PyTuple_New(argc);
		for (i = 0; i < argc; ++i)
		{
			pgarg = PyPgObject_FromPyPgTypeAndPyObject(
				PyTuple_GET_ITEM(argtypes, i),
				PyTuple_GET_ITEM(args, i)
			);
			PyTuple_SET_ITEM(pgargs, i, pgarg);
		}
		args = pgargs;
	}

	queryl = PyPgQuery_FetchQueryList(self);
	planl = PyPgQuery_FetchPlanList(self);
	forboth(queryc, queryl, planc, planl)
	{
		Query *query = (Query *) lfirst(queryc);
		Plan *plan = (Plan *) lfirst(planc);

		if (query->commandType == CMD_SELECT && query->into == NULL)
		{
			rob = PyPgPortal_New(self, args, query, plan);
		}
		else
		{
			MemoryContext former;
			PG_TRY();
			{
				former = MemoryContextSwitchTo(exec_context);
				if (query->commandType == CMD_INSERT
					|| query->commandType == CMD_UPDATE
					|| query->commandType == CMD_DELETE)
				{
					ob = execute(self, args, query, plan);
				}
				else if (query->commandType == CMD_UTILITY)
				{
					ob = utility(query->utilityStmt);
				}
			}
			PG_CATCH();
			{
				PyErr_SetPgError();
			}
			PG_END_TRY();
			MemoryContextSwitchTo(former);
			MemoryContextReset(exec_context);
			if (PyErr_Occurred())
				return(NULL);
		}

		/* Utility statements don't return anything, ob is NULL */
		if (ob)
		{
			if (rob)
			{
				if (PyList_CheckExact(rob))
					PyList_Append(rob, ob);
				else
				{
					PyObj l = PyList_New(0);
					PyList_Append(l, rob);
					DECREF(rob);
					PyList_Append(l, ob);
				}

				DECREF(ob);
			}
			else
				rob = ob;

			ob = NULL;
		}
	}

	if (rob)
		return(rob);
	else if (PyErr_Occurred())
		return(NULL);
	else
		RETURN_NONE;
}

static PyObj
query_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	PyObj qstr = NULL, rob;
	int q;

	if (PyCFunctionErr_NoKeywordsAllowed(kw))
		return(NULL);

	q = PySequence_Length(args);
	if (q != 1)
	{
		PyErr_Format(PyExc_TypeError,
			"instantiation requires 1 argument, given %d", q);
		return(NULL);
	}

	qstr = PySequence_GetItem(args, 0);
	if (qstr == NULL) return(NULL);

	rob = PyPgQuery_Initialize(subtype->tp_alloc(subtype, 0), qstr);
	DECREF(qstr);
	return(rob);
}

static PyObj
query_str(PyObj self)
{
	PyObj rob = PyPgQuery_FetchString(self);
	Py_INCREF(rob);
	return(rob);
}

static char PyPgQuery_Doc[] = "Postgres Query";
PyTypeObject PyPgQuery_Type = {
	PyObject_HEAD_INIT(NULL)
	0,										/* ob_size */
	"Postgres.Query",					/* tp_name */
	sizeof(struct PyPgQuery),		/* tp_basicsize */
	0,										/* tp_itemsize */
	query_dealloc,						/* tp_dealloc */
	NULL,									/* tp_print */
	NULL,									/* tp_getattr */
	NULL,									/* tp_setattr */
	NULL,									/* tp_compare */
	query_repr,							/* tp_repr */
	NULL,									/* tp_as_number */
	NULL,									/* tp_as_sequence */
	NULL,									/* tp_as_mapping */
	NULL,									/* tp_hash */
	query_call,							/* tp_call */
	query_str,							/* tp_str */
	NULL,									/* tp_getattro */
	NULL,									/* tp_setattro */
	NULL,									/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT,			   /* tp_flags */
	(char *) PyPgQuery_Doc,			/* tp_doc */
	NULL,									/* tp_traverse */
	NULL,									/* tp_clear */
	NULL,									/* tp_richcompare */
	0,										/* tp_weaklistoffset */
	NULL,									/* tp_iter */
	NULL,									/* tp_iternext */
	PyPgQuery_Methods,				/* tp_methods */
	PyPgQuery_Members,				/* tp_members */
	NULL,									/* tp_getset */
	NULL,									/* tp_base */
	NULL,									/* tp_dict */
	NULL,									/* tp_descr_get */
	NULL,									/* tp_descr_set */
	0,										/* tp_dictoffset */
	NULL,									/* tp_init */
	NULL,									/* tp_alloc */
	query_new,							/* tp_new */
};

PyObj
PyPgQuery_Initialize(PyObj self, PyObj query)
{
	PyObj qstr;

	if (self == NULL) return(NULL);

	PyPgQuery_FixLastOid(self, InvalidOid);
	PyPgQuery_FixArgTypes(self, NULL);
	PyPgQuery_FixRawParseTree(self, NULL);
	PyPgQuery_FixTransactionId(self, InvalidTransactionId);
	PyPgQuery_FixSubTransactionId(self, InvalidSubTransactionId);
	PyPgQuery_FixCommandId(self, FirstCommandId);

	qstr = PyObject_Str(query);
	if (qstr == NULL) goto free_self;
	PyPgQuery_FixString(self, qstr);

	if (prepare(self) < 0) goto DECREF_qstr;

	return(self);
DECREF_qstr:
	DECREF(qstr);
free_self:
	self->ob_type->tp_free(self);
	return(NULL);
}
/*
 * vim: ts=3:sw=3:noet:
 */
