/* $Id: function.c,v 1.6 2005/06/19 19:47:14 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * be/src/fcall.c,v 1.1.1.1 2004/12/16 00:32:18 flaw
 * imp/src/fcall.c,v 1.7 2004/10/20 16:00:35 flaw
 *//*
 * Implements Function Call interface; calls to non-Python Postgres Functions
 */
#include <setjmp.h>
#include <postgres.h>
#include <access/heapam.h>
#include <catalog/pg_type.h>
#include <catalog/pg_proc.h>
#include <commands/trigger.h>
#include <nodes/params.h>
#include <nodes/execnodes.h>
#include <parser/parse_type.h>
#include <parser/parse_oper.h>
#include <tcop/dest.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/catcache.h>
#include <utils/datum.h>
#include <utils/palloc.h>
#include <utils/builtins.h>
#include <utils/syscache.h>
#include <utils/tuplestore.h>
#include <pypg/postgres.h>
#include <pypg/environment.h>

#include <Python.h>
#include <compile.h>
#include <eval.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/error.h>
#include <pypg/utils.h>
#include <pypg/tupledesc.h>
#include <pypg/heaptuple.h>
#include <pypg/object.h>
#include <pypg/type.h>
#include <pypg/function.h>
#include <pypg/conv.h>
#include <pypg/call.h>
#include <pypg/call/function.h>

static void
FunctionCall_Arguments(Datum *args, bool *nulls, PyObj src)
{
	int nargs = PyList_GET_SIZE(src);
	int i;

	for (i = 0; i < nargs; ++i)
	{
		PyObj arg;
		arg = PyList_GET_ITEM(src, i);
		if (PyPgObject_IsNULL(arg))
		{
			args[i] = 0;
			nulls[i] = true;
		}
		else
		{
			/*
			 * XXX: Composite handling flaw
			 *
			 * In the case of calling a function of the same language,
			 * the argument will be initialized to 0, and it will be
			 * understood that the argument should be retrieved from the
			 * call's argument tuple.
			 *
			 * In the case of calling a function of a different language,
			 * a Datum tuple header should be duplicated out within the calling
			 * context(PyPgFunctionCall_FetchContext), which is always used when
			 * calling functions of different languages. Or if prior to 8, a
			 * tupletableslot should be created and passed in, instructing the
			 * receiving NOT to free the tuple descriptor or the heaptuple.
			 */
			args[i] = PyPgObject_FetchDatum(arg);
			nulls[i] = false;
		}
	}
}

static int
init_fcall(PyObj self)
{
	PyObj func = PyPgCall_FetchFunction(self);
	Form_pg_proc ps = PyPgFunction_FetchProcStruct(func);
	FunctionCallInfo fcinfo = PyPgFunctionCall_FetchFCInfo(self);
	FmgrInfo *flinfo = PyPgFunctionCall_FetchFMInfo(self);
	PyObj args = PyPgCall_FetchArguments(self);
	PyObj xtd = PyPgCall_FetchExpected(self);

	flinfo->fn_addr = PyPgFunction_FetchFunction(func);
	flinfo->fn_oid = PyPgObject_FetchOid(func);
	flinfo->fn_nargs = ps->pronargs;
	flinfo->fn_strict = ps->proisstrict;
	flinfo->fn_retset = ps->proretset;
	flinfo->fn_extra = NULL;
	/*
	 * Set a temporary function context instead.
	 */
	flinfo->fn_mcxt = AllocSetContextCreate(PythonMemoryContext,
		"PostgresPy Function Call Context",
		ALLOCSET_DEFAULT_MINSIZE,
		ALLOCSET_SMALL_INITSIZE,
		ALLOCSET_DEFAULT_MAXSIZE
	);
	flinfo->fn_expr = NULL;

	fcinfo->flinfo = flinfo;
	fcinfo->isnull = false;
	fcinfo->nargs = PyList_GET_SIZE(args);

	FunctionCall_Arguments(fcinfo->arg, fcinfo->argnull, args);

	if (xtd != Py_None || flinfo->fn_retset)
	{
		TupleDesc expectedDesc = NULL;
		ReturnSetInfo *rsi;
		if (xtd != Py_None)
		{
			if (PyPgTupleDesc_Check(xtd))
				expectedDesc = PyPgTupleDesc_FetchTupleDesc(xtd);
			else if (PyPgType_Check(xtd))
			{
				if (xtd != PyPgCall_FetchReturnType(self))
					expectedDesc = PyPgType_FetchTupleDesc(xtd);
			}
			else
			{
				MemoryContextDelete(flinfo->fn_mcxt);
				return(-1);
			}
		}

		rsi = MemoryContextAllocZero(flinfo->fn_mcxt, sizeof(ReturnSetInfo));
		fcinfo->resultinfo = (Node *) rsi;

		rsi->type = T_ReturnSetInfo;
		rsi->allowedModes = SFRM_ValuePerCall;
		rsi->isDone = ExprSingleResult;
		rsi->expectedDesc = expectedDesc;
	}
	else
		fcinfo->resultinfo = NULL;

	return(0);
}

static PyMethodDef PyPgFunctionCall_Methods[] = {
	/*{"name", FunctionRef, METH_NOARGS|METH_O|METH_VARARGS, "docstring"},*/
	{NULL}
};

static PyObj
return_set_item(PyObj self, TupleDesc expectedDesc, TupleDesc setDesc, Datum rd)
{
	PyObj retd = PyPgCall_FetchReturned(self);
	PyObj robtype;
	HeapTuple ht;
	PyObj rob = NULL;

	if (setDesc == NULL)
	{
		robtype = PyPgCall_FetchReturnType(self);
		if (PyPgType_IsComposite(robtype))
		{
			rob = PyPgObject_FromPyPgTypeAndHeapTuple(robtype, (HeapTuple) rd);
		}
		else
			rob = PyPgObject_FromPyPgTypeAndDatum(robtype, rd);
		return(rob);
	}

	if (setDesc == expectedDesc)
		robtype = PyPgCall_FetchExpected(self);
	else
	{
		if (retd != Py_None)
			robtype = PyPgHeapTuple_FetchPyPgTupleDesc(retd);
		else
		{
			setDesc = CreateTupleDescCopyConstr(setDesc);
			robtype = PyPgTupleDesc_New(setDesc);
			robtype->ob_refcnt = 0;
		}
	}

	ht = heap_copytuple((HeapTuple) rd);
	rob = PyPgHeapTuple_New(robtype, ht);
	return(rob);
}

static PyObj
invoke(PyObj self)
{
	ReturnSetInfo *rsi = PyPgFunctionCall_FetchRSI(self);
	volatile MemoryContext former;
	volatile PyObj rob = NULL;

	if (rsi != NULL)
	{
		if (rsi->isDone == ExprEndResult)
			RETURN_NONE;
	}

	PG_TRY();
	{
		Datum rd;
		former = PyPgFunctionCall_SwitchContext(self);
		rd = FunctionCallInvoke(PyPgFunctionCall_FetchFCInfo(self));
		MemoryContextSwitchTo(former);

		if (rsi == NULL)
		{
			PyObj rt = PyPgCall_FetchReturnType(self);
			rob = PyPgObject_FromPyPgTypeAndDatum(rt, rd);
		}
		else switch (rsi->isDone)
		{
			case ExprSingleResult:
			{
				rsi->isDone = ExprEndResult;
				rob = return_set_item(self, rsi->expectedDesc, rsi->setDesc, rd);
			}
			break;
			case ExprMultipleResult:
			{
				rsi->isDone = ExprSingleResult;
				rob = return_set_item(self, rsi->expectedDesc, rsi->setDesc, rd);
			}
			break;
			case ExprEndResult:
			{
				INCREF(Py_None);
				rob = Py_None;
			}
			break;
		}
	}
	PG_CATCH();
	{
		PyErr_SetPgError();
	}
	PG_END_TRY();

	return(rob);
}

static void
fcall_dealloc(PyObj self)
{
	MemoryContext fn_mcxt = PyPgFunctionCall_FetchFMInfo(self)->fn_mcxt;
	PyPgCall_Type.tp_dealloc(self);
	MemoryContextDelete(fn_mcxt);
}

static PyObj
fcall_iternext(PyObj self)
{
	PyObj rob = NULL;

	if (PyPgCall_FetchExpected(self) != Py_None)
		rob = invoke(self);

	if (rob == Py_None)
	{
		DECREF(rob);
		rob = NULL;
	}

	return(rob);
}

static PyObj
fcall_call(PyObj self, PyObj args, PyObj kw)
{
	PyObj rob;

	rob = invoke(self);
	if (rob == NULL) return(NULL);

	DECREF(PyPgCall_FetchReturned(self));
	PyPgCall_FixReturned(self, rob);
	INCREF(rob);

	return(rob);
}

static PyObj
fcall_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	PyObj rob = NULL;
	PyErr_SetString(PyExc_TypeError, "direct calls not yet supported");
	return(rob);
}

const char PyPgFunctionCall_Doc[] =
"Python interface to a Postgres trigger fire";

PyTypeObject PyPgFunctionCall_Type = {
	PyObject_HEAD_INIT(NULL)
	0,												/* ob_size */
	"Postgres.FunctionCall",				/* tp_name */
	sizeof(struct PyPgFunctionCall),		/* tp_basicsize */
	0,												/* tp_itemsize */
	(destructor)fcall_dealloc,				/* tp_dealloc */
	NULL,											/* tp_print */
	(getattrfunc)NULL,						/* tp_getattr */
	(setattrfunc)NULL,						/* tp_setattr */
	(cmpfunc)NULL,								/* tp_compare */
	(reprfunc)NULL,							/* tp_repr */
	NULL,											/* tp_as_number */
	NULL,											/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	(hashfunc)NULL,							/* tp_hash */
	(ternaryfunc)fcall_call,				/* tp_call */
	(reprfunc)NULL,							/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT|
	Py_TPFLAGS_HAVE_GC,						/* tp_flags */
	(char *) PyPgFunctionCall_Doc,		/* tp_doc */
	(traverseproc)NULL,						/* tp_traverse */
	(inquiry)NULL,								/* tp_clear */
	(richcmpfunc)NULL,						/* tp_richcompare */
	(long) 0,									/* tp_weaklistoffset */
	(getiterfunc)Py_RETURN_SELF,			/* tp_iter */
	(iternextfunc)fcall_iternext,			/* tp_iternext */
	PyPgFunctionCall_Methods,				/* tp_methods */
	NULL,											/* tp_members */
	NULL,											/* tp_getset */
	&PyPgCall_Type,							/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	fcall_new,									/* tp_new */
};

PyObj
PyPgFunctionCall_Initialize(PyObj self, PyObj func, PyObj args, PyObj xtd)
{
	self = PyPgCall_Initialize(self, func, args, xtd);
	if (self == NULL) return(NULL);
	if (init_fcall(self) < 0)
	{
		DECREF(self);
		self = NULL;
	}
	return(self);
}
/*
 * vim: ts=3:sw=3:noet:
 */
