/* $Id: pl.c,v 1.19 2005/12/14 19:25:54 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * Implements the Python interface to a procedural call
 */
#include <setjmp.h>
#include <postgres.h>
#include <miscadmin.h>
#include <access/heapam.h>
#include <catalog/pg_type.h>
#include <catalog/pg_proc.h>
#include <commands/trigger.h>
#include <executor/executor.h>
#include <nodes/params.h>
#include <nodes/execnodes.h>
#include <parser/parse_type.h>
#include <parser/parse_oper.h>
#include <tcop/dest.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/catcache.h>
#include <utils/datum.h>
#include <utils/palloc.h>
#include <utils/builtins.h>
#include <utils/syscache.h>
#include <utils/typcache.h>
#include <utils/tuplestore.h>
#include <pypg/postgres.h>
#include <pypg/environment.h>

#include <Python.h>
#include <compile.h>
#include <eval.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/error.h>
#include <pypg/tupledesc.h>
#include <pypg/heaptuple.h>
#include <pypg/type.h>
#include <pypg/type/object.h>
#include <pypg/type/record.h>
#include <pypg/function.h>
#include <pypg/call.h>
#include <pypg/call/pl.h>
#include <pypg/call/trigger.h>

static PyObj
materialize(PyObj self, PyObj iter)
{
	ReturnSetInfo *rsi = PyPgProceduralCall_FetchRSI(self);
	MemoryContext former, mmc = rsi->econtext->ecxt_per_query_memory;

	former = MemoryContextSwitchTo(mmc);
	PG_TRY();
	{
		HT_FromTDAndPyObj_Con cf = NULL;
		Tuplestorestate *tss;
		TupleDesc td = NULL;
		HeapTuple ht;
		PyObj ob;

		tss = tuplestore_begin_heap(true, true, work_mem);
		rsi->setResult = tss;

		ob = PyIter_Next(iter);
 		if (ob == NULL)
		{
			/*
			 * This should fall through the rest of the function giving
			 * an empty tuple store, as desired. If not, it will crash by
			 * calling a NULL function pointer.
			 */
		}
		else if (PyPgHeapTuple_Check(ob))
		{
			PyObj ptd = PyPgHeapTuple_FetchPyPgTupleDesc(ob);

			if (ptd == PyPgCall_FetchOutput(self))
				td = rsi->expectedDesc;
			else
				td = PyPgTupleDesc_FetchTupleDesc(ptd);
			cf = HeapTuple_FromTupleDescAndIterable;
		}
		else
		{
			PyObj func = PyPgCall_FetchObject(self);
			PyObj rt;
			rt = PyPgFunction_FetchOutput(func);

			if (PyPgType_IsComposite(rt))
			{
				cf = HeapTuple_FromTupleDescAndIterable;
				td = PyPgType_FetchTupleDesc(rt);
			}
			else
			{
				cf = HeapTuple_FromTupleDescAndPyObject;
				td = CreateTemplateTupleDesc(1, false);
				TupleDescInitEntry(td, 1, "?column?",
						PyPgType_FetchOid(rt), -1, -1, false);
			}
		}

		rsi->setDesc = td != rsi->expectedDesc ?
			CreateTupleDescCopyConstr(td) :
			rsi->expectedDesc;

		if (ob != NULL) do
		{
			ht = cf(rsi->setDesc, ob);
			Py_DECREF(ob);
			if (ht == NULL) break;
			tuplestore_puttuple(tss, (void *) ht);
			heap_freetuple(ht);
		} while ((ob = PyIter_Next(iter)) != NULL);
	}
	PG_CATCH();
	{
		PyErr_SetPgError();
	}
	PG_END_TRY();
	MemoryContextSwitchTo(former);

	if (PyErr_Occurred())
	{
		rsi->setResult = NULL;
		return(NULL);
	}

	RETURN_NONE;
}

static PyObj
srf_call(PyObj self, PyObj args, PyObj kw)
{
	PyObj rob = NULL;
	ReturnSetInfo *rsi = PyPgProceduralCall_FetchRSI(self);
	PyObj iter = PyPgCall_FetchReturned(self);

	if (iter == Py_None)
	{
		rob = self->ob_type->tp_base->tp_call(self, args, kw);
		if (rob == NULL)
			return(NULL);

		iter = PyObject_GetIter(rob);
		DECREF(rob);
		if (iter == NULL)
		{
			PyErr_Format(PyExc_TypeError, "expected iterable from SRF");
			return(NULL);
		}
		DECREF(PyPgCall_FetchReturned(self));
		PyPgCall_FixReturned(self, iter);
	}

	if (rsi->allowedModes & SFRM_Materialize)
	{
		rsi->returnMode = SFRM_Materialize;
		rsi->isDone = ExprSingleResult;
		rob = materialize(self, iter);
	}
	else
	{
		rsi->returnMode = SFRM_ValuePerCall;
		rob = PyIter_Next(iter);
		if (rob == NULL)
		{
			rsi->isDone = ExprEndResult;

			Py_DECREF(iter);

			PyPgCall_FixReturned(self, Py_None);
			Py_INCREF(Py_None);

			rob = Py_None;
			Py_INCREF(rob);
		}
		else
			rsi->isDone = ExprMultipleResult;
	}

	return(rob);
}

static PyObj
plcall_call(PyObj self, PyObj args, PyObj kw)
{
	static char *kwlist[] = {NULL};
	PyObj rob = NULL;
	Node *ri;
	ReturnSetInfo *rsi = NULL;

	if (!PyArg_ParseTupleAndKeywords(args, kw, "", kwlist))
		return(NULL);

	ri = PyPgProceduralCall_FetchResultInfo(self);
	rsi = (ri != NULL && IsA(ri, ReturnSetInfo)) ? (ReturnSetInfo *) ri : NULL;

	if (rsi && PyPgFunction_IsSRF(PyPgCall_FetchObject(self)))
		rob = srf_call(self, args, kw);
	else
	{
		rob = self->ob_type->tp_base->tp_call(self, args, kw);
		if (rob != NULL)
		{
			if (PyIter_Check(rob))
			{
				PyObj iter = rob;
				rob = PyIter_Next(iter);
				Py_DECREF(iter);
				if (rob == NULL && !PyErr_Occurred())
				{
					rob = Py_None;
					Py_INCREF(Py_None);
				}
			}
		}
	}

	if (rob != NULL && rob != Py_None && !PyPgObject_Check(rob))
	{
		PyObj expect = PyPgCall_FetchOutput(self);
		PyObj ob = rob;
		rob = Py_Call(expect, ob);
		DECREF(ob);
	}

	return(rob);
}

PyDoc_STRVAR(PyPgProceduralCall_Doc,
"Python interface to a Postgres trigger fire");

PyTypeObject PyPgProceduralCall_Type = {
	PyObject_HEAD_INIT(NULL)
	0,												/* ob_size */
	"Postgres.ProceduralCall",				/* tp_name */
	sizeof(struct PyPgProceduralCall),	/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	NULL,											/* tp_as_number */
	NULL,											/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	plcall_call,								/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT,						/* tp_flags */
	PyPgProceduralCall_Doc,					/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	NULL,											/* tp_getset */
	&PyPgCall_Type,							/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	NULL,											/* tp_new */
};

PyObj
PyPgProceduralCall_Initialize(PyObj self, FunctionCallInfo fcinfo)
{
	PyObj func, func_input, call_out;
	ReturnSetInfo *rsi;
	if (self == NULL) return(NULL);

	Assert(fcinfo != NULL);
	PyPgProceduralCall_FixFCInfo(self, fcinfo);
	rsi = PyPgProceduralCall_FetchRSI(self);

	func = PyPgFunction_FromOid(fcinfo->flinfo->fn_oid);
	if (func == NULL) goto free_self;
	PyPgCall_FixFunction(self, func);
	func_input = PyPgFunction_FetchInput(func);

	call_out = PyPgFunction_FetchOutput(func);
	if (rsi && PyPgFunction_IsSRF(func))
	{
		if (!(rsi->allowedModes & (SFRM_ValuePerCall|SFRM_Materialize)))
		{
			PyErr_Format(PyExc_TypeError,
				"only VPC and materialization are known");
			goto DECREF_func;
		}

		if (rsi->expectedDesc)
			call_out = PyPgTupleDesc_New(rsi->expectedDesc);
		else
			Py_INCREF(call_out);
	}
	else
		Py_INCREF(call_out);

	if (fcinfo->nargs > 0)
	{
		PyObj input;
		TupleDesc td;
		HeapTuple ht = NULL;

		td = PyPgTupleDesc_FetchTupleDesc(func_input);
		PgError_TRAP(
			ht = HeapTuple_FromTupleDescAndDatumNulls(
				td, fcinfo->arg, fcinfo->argnull
			)
		);
		if (PyErr_Occurred()) goto DECREF_call_out;

		input = PyPgHeapTuple_New(func_input, ht);
		heap_freetuple(ht);
		if (input == NULL) goto DECREF_call_out;
		PyPgCall_FixInput(self, input);
	}
	else
	{
		Py_INCREF(EmptyPyPgHeapTuple);
		PyPgCall_FixInput(self, EmptyPyPgHeapTuple);
	}

	PyPgCall_FixOutput(self, call_out);
	PyPgCall_FixReturned(self, Py_None);
	Py_INCREF(Py_None);

	return(self);
DECREF_call_out:
	Py_DECREF(call_out);
DECREF_func:
	Py_DECREF(func);
free_self:
	self->ob_type->tp_free(self);
	return(NULL);
}
/*
 * vim: ts=3:sw=3:noet:
 */
