/*****************************************************************************
 *
 * frame - Touch Frame Library
 *
 * Copyright (C) 2011-2012 Canonical Ltd.
 *
 * This library is free software: you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License version 3
 * as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranties of 
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
 * PURPOSE.  See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#include "device.h"

#include <assert.h>

#include "axis.h"

namespace oif {
namespace frame {

UFStatus UFDevice::GetAxisByIndex(unsigned int index, ::UFAxis* axis) const {
  if (index >= axes_.size())
    return UFStatusErrorInvalidAxis;

  auto it = axes_.cbegin();
  std::advance(it, index);

  *axis = it->second.get();

  return UFStatusSuccess;
}

UFStatus UFDevice::GetAxisByType(UFAxisType type, ::UFAxis* axis) const {
  auto it = axes_.find(type);
  if (it == axes_.end())
    return UFStatusErrorInvalidAxis;

  *axis = it->second.get();

  return UFStatusSuccess;
}

} // namespace frame
} // namespace oif

extern "C" {

FRAME_PUBLIC
UFStatus frame_device_get_property_string_(UFDevice device,
                                           UFDeviceProperty property,
                                           char **value) {
  return static_cast<const oif::frame::UFDevice*>(device)->GetProperty(
      property,
      value);
}

FRAME_PUBLIC
UFStatus frame_device_get_property_int_(UFDevice device,
                                        UFDeviceProperty property, int *value) {
  return static_cast<const oif::frame::UFDevice*>(device)->GetProperty(
      property,
      value);
}

FRAME_PUBLIC
UFStatus frame_device_get_property_unsigned_int_(UFDevice device,
                                                 UFDeviceProperty property,
                                                 unsigned int *value) {
  return static_cast<const oif::frame::UFDevice*>(device)->GetProperty(
      property,
      value);
}

#undef frame_device_get_property /* Override C11 generic selections macro */
FRAME_PUBLIC
UFStatus frame_device_get_property(UFDevice device, UFDeviceProperty property,
                                   void *value) {
  return static_cast<const oif::frame::UFDevice*>(device)->GetProperty(
      property,
      value);
}

UFStatus frame_device_get_axis_by_index(UFDevice device, unsigned int index,
                                        UFAxis* axis) {
  return static_cast<const oif::frame::UFDevice*>(device)->GetAxisByIndex(
      index,
      axis);
}

UFStatus frame_device_get_axis_by_type(UFDevice device, UFAxisType type,
                                       UFAxis* axis) {
  return static_cast<const oif::frame::UFDevice*>(device)->GetAxisByType(
      type,
      axis);
}

unsigned int frame_device_get_num_axes(UFDevice device) {
  unsigned int num_axes;
  UFStatus status = frame_device_get_property(device, UFDevicePropertyNumAxes,
                                              &num_axes);
  assert(status == UFStatusSuccess);
  return num_axes;
}

float frame_device_get_window_resolution_x(UFDevice device) {
  float resolution;
  UFStatus status = frame_device_get_property(device,
                                              UFDevicePropertyWindowResolutionX,
                                              &resolution);
  assert(status == UFStatusSuccess);
  return resolution;
}

float frame_device_get_window_resolution_y(UFDevice device) {
  float resolution;
  UFStatus status = frame_device_get_property(device,
                                              UFDevicePropertyWindowResolutionY,
                                              &resolution);
  assert(status == UFStatusSuccess);
  return resolution;
}

} // extern "C"
