/*
 * The following license applies to "mod_macro" version 1.1.4.
 * It is a third-party module by Fabien Coelho <fabien@coelho.net>
 * for the Apache Http Server (http://www.apache.org/).
 *
 * ====================================================================
 * Copyright (c) 1998-2002 Fabien Coelho. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by
 *    Fabien Coelho <fabien@coelho.net>
 *    for use in the mod_macro project
 *    (http://www.coelho.net/mod_macro/)."
 *
 * 4. The name "mod_macro" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For written permission, please contact
 *    Fabien Coelho <fabien@coelho.net>.
 *
 * 5. Products derived from this software may not be called "mod_macro"
 *    nor may "mod_macro" appear in their names without prior written
 *    permission of Fabien Coelho.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by
 *    Fabien Coelho <fabien@coelho.net>
 *    for use in the mod_macro project
 *    (http://www.coelho.net/mod_macro/)."
 *
 * 7. Any modification must be properly copyrighted by its author.
 *
 * THIS SOFTWARE IS PROVIDED BY FABIEN COELHO ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 */
/*
  $Id: mod_macro.src.c 1.70 2002/09/10 14:02:37 coelho Exp $

  mod_macro version 1.1.4.

  This modules allows the definition and use of macros within apache
  runtime configuration files. Patch suggestions may be sent to the
  author.

  Fabien Coelho <fabien@coelho.net>.
  URL: http://www.coelho.net/
*/

#include "httpd.h"
#include "http_config.h"
#include "http_log.h"

#include "apr.h"
#include "apr_strings.h"
#include "apr_portable.h"
#include "apr_file_io.h"


/************************************************ COMPILE TIME DEBUG CONTROL */
/*
   debug:
   #define MOD_MACRO_DEBUG 1

   gdb:
   run -f /users/cri/coelho/Misc/apache/mod_macro/Tests/test??.conf

   no warnings:
   #define MOD_MACRO_NO_WARNINGS 1
   #define MOD_MACRO_NO_CHAR_PREFIX_WARNINGS 1
*/

/* #define MOD_MACRO_DEBUG */

#if defined(debug)
#undef debug
#endif

#if defined(MOD_MACRO_DEBUG)
#define debug(stmt) stmt
#else
#define debug(stmt)
#endif

/************************************************************* ADVERTISEMENT */

module macro_module;

#define MACRO_MODULE_NAME       "mod_macro"
#define MACRO_MODULE_VERSION    "1.1.4"

/* I used to advertise the macro module. (I was quite young;-)
   ap_add_version_component(MACRO_MODULE_NAME "/" MACRO_MODULE_VERSION);
*/

/********************************************************** MACRO MANAGEMENT */

/* a macro: name, arguments, contents, location.
 */
typedef struct {
    char * name;              /* case-insensitive name of the macro. */
    apr_array_header_t * arguments; /* of char* */
    apr_array_header_t * contents;  /* of char* */
    char * location;          /* of the macro definition for error messages. */
} macro_t;

/* configuration tokens.
 */
#define BEGIN_MACRO "<Macro"
#define END_MACRO "</Macro>"
#define USE_MACRO "Use"

#define empty_string_p(p) (!(p) || *(p) == '\0')

/* macros are kept globally...
   they are not per-server or per-directory entities.
   there are initialized/closed by mod_macro_init/mod_macro_close
   this is also used as a tag to know whether it has been initialized.
   [bug on first reading/check of the configuration file...]
*/
static apr_array_header_t * macros = NULL;

/* returns the macro structure for name, or NULL if not found.
 */
static macro_t * get_macro_by_name(const char * name)
{
    int i;
    macro_t ** tab;

    debug(fprintf(stderr, "mod_macro:get_macro_by_name('%s')\n", name));

    ap_assert(macros);
    tab = (macro_t **)macros->elts;

    for (i = 0; i < macros->nelts; i++) {
        if (!strcasecmp(name, tab[i]->name)) {
            debug(fprintf(stderr, "macro found at index %d\n", i));
            return tab[i];
        }
    }
    debug(fprintf(stderr, "macro not found\n"));
    return NULL;
}

/* configuration state initialization.
   the state is simply an apr_array_header_t which holds the macros.
 */
static int mod_macro_init(
       apr_pool_t *pconf,
       apr_pool_t *plog,
       apr_pool_t *ptemp)
{
    debug(fprintf(stderr, "mod_macro_init(%p)\n", ptemp));

    /* ??? */
    if (macros) return 0;

    ap_assert(!macros);
    macros = apr_array_make(ptemp, 1, sizeof(macro_t *));
    return 0;
}

static int mod_macro_close(
        apr_pool_t *pconf,
        apr_pool_t *plog,
        apr_pool_t *ptemp,
        server_rec *s)
{
    debug(fprintf(stderr, "mod_macro_close(%p)\n", ptemp));

    /* ??? */
    if (!macros) return 0;

    ap_assert(macros);
    macros = NULL;
    return 0;
}

/*************************************************************** PARSE UTILS */

#define trim(line) while (*(line)==' ' || *(line)=='\t') (line)++

/* return configuration-parsed arguments from line as an array.
   the line is expected not to contain any '\n'?
 */
static apr_array_header_t * get_arguments(apr_pool_t * p, const char * line)
{
    apr_array_header_t * args = apr_array_make(p, 1, sizeof(char *));
    char * arg, ** new;

    trim(line);
    while (*line) {
        arg = ap_getword_conf(p, &line);
        new = apr_array_push(args);
        *new = arg;
        trim(line);
    }

    return args;
}

/* get read lines as an array till end_token.
   counts nesting for begin_token/end_token.
   it assumes a line-per-line configuration (thru getline).
   this function could be exported.
   begin_token may be NULL.
*/
static char * get_lines_till_end_token(apr_pool_t * p,
                                       ap_configfile_t * config_file,
                                       const char * end_token,
                                       const char * begin_token,
                                       const char * where,
                                       apr_array_header_t ** plines)
{
    apr_array_header_t * lines = apr_array_make(p, 1, sizeof(char *));
    char ** new, * first, * ptr;
    char line[MAX_STRING_LEN]; /* sorry, but that is expected by getline. */
    int macro_nesting = 1, any_nesting = 1, line_number = 0;

    while (!ap_cfg_getline(line, MAX_STRING_LEN, config_file)) {
        ptr = line;
        /* first char? or first non blank? */
        if (*line=='#') continue;
        first = ap_getword_conf_nc(p, &ptr);
        line_number++;
        if (first) {
            /* nesting... */
            if (!strncmp(first, "</", 2)) {
                any_nesting--;
#if !defined(MOD_MACRO_NO_WARNINGS)
                if (any_nesting<0) {
                    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_WARNING,
                                 0, NULL,
                                 "bad (negative) nesting on line %d of %s\n",
                                 line_number, where);
                }
#endif
            }
            else if (!strncmp(first, "<", 1)) {
                any_nesting++;
            }

            if (!strcasecmp(first, end_token)) { /* okay! */
                macro_nesting--;
                if (!macro_nesting) {
#if !defined(MOD_MACRO_NO_WARNINGS)
                    if (any_nesting) {
                        ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_WARNING,
                                     0, NULL,
                                     "bad cumulated nesting (%+d) in %s\n",
                                     any_nesting, where);
                    }
#endif
                    *plines = lines;
                    return NULL;
                }
            }
            else if (begin_token && !strcasecmp(first, begin_token)) {
                macro_nesting++;
            }
        }
        /* free first. */
        new  = apr_array_push(lines);
        *new = apr_psprintf(p, "%s\n", line); /* put '\n' back */
    }

    return apr_psprintf(p, "expected token not found: %s", end_token);
}

/* returns whether it looks like an argument, i.e. prefixed by ARGUMENT_PREFIX.
 */
#define ARGUMENT_PREFIX "$%&@#"

/* characters allowed in an argument? not used yet.
 */
#define ARGUMENT_CONTENT \
    "abcdefghijklmnopqrstuvwxyz" \
    "ABCDEFGHIJKLMNOPQRSTUVWXYZ" \
    "0123456789_" ARGUMENT_PREFIX

static int looks_like_an_argument(const char * word)
{
    return (int)strchr(ARGUMENT_PREFIX, *word);
}

/* generates an error on macro with two arguments of the same name.
   generates an error if a macro argument name is empty.
   generates a warning if arguments name prefixes conflict.
   generates a warning if the first char of an argument is not
       in ARGUMENT_PREFIX
*/
static const char * check_macro_arguments(apr_pool_t * p, const macro_t * macro)
{
    char ** tab = (char **)macro->arguments->elts;
    int nelts = macro->arguments->nelts, i, j;
    size_t ltabi, ltabj;

    for (i = 0; i < nelts; i++) {
        ltabi = strlen(tab[i]);

        if (ltabi == 0) {
            return apr_psprintf(p,
                               "macro '%s' (%s)\n\tempty argument #%d name\n",
                               macro->name, macro->location, i + 1);
        }

#if !defined(MOD_MACRO_NO_CHAR_PREFIX_WARNINGS) || \
    !defined(MOD_MACRO_NO_WARNINGS)
        if (!looks_like_an_argument(tab[i])) {
            ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_WARNING, 0, NULL,
                         "macro '%s' (%s)\n"
                        "\targument name '%s' (#%d) without expected prefix.\n"
         "\tit is good practice to prefix argument names with one of '%s'.\n",
                         macro->name, macro->location,
                         tab[i], i + 1, ARGUMENT_PREFIX);
        }
#endif

        for (j = i + 1; j < nelts; j++) {
            ltabj = strlen(tab[j]);

            if (!strcmp(tab[i], tab[j])) {
                return apr_psprintf(p,
                              "argument name conflict in macro '%s' (%s)\n"
                              "\targument '%s': #%d and #%d\n"
                              "\tchange argument names!",
                                   macro->name, macro->location,
                                   tab[i], i + 1, j + 1);
            }

#if !defined(MOD_MACRO_NO_WARNINGS)
            if (!strncmp(tab[i], tab[j], ltabi < ltabj ? ltabi : ltabj)) {
                ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_WARNING,
                             0, NULL,
                        "macro '%s' (%s)\n"
                        "\targument name prefix conflict (%s #%d and %s #%d)\n"
                        "\tbe careful about your macro definition!\n",
                             macro->name, macro->location,
                             tab[i], i + 1, tab[j], j + 1);
            }
#endif
        }
    }

    return NULL;
}

/* warn about empty strings in array.
 */
static void check_macro_use_arguments(const char * where,
                                      const apr_array_header_t * array)
{
    int i;
    char ** tab = (char **)array->elts;

#if !defined(MOD_MACRO_NO_WARNINGS)
    for (i = 0; i < array->nelts; i++) {
        if (empty_string_p(tab[i])) {
            ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_WARNING, 0, NULL,
                         "%s\n\tempty argument #%d\n", where, i + 1);
        }
    }
#endif
}

/******************************************************** SUBSTITUTION UTILS */

/* replace name by replacement at the beginning of buf of bufsize.
   returns an error message or NULL.
*/
static char * substitute(char * buf, int bufsize,
                         const char * name, const char * replacement)
{
    int lbuf  = strlen(buf),
        lname = strlen(name),
        lrepl = strlen(replacement),
        shift = lrepl - lname, size  = lbuf + shift, i;

    /* buf must starts with name */
    ap_assert(!strncmp(buf, name, lname));

    if (size >= bufsize) {
        /* could/should I reallocate? */
        return "cannot substitute, buffer size too small";
    }

    /* shift the end of line */
    if (shift < 0) {
        for (i = lname; i <= lbuf; i++)
            buf[i + shift] = buf[i];
    } else if (shift > 0) {
        for (i = lbuf; i >= lname; i--)
            buf[i + shift] = buf[i];
    }

    /* insert the replacement. */
    for (i = 0; i < lrepl; i++)
        buf[i] = replacement[i];

    return NULL;
}

/* find first occurence of args in buf.
   in case of conflict, the LONGEST argument is kept. (could be the FIRST?).
   returns the pointer and the whichone found, or NULL.
*/
static char * next_substitution(const char * buf,
                                const apr_array_header_t * args,
                                int * whichone)
{
    int i;
    char * chosen = NULL, * found, ** tab = (char **)args->elts;
    size_t lchosen = 0, lfound;

    for (i = 0; i < args->nelts; i++) {
        found = strstr(buf, tab[i]);
        lfound = strlen(tab[i]);
        if (found && (!chosen || found < chosen ||
                      (found == chosen && lchosen < lfound))) {
            chosen = found;
            lchosen = lfound;
            *whichone = i;
        }
    }

    return chosen;
}

/* substitute macro arguments by replacements in buf of bufsize.
   returns an error message or NULL.
   if used is defined, returns the used macro arguments.
*/
static char * substitute_macro_args(char * buf, int bufsize,
                                    const macro_t * macro,
                                    const apr_array_header_t * replacements,
                                    apr_array_header_t * used)
{
    char * ptr = buf, * errmsg,
        ** atab = (char **)macro->arguments->elts,
        ** rtab = (char **)replacements->elts;
    int whichone;

    if (used) {
        ap_assert(used->nalloc >= replacements->nelts);
    }

    while ((ptr = next_substitution(ptr, macro->arguments, &whichone))) {
        errmsg = substitute(ptr, buf - ptr + bufsize,
                            atab[whichone], rtab[whichone]);
        if (errmsg) {
            return errmsg;
        }
        ptr += strlen(rtab[whichone]);
        if (used) {
            used->elts[whichone] = 1;
        }
    }

    return NULL;
}

/* perform substitutions in a macro contents and
   return the result as a newly allocated array, if result is defined.
   may also return an error message.
   passes used down to substitute_macro_args.
*/
static const char * process_content(apr_pool_t * p,
                                    const macro_t * macro,
                                    const apr_array_header_t * replacements,
                                    apr_array_header_t * used,
                                    apr_array_header_t ** result)
{
    apr_array_header_t * contents = macro->contents;
    char ** new, * errmsg, line[MAX_STRING_LEN]; /* sorry again. */
    int i;

    if (result) {
        *result = apr_array_make(p, 1, sizeof(char *));
    }

    for (i = 0; i < contents->nelts; i++) {
        strncpy(line, ((char **)contents->elts)[i], MAX_STRING_LEN - 1);
        errmsg = substitute_macro_args(line, MAX_STRING_LEN,
                                       macro, replacements, used);
        if (errmsg) {
            return apr_psprintf(p, "while processing line %d of macro '%s'"
                               " (%s)\n\t%s",
                               i + 1, macro->name, macro->location, errmsg);
        }

        if (result) {
            new = apr_array_push(*result);
            *new = apr_pstrdup(p, line);
        }
    }

    return NULL;
}

/* warn if some macro arguments are not used.
 */
static const char * check_macro_contents(apr_pool_t * p, const macro_t * macro)
{
#if !defined(MOD_MACRO_NO_WARNINGS)

    int nelts = macro->arguments->nelts, i;
    apr_array_header_t * used;
    const char * errmsg;
    char ** names = (char **)macro->arguments->elts;

    if (macro->contents->nelts == 0) {
        ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_WARNING, 0, NULL,
                     "macro '%s' (%s)\n\tempty contents!\n",
                     macro->name, macro->location);
        return NULL; /* no need to further warnings... */
    }

    used = apr_array_make(p, nelts, sizeof(char));

    for (i = 0; i < nelts; i++) {
        used->elts[i] = 0;
    }

    errmsg = process_content(p, macro, macro->arguments, used, NULL);

    if (errmsg) {
        /* free used. */
        return errmsg;
    }

    for (i = 0; i < nelts; i++) {
        if (!used->elts[i]) {
            ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_WARNING, 0, NULL,
                         "macro '%s' (%s)\n\targument '%s' (#%d) never used\n",
                         macro->name, macro->location, names[i], i + 1);
        }
    }

    /* free used. */
#endif

    return NULL;
}

/********************************************************* MACRO CONFIG FILE */

/* the expanded content of the macro is to be parsed as a ap_configfile_t.
   the following struct stores the content.


   IMPORTANT NOTE:
   ---------------

   in http_config.c there is such a stuff made static,
   which does not implement getch().
   maybe this should be moved to util.c ???
*/
typedef struct {
    int index;                  /* current element. */
    int char_index;             /* current char in element. */
    int length;                 /* cached length of the current line. */
    apr_array_header_t * contents;      /* array of char * */
    ap_configfile_t * next;             /* next config once this one is processed. */
    ap_configfile_t ** upper;           /* hack: where to update it if needed. */
} array_contents_t;

/* next config if any. */
static int next_one(array_contents_t * ml)
{
    if (ml->next) {
        ap_assert(ml->upper);
        *(ml->upper) = ml->next;
        return 1;
    }
    return 0;
}

/* returns next char or -1.
 */
static int array_getch(void * param)
{
    array_contents_t * ml = (array_contents_t *)param;
    char ** tab = (char **)ml->contents->elts;

    while (ml->char_index >= ml->length) { /* next element */
        if (ml->index >= ml->contents->nelts) {
            /* maybe update. */
            if (ml->next && ml->next->getch && next_one(ml)) {
                return ml->next->getch(ml->next->param);
            }
            return -1;
        }
        ml->index++;
        ml->char_index = 0;
        ml->length = ml->index >= ml->contents->nelts
            ? 0 : strlen(tab[ml->index]);
    }

    return tab[ml->index][ml->char_index++];
}

/* returns a buf a la fgets.
   no more than a line at a time, otherwise the parsing is too much ahead...
   NULL at EOF.
*/
static void * array_getstr(void * buf, size_t bufsize, void * param)
{
    array_contents_t * ml = (array_contents_t *)param;
    char * buffer = (char *) buf;
    size_t i = 0;
    int next = 0;

    while (i < bufsize - 1 && next != '\n'
           && ((next = array_getch(param)) != -1)) {
        buffer[i++] = (char)next;
    }

    if (next == -1 && i == 0) { /* EOF */
        /* maybe update to next. */
        if (next_one(ml)) {
            ap_assert(ml->next->getstr);
            return ml->next->getstr(buf, bufsize, ml->next->param);
        }
        return NULL;
    }

    buffer[i] = '\0';
    return buf;
}

/* close the array stream?
 */
static int array_close(void * param)
{
    array_contents_t * ml = (array_contents_t *)param;
    ml->index = ml->contents->nelts;
    ml->char_index = ml->length;
    return 0;
}

/* this one could be exported.
 */
static ap_configfile_t * make_array_config(apr_pool_t * p,
                                        apr_array_header_t * contents,
                                        const char * where,
                                        ap_configfile_t * cfg,
                                        ap_configfile_t ** upper)
{
    array_contents_t * ls =
        (array_contents_t *)apr_palloc(p, sizeof(array_contents_t));

    ls->index      = 0;
    ls->char_index = 0;
    ls->contents   = contents;
    ls->length     = ls->contents->nelts < 1
                ? 0 : strlen(((char **)ls->contents->elts)[0]);
    ls->next       = cfg;
    ls->upper      = upper;

    return ap_pcfg_open_custom(p, where, (void *)ls,
                               array_getch, array_getstr, array_close);
}


/********************************************************** KEYWORD HANDLING */

/* handles: <Macro macroname arg1 arg2 ...> any trash...
 */
static const char * macro_section(cmd_parms * cmd,
                                  void * dummy,
                                  const char * arg)
{
    const char * errmsg, * where;
    char ** new, * name, * endp = strrchr(arg, '>');
    macro_t * macro, * old;

    /* LAZY ??? */
    if (!macros) {
      mod_macro_init(NULL, NULL, cmd->temp_pool);
    }

    debug(fprintf(stderr, "macro_section -%s-\n", arg));
    ap_assert(cmd->config_file);

    /* hmmm... drops out '>[^>]*$'
     */
    if (endp) {
        *endp = '\0';
    }

    /* get name. */
    name = ap_getword_conf(cmd->temp_pool, &arg);

    debug(fprintf(stderr, "macro name is '%s'\n", name));

    if (empty_string_p(name)) {
        return "macro definition: name not specified";
    }

    /* BUG on first reading??? */
    /* if (!macros) {
      debug(fprintf(stderr, "macros not initialized???\n"));
      return NULL;
      }*/

    old = get_macro_by_name(name);
    if (old) {
#if !defined(MOD_MACRO_NO_WARNINGS)
        /* already define: warn about the redefinition. */
        ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_WARNING, 0, NULL,
                     "macro '%s' multiply defined.\n"
                     "\t%s, redefined on line %d of %s",
                     old->name, old->location,
                     cmd->config_file->line_number, cmd->config_file->name);
#endif
        macro = old;
        debug(fprintf(stderr, "macro %s is old\n", name));
    }
    else {
        macro = (macro_t *)apr_palloc(cmd->temp_pool, sizeof(macro_t));
        debug(fprintf(stderr, "macro %s is new (%p)\n", name, macro));
    }

    macro->name = name;

    debug(fprintf(stderr, "cmd=%p config_file=%p\n", cmd, cmd->config_file));

    /* get arguments. */
    macro->location = apr_psprintf(cmd->temp_pool,
                                   "defined on line %d of %s",
                                   cmd->config_file->line_number,
                                   cmd->config_file->name);

    where = apr_psprintf(cmd->temp_pool, "macro '%s' (%s)",
                         macro->name, macro->location);

    debug(fprintf(stderr, "macro where='%s'\n", where));

#if !defined(MOD_MACRO_NO_CHAR_PREFIX_WARNINGS) || \
    !defined(MOD_MACRO_NO_WARNINGS)
    if (looks_like_an_argument(name)) {
        ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_WARNING, 0, NULL,
                     "%s\n\tit is good practice not "
                     "to prefix a macro name with any of '%s'\n",
                     where, ARGUMENT_PREFIX);
    }
#endif

    macro->arguments = get_arguments(cmd->temp_pool, arg);

    errmsg = check_macro_arguments(cmd->temp_pool, macro);

    if (errmsg) {
        return errmsg;
    }

    debug(fprintf(stderr, "macro getting contents\n"));

    /* get contents */
    errmsg = get_lines_till_end_token(cmd->temp_pool, cmd->config_file,
                                      END_MACRO, BEGIN_MACRO,
                                      where, &macro->contents);

    if (errmsg) {
        return apr_psprintf(cmd->temp_pool,
                           "%s\n\tcontents error: %s", where, errmsg);
    }

    errmsg = check_macro_contents(cmd->temp_pool, macro);

    if (errmsg) {
        return apr_psprintf(cmd->temp_pool,
                           "%s\n\tcontents checking error: %s", where, errmsg);
    }

    /* add the new macro. */
    new  = apr_array_push(macros);
    *new = (char *)macro;

    return NULL;
}

/* handles: Use name value1 value2 ...
 */
static const char * use_macro(cmd_parms * cmd, void * dummy, const char * arg)
{
    char * name, * where, * recursion;
    const char * errmsg;
    apr_array_header_t * contents, * replacements;
    macro_t * macro;

    debug(fprintf(stderr, "use_macro -%s-\n", arg));

    name = ap_getword_conf(cmd->temp_pool, &arg);

    if (empty_string_p(name)) {
        return "no macro name specified in " USE_MACRO;
    }

    macro = get_macro_by_name(name);

    if (!macro) {
        return apr_psprintf(cmd->temp_pool, "macro '%s' is not defined", name);
    }

    /* recursion is detected by looking at the config file name,
       which may already contains "macro 'foo'". Ok, it looks like a hack,
       but otherwise it is uneasy to keep this data available somewhere...
       the name has just the needed visibility and liveness.
    */
    recursion = apr_pstrcat(cmd->temp_pool, "macro '", macro->name, "'", NULL);

    if (strstr(cmd->config_file->name, recursion)) {
        return apr_psprintf(cmd->temp_pool,
                           "%s\n\trecursive use of macro '%s' is invalid.",
                           where, macro->name);
    }

    replacements = get_arguments(cmd->temp_pool, arg);

    if (macro->arguments->nelts != replacements->nelts) {
        return apr_psprintf(cmd->temp_pool,
                          "use of macro '%s' %s\n"
                          "\twith %d argument%s instead of %d",
                           macro->name, macro->location, replacements->nelts,
                           replacements->nelts > 1 ? "s" : "", /* grammar;-) */
                           macro->arguments->nelts);
    }

    where = apr_psprintf(cmd->temp_pool,
                        "macro '%s' (%s) used on line %d of %s",
                        macro->name, macro->location,
                        cmd->config_file->line_number,
                        cmd->config_file->name);

    check_macro_use_arguments(where, replacements);

    errmsg = process_content(cmd->temp_pool, macro, replacements,
                             NULL, &contents);

    if (errmsg) {
        return apr_psprintf(cmd->temp_pool,
                           "%s\n\terror while substituting:\n%s",
                           where, errmsg);
    }

    /* fix??? why is it wrong? should I -- the new one? */
    cmd->config_file->line_number++;

    cmd->config_file = make_array_config
        (cmd->temp_pool, contents, where, cmd->config_file, &cmd->config_file);

    return NULL;
}

/* handles: a lonely </Macro> or other unexpected keyword.
 * such a function already exists? where? could be in util.c or config.c?
 */
static const char * unexpected_keyword(cmd_parms *parms, void * dummy)
{
    return apr_psprintf(parms->temp_pool,
                       "unexpected %s encountered", (char *)parms->info);
}

/************************************ ERROR AND WARNING DURING CONFIGURATION */

/* maybe ConfigurationError and ConfigurationWarning could be used?
 */
#define ERROR_KEYWORD "Error"
#define WARNING_KEYWORD "Warning"

/* configuration generated errors or warnings.
 */
static const char * say_it(cmd_parms * parms, void * dummy, const char * arg)
{
    int level = (int)parms->info;
    trim(arg);

    ap_log_error(APLOG_MARK, APLOG_NOERRNO|level, 0, NULL,
                 "on line %d of %s:\n\t%s\n",
                 parms->config_file->line_number,
                 parms->config_file->name, arg);

    return level & APLOG_ERR ?
        "configuration file processing aborted by " ERROR_KEYWORD "." : NULL;
}

/************************************************************* EXPORT MODULE */

/* macro module commands.
 * the EXEC_ON_READ option is needed in apache 2 to get the config_file field.
 */
static const command_rec macro_cmds[] =
{
    /* configuration file macro stuff
     */
    AP_INIT_RAW_ARGS(BEGIN_MACRO, macro_section,
                     NULL, EXEC_ON_READ|OR_ALL,
                     "Beginning of a macro definition section."),
    AP_INIT_NO_ARGS(END_MACRO, unexpected_keyword,
                    (void*)END_MACRO, EXEC_ON_READ|OR_ALL,
                    "End of a macro definition section."),
    AP_INIT_RAW_ARGS(USE_MACRO, use_macro,
                     NULL, EXEC_ON_READ|OR_ALL,
                     "Use of a macro."),

    /* configuration errors and warnings.
     */
    AP_INIT_RAW_ARGS(ERROR_KEYWORD, say_it,
                     (void*)APLOG_ERR, EXEC_ON_READ|OR_ALL,
                     "Error in a configuration file."),
    AP_INIT_RAW_ARGS(WARNING_KEYWORD, say_it,
                     (void*)APLOG_WARNING, EXEC_ON_READ|OR_ALL,
                     "Warning in a configuration file."),

    { NULL }
};

/* module init and close
 */
static void register_hooks(apr_pool_t *p)
{
  debug(fprintf(stderr, "mod_macro:register_hooks()\n"));
  ap_hook_pre_config(mod_macro_init, NULL, NULL, APR_HOOK_LAST);
  ap_hook_post_config(mod_macro_close, NULL, NULL, APR_HOOK_LAST);
}

AP_DECLARE_DATA module macro_module = {
    STANDARD20_MODULE_STUFF,
    NULL,               /* create per-directory config  */
    NULL,               /* merge per-directory config structures */
    NULL,               /* create per-server config structure */
    NULL,               /* merge per-server config structures */
    macro_cmds,         /* command table, that's all there is! */
    register_hooks      /* register hooks */
};
