/*
 * Copyright 2011 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of either or both of the following licenses:
 *
 * 1) the GNU Lesser General Public License version 3, as published by the
 * Free Software Foundation; and/or
 * 2) the GNU Lesser General Public License version 2.1, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the applicable version of the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of both the GNU Lesser General Public
 * License version 3 and version 2.1 along with this program.  If not, see
 * <http://www.gnu.org/licenses/>
 */
#ifndef GRIP_INPUT_DEVICE_H_
#define GRIP_INPUT_DEVICE_H_

#include <glib-object.h>

G_BEGIN_DECLS

/**
 * GripDeviceType:
 * @GRIP_DEVICE_TOUCHSCREEN: device is a touchscreen.
 * @GRIP_DEVICE_TOUCHPAD: device is a touchpad
 * @GRIP_DEVICE_INDEPENDENT: device is a Magic Mouse
 *
 * Describes certain properties of a gesture input device.
 */
typedef enum {
  GRIP_DEVICE_TOUCHSCREEN = 1,
  GRIP_DEVICE_TOUCHPAD    = 2,
  GRIP_DEVICE_INDEPENDENT = 4,
} GripDeviceType;

/**
 * SECTION:GripInputDevice
 *
 * The Grip input device represents input devices known to the uTouch gesture
 * recognition stack.
 */

typedef struct GripAxisExtents
{
  gfloat minimum;
  gfloat maximum;
  gfloat resolution;
} GripAxisExtents;

#ifndef G_IR_IGNORE
/**
 * AxisExtents: (skip)
 *
 * A deprecated name for GripAxisExtents.
 */
typedef GripAxisExtents AxisExtents __attribute__((deprecated));
#endif

GType grip_input_device_get_type (void) G_GNUC_CONST;

#define GRIP_TYPE_INPUT_DEVICE \
                (grip_input_device_get_type ())
#define GRIP_INPUT_DEVICE(obj) \
                (G_TYPE_CHECK_INSTANCE_CAST ((obj), \
                 GRIP_TYPE_INPUT_DEVICE, \
                 GripInputDevice))
#define GRIP_INPUT_DEVICE_CLASS(obj) \
                (G_TYPE_CHECK_CLASS_CAST ((obj), \
                 GRIP_TYPE_INPUT_DEVICE, \
                 GripInputDeviceClass))
#define GRIP_IS_INPUT_DEVICE(obj) \
                (G_TYPE_CHECK_INSTANCE_TYPE ((obj), \
                 GRIP_TYPE_INPUT_DEVICE))
#define GRIP_IS_INPUT_DEVICE_CLASS(obj) \
                (G_TYPE_CHECK_CLASS_TYPE ((obj), \
                 GRIP_TYPE_INPUT_DEVICE))
#define GRIP_INPUT_DEVICE_GET_CLASS(obj) \
                (G_TYPE_INSTANCE_GET_CLASS ((obj), \
                 GRIP_TYPE_INPUT_DEVICE, \
                 GripInputDeviceClass))

typedef struct GripInputDevicePrivate GripInputDevicePrivate;

/**
 * GripInputDevice:
 *
 * This struct contains private data only, and should be accessed using the
 * functions below.
 */
typedef struct {
  GObject parent;
  GripInputDevicePrivate *priv;
} GripInputDevice;

typedef struct {
        GObjectClass parent_class;
} GripInputDeviceClass;

guint
grip_input_device_get_id(GripInputDevice *input_device);

const gchar *
grip_input_device_get_name(GripInputDevice *input_device);

gboolean
grip_input_device_is_direct(GripInputDevice *input_device);

gboolean
grip_input_device_is_independent(GripInputDevice *input_device);

const GripAxisExtents *
grip_input_device_get_x_extents(GripInputDevice *input_device);

const GripAxisExtents *
grip_input_device_get_y_extents(GripInputDevice *input_device);


GripDeviceType
grip_get_device_type (GripInputDevice *input_device);

G_END_DECLS

#endif /* GRIP_INPUT_DEVICE_H_ */
