/*
 * Copyright (C) 2011 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

using GLib;
using Dee;

namespace Unity {

/*
 * The private implementation of the Lens. This makes sure that none of the 
 * implementation details leak out into the public interface.
 */
private class LensImpl : GLib.Object, LensService
{
  private unowned Lens _owner;
  private uint _dbus_id;
  private uint _info_changed_id;
  private uint _sources_update_id;

  private Dee.SharedModel _results_model;
  private Dee.SharedModel _global_results_model;
  private Dee.SharedModel _categories_model;
  private Dee.SharedModel _filters_model;

  private OptionsFilter _sources;

  private ScopeFactory _scope_factory;

  private FiltersSynchronizer _filters_sync;
  private ResultsSynchronizer _results_sync;
  private ResultsSynchronizer _global_results_sync;

  public ResultsSynchronizer results_sync { get { return _results_sync; } }
  public ResultsSynchronizer global_results_sync { get { return _global_results_sync; } }

  // for testing
  private static unowned string? LENS_DIRECTORY = null;
  static construct
  {
    // FIXME: enclose this in a define to compile it away?
    LENS_DIRECTORY = Environment.get_variable ("LIBUNITY_LENS_DIRECTORY");
    if (LENS_DIRECTORY == null) LENS_DIRECTORY = "";
  }

  public LensImpl (Lens owner)
  {
    /* NOTE: Vala isn't allowing me to make Owner a construct variable so our
     * construction happens here instead of in construct {}
     */
    _owner = owner;
    _owner.notify["search-in-global"].connect (queue_info_changed);
    _owner.notify["visible"].connect (queue_info_changed);
    _owner.notify["search-hint"].connect (queue_info_changed);
    _owner.notify["sources-display-name"].connect (sources_display_name_changed);

    _sources = new CheckOptionFilter (Lens.SOURCES_FILTER_ID,
                                      _owner.sources_display_name,
                                      null, true);
    create_models (create_dbus_name ());
    create_synchronizers ();

    /* ScopeFactory handles finding Scopes and creating their proxies */
    _scope_factory = new ScopeFactory (_owner.id, LENS_DIRECTORY);
    _scope_factory.scope_added.connect (on_scope_added);
  }

  /* Create usable name prefix for the models */
  private string create_dbus_name ()
  {
    /* We make sure the name is unique to this instance so there is no chance of
     * old instances, or Unity itself, from stopping the models we create be the
     * swarm leaders
     */
    TimeVal tv = TimeVal ();
    int64 milli = tv.tv_sec;
    milli = milli * 1000 + tv.tv_usec / 1000;
    return ("com.canonical.Unity.Lens.%s.T%" + int64.FORMAT + "").printf (
      _owner.id, milli);
  }

  private Dee.SharedModel create_master_model (string swarm_name)
  {
    /* Create a peer instance that tries to grab the swarm leadership */
    var peer = Object.@new (typeof (Dee.Peer),
                            "swarm-name", swarm_name,
                            "swarm-owner", true, null) as Dee.Peer;
    /* This will create a SharedModel where only the leader can write to it */
    var access_mode = Dee.SharedModelAccessMode.LEADER_WRITABLE;
    return Object.@new (typeof (Dee.SharedModel),
                        "peer", peer,
                        "back-end", new Dee.SequenceModel (),
                        "access-mode", access_mode, null) as Dee.SharedModel;
  }

  private void create_models (string dbus_name)
  {
    /* Schema definitions come from the Lens specification */
    _results_model = create_master_model (dbus_name + ".Results");
    _results_model.set_schema ("s", "s", "u", "s", "s", "s", "s");

    _global_results_model = create_master_model (dbus_name + ".GlobalResults");
    _global_results_model.set_schema ("s", "s", "u", "s", "s", "s", "s");

    _categories_model = create_master_model (dbus_name + ".Categories");
    _categories_model.set_schema ("s", "s", "s", "a{sv}");

    /* Filters are updated by unity, so we can't create a master model */
    _filters_model = new Dee.SharedModel (dbus_name + ".Filters");
    _filters_model.set_schema ("s", "s", "s", "s", "a{sv}", "b", "b", "b");
  }

  private void create_synchronizers ()
  {
    _results_sync = new ResultsSynchronizer (_results_model);
    _global_results_sync = new ResultsSynchronizer (_global_results_model);
    _filters_sync = new FiltersSynchronizer (_filters_model);

    /* need to special-case sources filter */
    _filters_model.row_changed.connect ((model, iter) =>
    {
      if (model.get_string (iter, 0) != Lens.SOURCES_FILTER_ID) return;

      sources_filter_changed (model, iter);
    });
  }

  public void export () throws IOError
  {
    var conn = Bus.get_sync (BusType.SESSION);
    _dbus_id = conn.register_object (_owner.dbus_path, this as LensService);

    queue_info_changed ();
  }

  public void load_categories (List<Category> categories)
  {
    foreach (Category category in categories)
    {
      string icon_hint_s = category.icon_hint != null ?
                                         category.icon_hint.to_string() : "";
      _categories_model.append (category.name,
                                icon_hint_s,
                                category.renderer,
                                Tools.hash_table_to_asv (category.hints));
    }
  }

  public void load_filters (List<Filter> filters)
  {
    _filters_model.clear ();

    /* install change notifiers on the filters */
    foreach (unowned Filter filter in filters)
    {
      filter.changed.connect (on_filter_changed);
    }

    List<unowned Filter> filters_and_sources = filters.copy ();
    if (_sources.options.length () > 0)
      filters_and_sources.append (_sources);

    foreach (unowned Filter filter in filters_and_sources)
    {
      string icon_hint_s = filter.icon_hint != null ?
                                         filter.icon_hint.to_string() : "";
      _filters_model.append (filter.id,
                             filter.display_name,
                             icon_hint_s,
                             Filter.get_renderer_name (filter.renderer),
                             Tools.hash_table_to_asv (filter.get_hints()),
                             filter.visible,
                             filter.collapsed,
                             filter.filtering);
    }
  }

  /* Queue up info-changed requests as we don't want to be spamming Unity with
   * them.
   */
  private void queue_info_changed ()
  {
    if (_info_changed_id == 0)
    {
      _info_changed_id = Idle.add (emit_info_changed);
    }
  }

  private bool emit_info_changed ()
  {
    var info = LensInfo();
    info.dbus_path = _owner.dbus_path;
    info.search_in_global = _owner.search_in_global;
    info.search_hint = _owner.search_hint;
    info.visible = _owner.visible;
    info.private_connection_name = "<not implemented>";
    info.results_model_name = _results_model.get_swarm_name ();
    info.global_results_model_name = _global_results_model.get_swarm_name ();
    info.categories_model_name = _categories_model.get_swarm_name ();
    info.filters_model_name = _filters_model.get_swarm_name ();
    info.hints = new HashTable<string, Variant> (null, null);
    
    changed (info);

    _info_changed_id = 0;
    return false;
  }

  private void on_scope_added (ScopeFactory factory, ScopeProxy scope)
  {
    if (scope.results_model is Dee.Model)
      on_scope_results_model_changed (scope, null);
    
    if (scope.global_results_model is Dee.Model)
      on_scope_global_results_model_changed (scope, null);
    
    if (scope.filters_model is Dee.Model)
      on_scope_filters_model_changed (scope, null);
    
    /* Update the synchronizers if the models are switched.
     * Needed fx. because remote scopes sets up Dee.SharedModels lazily.
     * Local scopes emits the expected signals in an idle call after
     * construction */
    scope.notify["results-model"].connect (on_scope_results_model_changed);
    scope.notify["global-results-model"].connect (on_scope_global_results_model_changed);
    scope.notify["filters-model"].connect (on_scope_filters_model_changed);
    scope.notify["sources"].connect (on_scope_sources_updated);
  }

  private void on_scope_results_model_changed (Object obj, ParamSpec? pspec)
  {
    ScopeProxy? scope = obj as ScopeProxy;
    
    // FIXME: Remove existing provider, if any
    
    _results_sync.add_provider (scope.results_model, uid_for_scope (scope));
  }

  private void on_scope_global_results_model_changed (Object obj, ParamSpec? pspec)
  {
    ScopeProxy? scope = obj as ScopeProxy;
    
    // FIXME: Remove existing provider, if any
    
    _global_results_sync.add_provider (scope.global_results_model, "%p".printf(scope));
  }

  private void on_scope_filters_model_changed (Object obj, ParamSpec? pspec)
  {
    ScopeProxy? scope = obj as ScopeProxy;
    
    // FIXME: Remove existing receiver, if any
    
    /* Because we need to manipulate the filters straight away,
     * we have to wait until it's synchronized
     */
    if (scope is ScopeProxyRemote && (scope.filters_model as Dee.SharedModel).synchronized == false)
    {
      scope.filters_model.notify["synchronized"].connect (() =>
      {
        _filters_sync.add_receiver (scope.filters_model);
      });
    }
    else
      _filters_sync.add_receiver (scope.filters_model);
  }

  private void on_scope_sources_updated (Object obj, ParamSpec pspec)
  {
    ScopeProxy? scope = obj as ScopeProxy;

    var touched_filters = new Gee.HashSet<FilterOption> ();
    
    foreach (var filter in scope.sources.options)
    {
      var mangled_id = "%s:%s".printf (uid_for_scope (scope), filter.id);
      var option = _sources.get_option (mangled_id);
      if (option == null)
      {
        var new_option = _sources.add_option (mangled_id,
                                              filter.display_name,
                                              filter.icon_hint);
        new_option.active = filter.active;
        touched_filters.add (new_option);
      }
      else
      {
        option.active = filter.active;
        touched_filters.add (option);
      }
    }

    /* remove options that were removed */
    var prefix = "%s:".printf (uid_for_scope (scope));
    var removed_ids = new Gee.HashSet<unowned string> ();

    foreach (var filter_option in _sources.options)
    {
      if (filter_option.id.has_prefix (prefix) && 
        !(filter_option in touched_filters))
      {
        removed_ids.add (filter_option.id);
      }
    }

    foreach (unowned string id in removed_ids)
    {
      _sources.remove_option (id);
    }

    queue_sources_update ();
  }

  private void on_filter_changed (Filter filter)
  {
    var iter = _filters_model.get_first_iter ();
    while (iter != _filters_model.get_last_iter ())
    {
      if (_filters_model.get_string (iter, FilterColumn.ID) == filter.id)
      {
        string icon_hint_s = filter.icon_hint != null ?
                                           filter.icon_hint.to_string() : "";
        _filters_model.set (iter,
                            filter.id,
                            filter.display_name,
                            icon_hint_s,
                            Filter.get_renderer_name (filter.renderer),
                            Tools.hash_table_to_asv (filter.get_hints()),
                            filter.visible,
                            filter.collapsed,
                            filter.filtering);

        break;
      }
    }
  }

  private void sources_display_name_changed (Object obj, ParamSpec pspec)
  {
    _sources.display_name = _owner.sources_display_name;
    queue_sources_update ();
  }

  private void queue_sources_update ()
  {
    if (_sources_update_id == 0)
    {
      _sources_update_id = Idle.add (() =>
      {
        /* the last row in the model have to be sources */
        bool empty_model =
          _filters_model.get_first_iter () == _filters_model.get_last_iter ();

        if (!empty_model)
        {
          var iter = _filters_model.get_last_iter ();
          iter = _filters_model.prev (iter);
          if (_filters_model.get_string (iter, FilterColumn.ID) == Lens.SOURCES_FILTER_ID)
          {
            _filters_model.remove (iter);
          }
        }

        if (_sources.options.length () > 0)
        {
          var filter = _sources;

          string icon_hint_s = filter.icon_hint != null ?
                                             filter.icon_hint.to_string() : "";
          _filters_model.append (filter.id,
                                 filter.display_name,
                                 icon_hint_s,
                                 Filter.get_renderer_name (filter.renderer),
                                 Tools.hash_table_to_asv (filter.get_hints()),
                                 filter.visible,
                                 filter.collapsed,
                                 filter.filtering);
        }

        _sources_update_id = 0;
        return false;
      });
    }
  }

  private void sources_filter_changed (Dee.Model model, Dee.ModelIter iter)
  {
    /* this is called when unity changes the filter model */
    var temp = new CheckOptionFilter (_sources.id, _sources.display_name,
                                      _sources.icon_hint, _sources.collapsed);
    temp.filtering = model.get_bool (iter, FilterColumn.FILTERING);
    var properties = model.get_value (iter, FilterColumn.RENDERER_STATE);
    temp.update (properties);

    var updated_scopes = new Gee.HashSet<string> ();
    bool force_update = false;

    /* if the filtering state was toggled, we need to update all scopes */
    if (temp.filtering != _sources.filtering) force_update = true;

    /* get a list of scopes whose sources changed */
    foreach (var filter_option in temp.options)
    {
      FilterOption? current_value = _sources.get_option (filter_option.id);
      if (current_value == null) continue;
      if (force_update || current_value.active != filter_option.active)
      {
        string[] tokens = filter_option.id.split (":", 2);
        updated_scopes.add (tokens[0]);
      }
    }

    _sources = temp;

    update_active_sources (updated_scopes);
  }

  public void add_local_scope (Scope scope)
  {
    _scope_factory.add_local_scope (scope);
  }

  public unowned OptionsFilter get_sources ()
  {
    return _sources;
  }

  public unowned Dee.Model? get_model (int index)
    requires (index >= 0 && index <= 3)
  {
    switch (index)
    {
      case 0: return _results_model;
      case 1: return _global_results_model;
      case 2: return _filters_model;
      case 3: return _categories_model;
    }

    return null;
  }

  /*
   * DBus Interface Implementation
   */
  public async void info_request ()
  {
    queue_info_changed ();
  }

  public async ActivationReplyRaw activate (string uri,
                                            uint action_type) throws IOError
  {
    string[] tokens = uri.split(":", 2);
    ScopeProxy? scope = get_scope_for_uid (tokens[0]);
    var raw = ActivationReplyRaw();
    raw.handled = HandledType.NOT_HANDLED;
    raw.hints = new HashTable<string, Variant> (null, null);

    if (scope is ScopeProxy)
      raw = yield scope.activate(tokens[1], action_type);

    raw.uri = uri;
    return raw;
  }

  public async HashTable<string, Variant> search (
      string search_string, HashTable<string, Variant> hints) throws IOError
  {
    var result = new HashTable<string, Variant> (str_hash, str_equal);
    int num_scopes = 0;

    AsyncReadyCallback cb = (obj, res) =>
    {
      var scope = obj as ScopeProxy;
      var results = scope.search.end (res);

      HashTableIter<string, Variant> iter;
      iter = HashTableIter<string, Variant> (results);

      unowned string key;
      unowned Variant variant;
      bool models_updated = true;
      while (iter.next (out key, out variant))
      {
        // check model seqnum
        if (key == "model-seqnum")
        {
          uint64 seqnum = variant.get_uint64 ();
          unowned Dee.SerializableModel model = scope.results_model;
          if (model.get_seqnum () < seqnum)
          {
            ulong update_sig_id = 0;
            update_sig_id = (model as Dee.SharedModel).end_transaction.connect ((m, begin_seqnum, end_seqnum) =>
            {
              if (end_seqnum < seqnum) return;

              /* disconnect from within the signal handler... awesome, right? */
              SignalHandler.disconnect (m, update_sig_id);
              if (--num_scopes == 0) search.callback ();
            });
            /* don't wait if the signal connection failed */
            models_updated = update_sig_id == 0;
          }
        }
        else
        {
          // pass on the other hints
          result.insert (key, variant);
        }
      }

      if (!models_updated) return;

      if (--num_scopes == 0) search.callback ();
    };

    foreach (ScopeProxy scope in _scope_factory.scopes)
    {
      num_scopes++;
      scope.search.begin (search_string, hints, cb);
    }

    // wait for the results from all scopes - yield will wait for
    // search.callback to resume execution, and that will be called once all
    // scopes finish search (thanks to closure magic)
    if (num_scopes > 0) yield;

    result.insert ("model-seqnum", new Variant.uint64 (_results_model.get_seqnum ()));

    return result;
  }

  public async HashTable<string, Variant> global_search (
      string search_string, HashTable<string, Variant> hints) throws IOError
  {
    int num_scopes = 0;

    AsyncReadyCallback cb = (obj, result) =>
    {
      var scope = obj as ScopeProxy;
      var results = scope.global_search.end (result);
      // check model seqnum
      bool models_updated = true;
      unowned Variant? seqnum_v = results.lookup ("model-seqnum");
      if (seqnum_v != null)
      {
        uint64 seqnum = seqnum_v.get_uint64 ();
        unowned Dee.SerializableModel model = scope.global_results_model;
        if (model.get_seqnum () < seqnum)
        {
          ulong update_sig_id = 0;
          update_sig_id = (model as Dee.SharedModel).end_transaction.connect ((m, begin_seqnum, end_seqnum) =>
          {
            if (end_seqnum < seqnum) return;

            /* disconnect from within the signal handler... awesome, right? */
            SignalHandler.disconnect (m, update_sig_id);
            if (--num_scopes == 0) global_search.callback ();
          });
          /* don't wait if the signal connection failed */
          models_updated = update_sig_id == 0;
        }
      }

      if (!models_updated) return;

      if (--num_scopes == 0) global_search.callback ();
    };

    foreach (ScopeProxy scope in _scope_factory.scopes)
    {
      if (scope.search_in_global)
      {
        num_scopes++;
        scope.global_search.begin (search_string, hints, cb);
      }
    }

    // wait for the results from all scopes - yield will wait for
    // search.callback to resume execution, and that will be called once all
    // scopes finish search (thanks to closure magic)
    if (num_scopes > 0) yield;

    var result = new HashTable<string, Variant> (str_hash, str_equal);
    result.insert ("model-seqnum", new Variant.uint64 (_global_results_model.get_seqnum ()));

    return result;
  }

  public async PreviewReplyRaw preview (string uri) throws IOError
  {
    string[] tokens = uri.split(":", 2);
    ScopeProxy? scope = get_scope_for_uid (tokens[0]);
    var raw = PreviewReplyRaw ();
    raw.renderer_name = "preview-none";
    raw.properties = new HashTable<string, Variant> (null, null);

    if (scope is ScopeProxy)
      raw = yield scope.preview (tokens[1]);

    raw.uri = uri;
    return raw;
  }

  public async void update_filter (string filter_name,
                                   HashTable<string, Variant> properties) throws IOError
  {
    Trace.log_object (this, "Update file '%s'", filter_name);
  }

  protected async void update_active_sources (Gee.Set<string> updated_scope_uids)
  {
    /* grab ids of the active sources per-scope */
    foreach (var scope_uid in updated_scope_uids)
    {
      var scope = get_scope_for_uid (scope_uid);
      if (scope == null) continue;
      string[] active_sources = {};
      foreach (var filter_option in _sources.options)
      {
        if (filter_option.id.has_prefix (scope_uid) && filter_option.active)
          active_sources += filter_option.id.substring (scope_uid.length + 1);
      }

      /* no need to wait for the return */
      scope.set_active_sources.begin (active_sources);
    }
  }

  public async void set_view_type (uint view_type) throws IOError
  {
    foreach (ScopeProxy scope in _scope_factory.scopes)
    {
      scope.view_type = (ViewType) view_type;
    }

    _owner.set_active_internal (view_type != ViewType.HIDDEN);
  }

  private ScopeProxy? get_scope_for_uid (string uid)
  {
    foreach(ScopeProxy scope in _scope_factory.scopes)
    {
      if (uid == uid_for_scope (scope))
        return scope;
    }
    return null;
  }

  private string uid_for_scope (ScopeProxy scope)
  {
    return "%p".printf (scope);
  }
}

} /* namespace */
