/*
 *  Copyright (c) 2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "MemoryManager_p.h"

#include <list>
#include <map>

#include "Debug.h"
#include "StdTypes.h"
#include "Macros_p.h"

using namespace GTLCore;

#define DEFAULT_SEGMENT_SIZE 100000

// #define DEBUG_MEMORY_MANAGER

#ifdef DEBUG_MEMORY_MANAGER

#define MM_DEBUG(msg) GTL_DEBUG(msg)

#else

// #define MM_DEBUG(msg) std::cout << msg << std::endl;
#define MM_DEBUG(msg)

#endif


struct MemoryManager::Header {
  enum Status {
    USED, ///< Indicate the memory is still used
    FREED ///< Indicate the memory has been freed
  };
  Status status;
  int size;
  Header* previous;
  Header* next;
  gtl_uint8* data;
};

struct MemoryManager::Segment {
  inline Segment( int size )
  {
    size += sizeof(Header);
    start = new gtl_uint8[size];
    end = start + size * sizeof(gtl_uint8);
    nextFree = start;
    previous = 0;
  }
  ~Segment() {
    delete[] start;
  }
  inline bool contains( void* ptr ) const
  {
    return (ptr >= start and ptr < end)
        or ( ptr == end and (
          reinterpret_cast<Header*>(reinterpret_cast<gtl_uint8*>(ptr)-sizeof(Header))->size == 0 )  );
  }
  inline bool canContains(int size) const {
	return std::size_t(end - nextFree) >= std::size_t(size + sizeof(Header));
  }
  inline void* allocate(int size)
  {
    MM_DEBUG( "Begin allocate" );
    printStatus();
    GTL_ASSERT( canContains(size) );
    int sizeTotal = size + sizeof(Header);
    Header* nextFreeH = reinterpret_cast<Header*>(nextFree);
    nextFreeH->status = Header::USED;
    nextFreeH->size = size;
    nextFreeH->previous = previous;
    nextFreeH->next = 0;
    nextFreeH->data = nextFree + sizeof(Header);
    MM_DEBUG( nextFreeH << " with a previous of " << previous << " data " << (void*)nextFreeH->data << " and size of " << size );
    if(previous)
    {
      previous->next = nextFreeH;
    }
    previous = nextFreeH;
    nextFree += sizeTotal;
    printStatus();
    MM_DEBUG( "End allocate" );
    return nextFreeH->data;
  }
  inline void desallocate(void* ptr)
  {
    MM_DEBUG( "Begin desallocate" );
    printStatus();
    GTL_ASSERT(contains(ptr));
    gtl_uint8* ptr8 = reinterpret_cast<gtl_uint8*>(ptr);
    Header* currentHeader = reinterpret_cast<Header*>(ptr8-sizeof(Header));
    GTL_ASSERT(currentHeader->status == Header::USED);
    currentHeader->status = Header::FREED;
    if(currentHeader->next == 0)
    {
      while(true)
      {
        if(currentHeader->previous == 0)
        {
          MM_DEBUG("No previous, free = " << (unsigned int)(end - nextFree) );
          nextFree = reinterpret_cast<gtl_uint8*>(currentHeader);
          previous = 0;
          printStatus();
          MM_DEBUG( "End desallocate" );
          return;
        }
        Header* previousCurrentHeader = currentHeader->previous;
        if( previousCurrentHeader->status == Header::FREED )
        {
          MM_DEBUG("Has free previous = " << previousCurrentHeader << " " << (void*)previousCurrentHeader->data );
          currentHeader = previousCurrentHeader;
          MM_DEBUG( currentHeader->next );
        } else {
          MM_DEBUG("Previous (" << previousCurrentHeader << ") isn't free = " << (unsigned int)(end - nextFree) );
          previous = previousCurrentHeader;
          previous->next = 0;
          nextFree = reinterpret_cast<gtl_uint8*>(currentHeader);
          MM_DEBUG("Now free" << (unsigned int)(end - nextFree));
          printStatus();
          MM_DEBUG( "End desallocate" );
          return;
        }
      }
    }
    printStatus();
  }
  inline void printStatus()
  {
    MM_DEBUG( "### nextFree = " << (void*)nextFree << " previous = " << (void*)previous << " start = " << (void*)start << " end = " << (void*)end << " Free space = " << (unsigned int)(end - nextFree) );
    if(previous)
    {
      MM_DEBUG( "### previous->status = " << previous->status << " previous->size = " << previous->size
             << " previous->previous = " << previous->previous << " previous->next = " << previous->next
             << " previous->data = " << (void*)previous->data );
    }
  }
  gtl_uint8* nextFree;
  Header* previous;
  gtl_uint8* start;
  gtl_uint8* end;
};

#ifdef OPENGTL_HAVE_PTHREAD
#define OPENGTL_USE_PTHREAD_SPECIFIC
#endif

#ifdef OPENGTL_USE_PTHREAD_SPECIFIC
#include <pthread.h>
#else

#ifdef POSIX
#include <pthread.h>
#endif

#ifdef _WIN32
#include <Windows.h>
#endif

#include <llvm/System/Mutex.h>

#endif

struct MemoryManager::Private {
  Private() : uniqueSegment(new Segment(DEFAULT_SEGMENT_SIZE)) {}
  Segment* uniqueSegment;
  std::list<Segment*> segments;
  struct Instances {
    Instances();
#ifdef OPENGTL_USE_PTHREAD_SPECIFIC
    pthread_key_t key;
#else
    std::map<int, Private*> instances;
    llvm::sys::Mutex mutex;
#endif
    Private* currentInstance();
    int getThreadId();
  };
  static Instances* instances;
};

MemoryManager::Private::Instances::Instances()
{
#ifdef OPENGTL_USE_PTHREAD_SPECIFIC
  pthread_key_create(&key, NULL);
#endif
}

MemoryManager::Private* MemoryManager::Private::Instances::currentInstance()
{
#ifdef OPENGTL_USE_PTHREAD_SPECIFIC
  void* private_void = pthread_getspecific(key);
  if(private_void)
  {
    return static_cast<Private*>(private_void);
  }
  Private* priv = new Private;
  pthread_setspecific(key, priv);
  return priv;
#else
  mutex.acquire();
  int thread_id = getThreadId();
  Private* priv = instances[thread_id];
  if(not priv)
  {
    priv = new Private;
    instances[thread_id] = priv;
  }
  mutex.release();
  return priv;
#endif
}

int MemoryManager::Private::Instances::getThreadId()
{
#if defined(_WIN32)
  // It looks like thread id's are always multiples of 4, so...
  return GetCurrentThreadId() >> 2;
#elif defined(__BEOS__)
  return find_thread(0);
#elif defined(__linux) || defined(PTHREAD_KEYS_MAX)
  // Consecutive thread id's in Linux are 1024 apart;
  // dividing off the 1024 gives us an appropriate thread id.
  return int64_t(pthread_self()) >> 10; // (>> 10 = / 1024)
#elif defined(POSIX) || defined(__SVR4) // FIX ME??
  return (int) pthread_self();
#elif USE_SPROC
  // This hairiness has the same effect as calling getpid(),
  // but it's MUCH faster since it avoids making a system call
  // and just accesses the sproc-local data directly.
  int pid = (int) PRDA->sys_prda.prda_sys.t_pid;
  return pid;
#else
  #error "Unsupported platform"
  return 0;
#endif
}

MemoryManager::Private::Instances* MemoryManager::Private::instances = new MemoryManager::Private::Instances;

void* MemoryManager::allocate(int size)
{
  Private* priv = Private::instances->currentInstance();
  MM_DEBUG("Allocate: " << size);
  if( priv->uniqueSegment )
  {
    if( priv->uniqueSegment->canContains(size) )
    {
      void* ptr = priv->uniqueSegment->allocate(size);
      MM_DEBUG("ptr = " << ptr);
      return ptr;
    } else {
      priv->segments.push_back(priv->uniqueSegment);
      priv->uniqueSegment = 0;
    }
  } else {
    foreach(Segment* segment, priv->segments)
    {
      if( segment->canContains(size) )
      {
        void* ptr = segment->allocate(size);
        GTL_ASSERT(segment->contains(ptr));
        MM_DEBUG("ptr = " << ptr);
        return ptr;
      }
    }
  }
  MM_DEBUG("Create new segment among " << priv->segments.size() << " other segments" );
  int newSegSize = size > DEFAULT_SEGMENT_SIZE ? size : DEFAULT_SEGMENT_SIZE;
  Segment* segment = new Segment(newSegSize);
  priv->segments.push_back(segment);
  void* ptr = segment->allocate(size);
  GTL_ASSERT(segment->contains(ptr));
  MM_DEBUG("ptr = " << ptr);
  return ptr;
}

void MemoryManager::desallocate(void* ptr)
{
  Private* priv = Private::instances->currentInstance();
  MM_DEBUG("Desallocate ptr = " << ptr);
  if( priv->uniqueSegment )
  {
    GTL_ASSERT(priv->uniqueSegment->contains(ptr));
    priv->uniqueSegment->desallocate(ptr);
  } else {
    foreach(Segment* segment, priv->segments)
    {
      if( segment->contains(ptr) )
      {
        segment->desallocate(ptr);
        return;
      }
    }
    GTL_ABORT("Not allocated pointer.");
  }
}
