# -*- coding: utf-8 -*-
# Author: Manuel de la Pena <manuel@canonical.com>
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# In addition, as a special exception, the copyright holders give
# permission to link the code of portions of this program with the
# OpenSSL library under certain conditions as described in each
# individual source file, and distribute linked combinations
# including the two.
# You must obey the GNU General Public License in all respects
# for all of the code used other than OpenSSL.  If you modify
# file(s) with this exception, you may extend this exception to your
# version of the file(s), but you are not obligated to do so.  If you
# do not wish to do so, delete this exception statement from your
# version.  If you delete this exception statement from all source
# files in the program, then also delete it here.
"""Test the ui functions."""

from unittest import TestCase

from ubuntu_sso.utils.ui import (get_password_strength,
                                 is_min_required_password,
                                 is_correct_email)


class GetPasswordStrengTestCase(TestCase):
    """Test the function that returns the strengh of a password."""

    def test_too_small_password(self):
        """Test the points given to a very small password."""
        password = 'abc'
        self.assertEqual(1, get_password_strength(password))

    def test_small_password(self):
        """Test the points given to a small passwod 4 or more chars."""
        password = 'testtwe'
        self.assertEqual(0, get_password_strength(password))

    def test_eight_chars_password(self):
        """Test the points given to a normal 8 chars password."""
        password = 'abcdabcd'
        self.assertEqual(1, get_password_strength(password))

    def test_eight_chars_and_num(self):
        """Test the points given to a 8 chars password with a num."""
        password = 'abcdabc8'
        self.assertEqual(2, get_password_strength(password))

    def test_eight_chars_low_and_cap(self):
        """Test the points given to a 8 chars password with a capital."""
        password = 'abcdabcD'
        self.assertEqual(2, get_password_strength(password))

    def test_eight_chars_low_canp_num(self):
        """Test the points given to a 8 chars password with capitals & num."""
        password = 'abcdab7D'
        self.assertEqual(3, get_password_strength(password))

    def test_eiqgh_chars_and_special(self):
        """Test the points given to a 8 chars password with special chars."""
        password = 'abcdabc*'
        self.assertEqual(2, get_password_strength(password))

    def test_long_password(self):
        """Test the points goven to a long password."""
        password = 'abcdabcdabcd'
        self.assertEqual(2, get_password_strength(password))

    def test_eleven_chars_and_num(self):
        """Test the points of a loong password with a num."""
        password = 'abcdabcdabcd99'
        self.assertEqual(3, get_password_strength(password))

    def test_eleven_chars_low_cap(self):
        """Test the points of a long password with low and cap."""
        password = 'abcdabcdabcdbABCD'
        self.assertEqual(3, get_password_strength(password))

    def test_eleven_num_low_cap(self):
        """Test the points of a long password with num and diff cap."""
        password = 'ABCDabcdacbd723'
        self.assertEqual(4, get_password_strength(password))

    def test_eleven_num_special(self):
        """Test the point of a long password with a number and special char."""
        password = 'abcdabcdabcd*9'
        self.assertEqual(4, get_password_strength(password))


class IsMinRequiredPasswordTestCase(TestCase):
    """Test the fnction that returns if the password is the min required."""

    def test_no_enough_chars(self):
        """Test a password that does not have enough chars."""
        password = 'Test8'
        self.assertFalse(is_min_required_password(password))

    def test_no_uppercase(self):
        """Test a password that does not have an uppercase."""
        password = 'longenoughtobeapassword8'
        self.assertFalse(is_min_required_password(password))

    def test_no_number(self):
        """Test a password that does not have a number."""
        password = 'longenoughtobeapassworD'
        self.assertFalse(is_min_required_password(password))

    def test_correct_password(self):
        """Test a password that is correct."""
        password = 'TodasLasPaswordPasan88'
        self.assertTrue(is_min_required_password(password))


class IsCorrectEmailAddress(TestCase):
    """Test the is_correct_email funtion."""

    def test_is_correct_email_true(self):
        """Test when the email is correct."""
        email = 'manuel@canonical.com'
        self.assertTrue(is_correct_email(email))

    def test_is_correct_email_false(self):
        """Test when the email is not correct."""
        email = 'manuelcanonical.com'
        self.assertFalse(is_correct_email(email))
