# -*- coding: utf-8 -*-

# Authors: Natalia B Bidart <natalia.bidart@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the uri_hook helper."""

import logging

from PyQt4 import QtGui, QtCore
from twisted.internet import defer

from ubuntuone.controlpanel.logger import setup_logging
from ubuntuone.controlpanel.gui import (
    GENERAL_ERROR_TITLE,
    GENERAL_ERROR_MSG,
)
from ubuntuone.controlpanel.gui.qt import uri_hook, handle_errors
from ubuntuone.controlpanel.gui.qt.tests import (
    BaseTestCase,
    FakedDialog,
)


class UriHookTestCase(BaseTestCase):
    """The test suite for the uri_hook."""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(UriHookTestCase, self).setUp()
        self.patch(QtGui.QDesktopServices, 'openUrl', self._set_called)

    def test_opens_url(self):
        """The url is opened properly."""
        expected = 'foo bar'
        uri_hook(expected)
        self.assertEqual(self._called, ((QtCore.QUrl(expected),), {}))

    def test_opens_url_encoded(self):
        """The encoded url is opened properly."""
        expected = 'http://foo.bar/?next=https://one.ubuntu.com/'
        uri_hook(expected)
        self.assertEqual(self._called, ((QtCore.QUrl(expected),), {}))


class HandleErrorTestCase(BaseTestCase):
    """Test suite for the generic error handler."""

    error_handler = None
    use_logger = False
    logger = logging.getLogger()  # root logger

    @defer.inlineCallbacks
    def setUp(self):
        yield super(HandleErrorTestCase, self).setUp()
        self.called = None
        self.result = None
        self.failure = None
        self.error_handler_called = None

        if self.use_logger:
            logger = self.logger
        else:
            logger = None

        self.decorate_me = handle_errors(error_handler=self.error_handler,
                                         logger=logger)(self._decorate_me)

    @defer.inlineCallbacks
    def _decorate_me(self, *args, **kwargs):
        """Helper to test thye decorator."""
        if self.failure:
            # Raising only classes, instances or string are allowed
            # pylint: disable=E0702
            raise self.failure

        yield
        self.called = (args, kwargs)
        defer.returnValue(self.result)

    def assert_detailed_text_correct(self, expected):
        """Check that the QMessageBox detailed text is correct."""
        self.assertIn('detailed_text', FakedDialog.properties)
        actual = FakedDialog.properties['detailed_text']
        self.assertEqual(expected, actual)

    @defer.inlineCallbacks
    def test_is_decorator(self):
        """Is a decorator."""
        yield self.decorate_me()

    @defer.inlineCallbacks
    def test_params_are_passed(self):
        """Named and unnamed arguments are passed."""
        args = ({}, object(), 'foo')
        kwargs = {'1': 1, 'test': None, '0': ['a']}
        yield self.decorate_me(*args, **kwargs)

        self.assertTrue(self.called, (args, kwargs))

    @defer.inlineCallbacks
    def test_result_is_returned(self):
        """Result is returned."""
        self.result = object()
        result = yield self.decorate_me()

        self.assertEqual(self.result, result)

    def test_name_is_preserved(self):
        """The method name is not masqueraded."""
        self.assertEqual(self.decorate_me.__name__, self._decorate_me.__name__)

    @defer.inlineCallbacks
    def test_exeptions_are_handled(self):
        """Any exception is handled and logged in the root logger."""
        msg = 'This is me failing badly.'
        self.failure = Exception(msg)

        yield self.decorate_me()

        logged = self.memento.check_exception(self.failure.__class__, msg)
        recs = '\n'.join(rec.exc_text for rec in self.memento.records
                         if rec.exc_text is not None)
        self.assertTrue(logged, 'Exception must be logged, got:\n%s' % recs)

    @defer.inlineCallbacks
    def test_show_error_message(self):
        """On error, show an error message."""
        self.failure = Exception()

        yield self.decorate_me()

        args = (QtGui.QMessageBox.Critical,
                GENERAL_ERROR_TITLE, GENERAL_ERROR_MSG,
                QtGui.QMessageBox.Close)
        self.assertEqual(FakedDialog.args, args)
        self.assertEqual(FakedDialog.kwargs, {'parent': None})
        self.assertEqual(FakedDialog.properties['shown'], 1)

    @defer.inlineCallbacks
    def test_show_error_message_default_button(self):
        """On error, 'Close' is the default button."""
        self.failure = Exception()

        yield self.decorate_me()

        self.assertIn('default_button', FakedDialog.properties)
        expected = QtGui.QMessageBox.Close
        actual = FakedDialog.properties['default_button']
        self.assertEqual(expected, actual)

    @defer.inlineCallbacks
    def test_show_error_message_detailed_text(self):
        """On error, show optional detailed text."""
        self.failure = Exception()

        yield self.decorate_me()

        expected = self.failure.__class__.__name__
        self.assert_detailed_text_correct(expected)

    @defer.inlineCallbacks
    def test_show_error_message_detailed_text_if_args(self):
        """On error, show an error message."""
        msg1 = 'This is me failing badly.'
        msg2 = 'More details about what went wrong.'
        obj = object()
        self.failure = Exception(msg1, msg2, obj)

        yield self.decorate_me()

        msg = '\n'.join(map(repr, (msg1, msg2, obj)))
        msg = self.failure.__class__.__name__ + '\n' + msg
        self.assert_detailed_text_correct(msg)

    @defer.inlineCallbacks
    def test_show_error_message_detailed_text_if_mapping(self):
        """On error, show an error message."""
        msg1 = 'This is me failing badly.'
        msg2 = 'More details about what went wrong.'
        errdict = {'foo': msg1, 'bar': msg2}
        self.failure = Exception(errdict)

        yield self.decorate_me()

        msg = '\n'.join((msg1, msg2))
        self.assert_detailed_text_correct(msg)

    @defer.inlineCallbacks
    def test_no_error_message_if_no_exception(self):
        """On success, do not show an error message."""
        yield self.decorate_me()

        self.assertEqual(FakedDialog.args, None)
        self.assertEqual(FakedDialog.kwargs, None)

    @defer.inlineCallbacks
    def test_call_error_handler(self):
        """On success, do not execute error_handler."""
        yield self.decorate_me()
        self.assertFalse(self.error_handler_called)


class HandleErrorWithCustomLoggerTestCase(HandleErrorTestCase):
    """Test suite for the generic error handler."""

    use_logger = True
    logger = setup_logging('HandleErrorWithoutLoggerTestCase')  # custom logger


class HandleErrorWithHandlerTestCase(HandleErrorTestCase):
    """Test suite for the generic error handler when using a custom handler."""

    @defer.inlineCallbacks
    def error_handler(self, *a, **kw):
        """Implement an error handler."""
        self.error_handler_called = (a, kw)
        yield

    @defer.inlineCallbacks
    def test_call_error_handler(self):
        """On error, execute error_handler."""
        msg = 'This is me failing badly.'
        self.failure = Exception(msg)

        yield self.decorate_me()

        self.assertEqual(self.error_handler_called, ((), {}))
