/*****************************************************************************
 *
 * utouch-frame - Touch Frame Library
 *
 * Copyright (C) 2011-2012 Canonical Ltd.
 *
 * This library is free software: you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License version 3
 * as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranties of 
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
 * PURPOSE.  See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#include "v2/frame.h"

#include <assert.h>
#include <stdio.h>

#include "v2/touch.h"
#include "v2/window.h"

namespace utouch {
namespace frame {

UFFrame::UFFrame(const SharedWindow& window, const SharedUFFrame& prev)
    : prev_(prev),
      window_(window),
      touches_array_(),
      touches_map_() {
  for (const SharedUFTouch& prev_touch : prev_->touches_array_) {
    if (prev_touch->state() == UFTouchStateBegin) {
      touches_map_[prev_touch->id()] = touches_array_.size();
      SharedUFTouch touch(prev->CopyTouch(prev_touch->id(),
                                           UFTouchStateUpdate));
      touches_array_.push_back(touch);
    } else if (prev_touch->state() == UFTouchStateUpdate) {
      touches_map_[prev_touch->id()] = touches_array_.size();
      touches_array_.push_back(prev_touch);
    }
  }

  const Value* value;
  value = new Value(static_cast<unsigned int>(touches_map_.size()));
  InsertProperty(UFFramePropertyNumTouches, value);
  value = new Value(static_cast<unsigned int>(touches_map_.size()));
  InsertProperty(UFFramePropertyActiveTouches, value);
}

UFTouch* UFFrame::CopyTouch(UFTouchId touchid, UFTouchState new_state) const {
  auto map_it = touches_map_.find(touchid);
  if (map_it == touches_map_.end()) {
    fprintf(stderr, "Failed to copy non-existent touch\n");
    return NULL;
  }

  return new UFTouch(*touches_array_[map_it->second].get(), new_state);
}

bool UFFrame::IsTouchOwned(UFTouchId touchid) {
  auto map_it = touches_map_.find(touchid);
  if (map_it == touches_map_.end())
    return false;

  SharedUFTouch& touch = touches_array_[map_it->second];

  int owned;
  UFStatus status = touch->GetProperty(UFTouchPropertyOwned, &owned);
  if (status != UFStatusSuccess)
    /* If the window server doesn't support touch ownership, then all touches
     * are implicitly owned. */
    return true;

  return owned;
}

namespace {

void CopyStartTime(const SharedUFTouch& src,
                   const SharedUFTouch& dst) {
  uint64_t start_time = 0;
  UFStatus status = src->GetProperty(UFTouchPropertyStartTime, &start_time);
  assert(status == UFStatusSuccess);
  const Value* value = new Value(start_time);
  dst->InsertProperty(UFTouchPropertyStartTime, value);
}

} // namespace

void UFFrame::UpdateTouch(const SharedUFTouch& touch) {
  auto map_it = touches_map_.find(touch->id());

  switch (touch->state()) {
    case UFTouchStateBegin:
      if (map_it != touches_map_.end()) {
        CopyStartTime(touches_array_[map_it->second], touch);
        touches_array_[map_it->second] = touch;
      } else {
        touches_map_[touch->id()] = touches_array_.size();
        touches_array_.push_back(touch);

        const Value *value;
        value = new Value(static_cast<unsigned int>(touches_map_.size()));
        InsertProperty(UFFramePropertyNumTouches, value);
        value = new Value(static_cast<unsigned int>(touches_map_.size()));
        InsertProperty(UFFramePropertyActiveTouches, value);
      }
      break;

    case UFTouchStateUpdate: {
    case UFTouchStateEnd:
      if (map_it == touches_map_.end()) {
        fprintf(stderr, "Warning: ignoring update for unknown touch %ju\n",
                touch->id());
        break;
      }

      CopyStartTime(touches_array_[map_it->second], touch);
      touches_array_[map_it->second] = touch;
      break;
    }
  }
}

bool UFFrame::IsEnded() const {
  for (const SharedUFTouch& touch : touches_array_)
    if (touch->state() != UFTouchStateEnd)
      return false;

  return true;
}

UFStatus UFFrame::GetPreviousTouchProperty(const UFTouch* touch,
                                            UFTouchProperty property,
                                            void* value) const {
  if (!prev_)
    return UFStatusErrorInvalidTouch;

  auto it = prev_->touches_map_.find(touch->id());
  if (it == prev_->touches_map_.end())
    return UFStatusErrorInvalidTouch;

  return prev_->touches_array_[it->second]->GetProperty(property, value);
}

UFStatus UFFrame::GetPreviousTouchValue(const UFTouch* touch, UFAxisType type,
                                         float* value) const {
  if (!prev_)
    return UFStatusErrorInvalidTouch;

  auto it = prev_->touches_map_.find(touch->id());
  if (it == prev_->touches_map_.end())
    return UFStatusErrorInvalidTouch;

  return prev_->touches_array_[it->second]->GetValue(type, value);
}

UFStatus UFFrame::GetTouchByIndex(unsigned int index, ::UFTouch* touch) const {
  if (index >= touches_array_.size())
    return UFStatusErrorInvalidTouch;

  *touch = touches_array_[index].get();
  return UFStatusSuccess;
}

UFStatus UFFrame::GetTouchById(UFTouchId touch_id, ::UFTouch* touch) const {
  auto it = touches_map_.find(touch_id);
  if (it == touches_map_.end())
    return UFStatusErrorInvalidTouch;

  *touch = touches_array_[it->second].get();

  return UFStatusSuccess;
}

void UFFrame::ReleasePreviousFrame() {
  prev_.reset();
}

} // namespace frame
} // namespace utouch

extern "C" {

FRAME_PUBLIC
UFStatus frame_frame_get_property_device_(UFFrame frame,
                                          UFFrameProperty property,
                                          UFDevice *value) {
  const utouch::frame::UFFrame* ufframe =
      static_cast<const utouch::frame::UFFrame*>(frame);
  return ufframe->GetProperty(property, value);
}

FRAME_PUBLIC
UFStatus frame_frame_get_property_uint64_(UFFrame frame,
                                          UFFrameProperty property,
                                          uint64_t *value) {
  const utouch::frame::UFFrame* ufframe =
      static_cast<const utouch::frame::UFFrame*>(frame);
  return ufframe->GetProperty(property, value);
}

FRAME_PUBLIC
UFStatus frame_frame_get_property_unsigned_int_(UFFrame frame,
                                                UFFrameProperty property,
                                                unsigned int *value) {
  const utouch::frame::UFFrame* ufframe =
      static_cast<const utouch::frame::UFFrame*>(frame);
  return ufframe->GetProperty(property, value);
}

#undef frame_frame_get_property /* Override C11 generic selections macro */
FRAME_PUBLIC
UFStatus frame_frame_get_property(UFFrame frame, UFFrameProperty property,
                                  void *value) {
  const utouch::frame::UFFrame* ufframe =
      static_cast<const utouch::frame::UFFrame*>(frame);
  return ufframe->GetProperty(property, value);
}

UFStatus frame_frame_get_touch_by_index(UFFrame frame, unsigned int index,
                                        UFTouch *touch) {
  const utouch::frame::UFFrame* ufframe =
      static_cast<const utouch::frame::UFFrame*>(frame);
  return ufframe->GetTouchByIndex(index, touch);
}

UFStatus frame_frame_get_touch_by_id(UFFrame frame, UFTouchId touch_id,
                                     UFTouch *touch) {
  const utouch::frame::UFFrame* ufframe =
      static_cast<const utouch::frame::UFFrame*>(frame);
  return ufframe->GetTouchById(touch_id, touch);
}

UFStatus frame_frame_get_previous_touch_property(UFFrame frame, UFTouch touch,
                                                 UFTouchProperty property,
                                                 void *value) {
  const utouch::frame::UFFrame* ufframe =
      static_cast<const utouch::frame::UFFrame*>(frame);
  return ufframe->GetPreviousTouchProperty(
      static_cast<const utouch::frame::UFTouch*>(touch),
      property,
      value);
}

UFStatus frame_frame_get_previous_touch_value(UFFrame frame, UFTouch touch,
                                              UFAxisType type, float *value) {
  const utouch::frame::UFFrame* ufframe =
      static_cast<const utouch::frame::UFFrame*>(frame);
  return ufframe->GetPreviousTouchValue(
      static_cast<const utouch::frame::UFTouch*>(touch),
      type,
      value);
}

UFWindowId frame_frame_get_window_id(UFFrame frame) {
  UFWindowId window_id;
  const utouch::frame::UFFrame* ufframe =
      static_cast<const utouch::frame::UFFrame*>(frame);
  UFStatus status = ufframe->GetProperty(UFFramePropertyWindowId, &window_id);
  assert(status == UFStatusSuccess);
  return window_id;
}

uint32_t frame_frame_get_num_touches(UFFrame frame) {
  unsigned int num_touches;
  const utouch::frame::UFFrame* ufframe =
      static_cast<const utouch::frame::UFFrame*>(frame);
  UFStatus status = ufframe->GetProperty(UFFramePropertyNumTouches,
                                         &num_touches);
  assert(status == UFStatusSuccess);
  return num_touches;
}

UFDevice frame_frame_get_device(UFFrame frame) {
  UFDevice device;
  const utouch::frame::UFFrame* ufframe =
      static_cast<const utouch::frame::UFFrame*>(frame);
  UFStatus status = ufframe->GetProperty(UFFramePropertyDevice, &device);
  assert(status == UFStatusSuccess);
  return device;
}

} // extern "C"
