/**
 * GTest-based test suite for GEIS subscription actions.
 *
 * Copyright 2012 Canonical Ltd.
 */

#include "gtest_evemu_device.h"
#include "gtest_geis_fixture.h"
#include <sys/select.h>
#include <sys/time.h>
#include <X11/extensions/XInput2.h>


static const std::string TEST_DEVICE_PROP_FILE(
    TEST_ROOT_DIR "recordings/touchscreen_a/device.prop");
static const std::string TEST_DEVICE_EVENTS_FILE(
    TEST_ROOT_DIR "recordings/touchscreen_a/rotate90.record");

/**
 * Tests need to make sure at least one multi-touch device is available.
 */
class GeisSubscriptionTests
: public GTestGeisFixture
{
public:
  GeisSubscriptionTests()
  : evemu_device_(TEST_DEVICE_PROP_FILE)
  { }

protected:
  Testsuite::EvemuDevice evemu_device_;
};


/**
 * Regression test for lp:937021: Geis subscription touch grabs need to check
 * for failure.
 *
 * This test creates two subscriptions, both attached to the root window but
 * each on a different X client.  Since multiple subscriptions for the same
 * window but a different client are expected to fail using the XI2.2-based
 * grail back end, activating the second subscription is extected to resut in
 * a failure.
 *
 * This behaviour is very dependent on the particular back end used and internal
 * implementattion details of that back end.  It is not a unit test or a
 * reliable group or system test.
 */
TEST_F(GeisSubscriptionTests, duplicate_window_subscription)
{
  Geis geis2 = geis_new(GEIS_INIT_SYNCHRONOUS_START,
                        GEIS_INIT_NO_ATOMIC_GESTURES,
                        NULL);
  EXPECT_TRUE(geis2 != NULL) << "can not create second geis instance";

  GeisSubscription sub1 = geis_subscription_new(geis_,
                                                "subscription 1",
                                                GEIS_SUBSCRIPTION_NONE);
  EXPECT_TRUE(sub1 != NULL) << "can not create first subscription";

  GeisSubscription sub2 = geis_subscription_new(geis2,
                                                "subscription 2",
                                                GEIS_SUBSCRIPTION_NONE);
  EXPECT_TRUE(sub1 != NULL) << "can not create second subscription";

  GeisFilter filter1 = geis_filter_new(geis_, "root window 1");
  EXPECT_TRUE(filter1 != NULL) << "can not create filter 1";

  GeisStatus fs = geis_filter_add_term(filter1,
               GEIS_FILTER_CLASS,
               GEIS_CLASS_ATTRIBUTE_NAME, GEIS_FILTER_OP_EQ, GEIS_GESTURE_ROTATE,
               GEIS_GESTURE_ATTRIBUTE_TOUCHES, GEIS_FILTER_OP_GT, 1,
               NULL);
  EXPECT_EQ(fs, GEIS_STATUS_SUCCESS) << "can not add class to filter 1";
  fs = geis_subscription_add_filter(sub1, filter1);
  EXPECT_EQ(fs, GEIS_STATUS_SUCCESS) << "can not subscribe filter 1";
  geis_filter_delete(filter1);

  GeisFilter filter2 = geis_filter_new(geis2, "root window 2");
  EXPECT_TRUE(filter2 != NULL) << "can not create filter 2";

  fs = geis_filter_add_term(filter2,
               GEIS_FILTER_CLASS,
               GEIS_CLASS_ATTRIBUTE_NAME, GEIS_FILTER_OP_EQ, GEIS_GESTURE_ROTATE,
               GEIS_GESTURE_ATTRIBUTE_TOUCHES, GEIS_FILTER_OP_GT, 1,
               NULL);
  EXPECT_EQ(fs, GEIS_STATUS_SUCCESS) << "can not add class to filter 2";
  fs = geis_subscription_add_filter(sub2, filter2);
  EXPECT_EQ(fs, GEIS_STATUS_SUCCESS) << "can not subscribe filter 2";
  geis_filter_delete(filter2);

  EXPECT_EQ(GEIS_STATUS_SUCCESS, geis_subscription_activate(sub1))
              << "can not activate subscription 1";
  EXPECT_NE(GEIS_STATUS_SUCCESS, geis_subscription_activate(sub2))
              << "mistakenly activated subscription 2";

  geis_subscription_delete(sub2);
  geis_subscription_delete(sub1);
  geis_delete(geis2);
}

/**
 * Regression test 1 for lp:934207: gesture rejection
 */
TEST_F(GeisSubscriptionTests, reject_gesture)
{
  GeisGestureClass rotate_class = NULL;
  GeisGestureId    rejected_gesture_id = 0;
  bool             gesture_rejected = false;
  GeisSubscription sub = geis_subscription_new(geis_,
                                               "subscription",
                                               GEIS_SUBSCRIPTION_NONE);
  EXPECT_TRUE(sub != NULL) << "can not create subscription";
  GeisFilter filter = geis_filter_new(geis_, "root window");
  EXPECT_TRUE(filter != NULL) << "can not create filter";
  GeisStatus fs = geis_filter_add_term(filter,
               GEIS_FILTER_CLASS,
               GEIS_CLASS_ATTRIBUTE_NAME, GEIS_FILTER_OP_EQ, GEIS_GESTURE_ROTATE,
               GEIS_GESTURE_ATTRIBUTE_TOUCHES, GEIS_FILTER_OP_GT, 1,
               NULL);
  EXPECT_EQ(fs, GEIS_STATUS_SUCCESS) << "can not add class to filter";
  fs = geis_subscription_add_filter(sub, filter);
  EXPECT_EQ(fs, GEIS_STATUS_SUCCESS) << "can not subscribe filter";
  geis_filter_delete(filter);
  EXPECT_EQ(GEIS_STATUS_SUCCESS, geis_subscription_activate(sub))
              << "can not activate subscription";

  GeisStatus dispatch_status= geis_dispatch_events(geis_);
  while (dispatch_status == GEIS_STATUS_CONTINUE
      || dispatch_status == GEIS_STATUS_SUCCESS)
  {
    GeisEvent  event;
    GeisStatus event_status = geis_next_event(geis_, &event);
    while (event_status == GEIS_STATUS_CONTINUE
        || event_status == GEIS_STATUS_SUCCESS)
    {
      switch (geis_event_type(event))
      {
        case GEIS_EVENT_INIT_COMPLETE:
          evemu_device_.play(TEST_DEVICE_EVENTS_FILE);
          break;

        case GEIS_EVENT_CLASS_AVAILABLE:
        {
          GeisAttr attr = geis_event_attr_by_name(event,
                                                  GEIS_EVENT_ATTRIBUTE_CLASS);
          EXPECT_TRUE(attr != NULL) << "event is missing class attr";
          GeisGestureClass gesture_class = (GeisGestureClass)geis_attr_value_to_pointer(attr);
          for (GeisSize i = 0; i < geis_gesture_class_attr_count(gesture_class); ++i)
          {
            GeisAttr attr2 = geis_gesture_class_attr(gesture_class, i);
            if (0 == strcmp(geis_attr_name(attr2), GEIS_CLASS_ATTRIBUTE_NAME))
            {
              if (0 == strcmp(geis_attr_value_to_string(attr2),
                              GEIS_GESTURE_ROTATE))
              {
                rotate_class = gesture_class;
              }
            }
          }
          break;
        }

        case GEIS_EVENT_GESTURE_BEGIN:
        case GEIS_EVENT_GESTURE_UPDATE:
        {
          GeisAttr attr = geis_event_attr_by_name(event,
                                                  GEIS_EVENT_ATTRIBUTE_GROUPSET);
          EXPECT_TRUE(attr != NULL) << "event is missing groupset attr";
          GeisGroupSet groupset = (GeisGroupSet)geis_attr_value_to_pointer(attr);
          EXPECT_TRUE(groupset != NULL) << "event is missing groupset";
          for (GeisSize i = 0; i < geis_groupset_group_count(groupset); ++i)
          {
            GeisGroup group = geis_groupset_group(groupset, i);
            EXPECT_TRUE(group != NULL) << "group " << i
                                       << " not found in groupset";

            for (GeisSize j = 0; j < geis_group_frame_count(group); ++j)
            {
              GeisFrame frame = geis_group_frame(group, j);
              EXPECT_TRUE(frame != NULL) << "frame " << j
                                         << " not found in group";

              if (geis_frame_is_class(frame, rotate_class))
              {
                if (!gesture_rejected)
                {
                  rejected_gesture_id = geis_frame_id(frame);
                  GeisStatus gs = geis_gesture_reject(geis_,
                                                      group,
                                                      rejected_gesture_id);
                  EXPECT_EQ(gs, GEIS_STATUS_SUCCESS) << "rejection failed";
                  gesture_rejected = true;
                }
                else
                {
                  EXPECT_NE(geis_frame_id(frame), rejected_gesture_id)
                            << "gesture events after gesture rejected";
                }
              }
            }
          }
          break;
        }

        default:
          break;
      }

      geis_event_delete(event);

      event_status = geis_next_event(geis_, &event);
    }

    fd_set read_fds;
    FD_ZERO(&read_fds);
    FD_SET(geis_fd(), &read_fds);
    timeval tmo = { 5, 0 };
    int sstat = select(geis_fd() + 1, &read_fds, NULL, NULL, &tmo);
    EXPECT_GT(sstat, -1) << "error in select";
    if (sstat == 0)
      break;
    dispatch_status = geis_dispatch_events(geis_);
  }

  EXPECT_EQ(gesture_rejected, true) << "gesture was never rejected";
  geis_subscription_delete(sub);
}


/**
 * Regression test 2 for lp:934207: gesture acceptance
 */
TEST_F(GeisSubscriptionTests, accept_gesture)
{
  GeisGestureClass rotate_class = NULL;
  bool gesture_accepted = false;
  bool gesture_ended = false;
  GeisSubscription sub = geis_subscription_new(geis_,
                                               "subscription",
                                               GEIS_SUBSCRIPTION_NONE);
  EXPECT_TRUE(sub != NULL) << "can not create subscription";
  GeisFilter filter = geis_filter_new(geis_, "root window");
  EXPECT_TRUE(filter != NULL) << "can not create filter";
  GeisStatus fs = geis_filter_add_term(filter,
               GEIS_FILTER_CLASS,
               GEIS_CLASS_ATTRIBUTE_NAME, GEIS_FILTER_OP_EQ, GEIS_GESTURE_ROTATE,
               GEIS_GESTURE_ATTRIBUTE_TOUCHES, GEIS_FILTER_OP_GT, 1,
               NULL);
  EXPECT_EQ(fs, GEIS_STATUS_SUCCESS) << "can not add class to filter";
  fs = geis_subscription_add_filter(sub, filter);
  EXPECT_EQ(fs, GEIS_STATUS_SUCCESS) << "can not subscribe filter";
  geis_filter_delete(filter);
  EXPECT_EQ(GEIS_STATUS_SUCCESS, geis_subscription_activate(sub))
              << "can not activate subscription";

  GeisStatus dispatch_status= geis_dispatch_events(geis_);
  while (dispatch_status == GEIS_STATUS_CONTINUE
      || dispatch_status == GEIS_STATUS_SUCCESS)
  {
    GeisEvent  event;
    GeisStatus event_status = geis_next_event(geis_, &event);
    while (event_status == GEIS_STATUS_CONTINUE
        || event_status == GEIS_STATUS_SUCCESS)
    {
      switch (geis_event_type(event))
      {
        case GEIS_EVENT_INIT_COMPLETE:
          evemu_device_.play(TEST_DEVICE_EVENTS_FILE);
          break;

        case GEIS_EVENT_CLASS_AVAILABLE:
        {
          GeisAttr attr = geis_event_attr_by_name(event,
                                                  GEIS_EVENT_ATTRIBUTE_CLASS);
          EXPECT_TRUE(attr != NULL) << "event is missing class attr";
          GeisGestureClass gesture_class = (GeisGestureClass)geis_attr_value_to_pointer(attr);
          for (GeisSize i = 0; i < geis_gesture_class_attr_count(gesture_class); ++i)
          {
            GeisAttr attr2 = geis_gesture_class_attr(gesture_class, i);
            if (0 == strcmp(geis_attr_name(attr2), GEIS_CLASS_ATTRIBUTE_NAME))
            {
              if (0 == strcmp(geis_attr_value_to_string(attr2),
                              GEIS_GESTURE_ROTATE))
              {
                rotate_class = gesture_class;
              }
            }
          }
          break;
        }

        case GEIS_EVENT_GESTURE_BEGIN:
        case GEIS_EVENT_GESTURE_UPDATE:
        {
          GeisAttr attr = geis_event_attr_by_name(event,
                                                  GEIS_EVENT_ATTRIBUTE_GROUPSET);
          EXPECT_TRUE(attr != NULL) << "event is missing groupset attr";
          GeisGroupSet groupset = (GeisGroupSet)geis_attr_value_to_pointer(attr);
          EXPECT_TRUE(groupset != NULL) << "event is missing groupset";
          for (GeisSize i = 0; i < geis_groupset_group_count(groupset); ++i)
          {
            GeisGroup group = geis_groupset_group(groupset, i);
            EXPECT_TRUE(group != NULL) << "group " << i
                                       << " not found in groupset";

            for (GeisSize j = 0; j < geis_group_frame_count(group); ++j)
            {
              GeisFrame frame = geis_group_frame(group, j);
              EXPECT_TRUE(frame != NULL) << "frame " << j
                                         << " not found in group";

              if (!gesture_accepted && geis_frame_is_class(frame, rotate_class))
              {
                GeisStatus gs = geis_gesture_accept(geis_,
                                                    group,
                                                    geis_frame_id(frame));
                EXPECT_EQ(gs, GEIS_STATUS_SUCCESS) << "aceptance failed";
                gesture_accepted = true;
              }
            }
          }
          break;
        }

        case GEIS_EVENT_GESTURE_END:
          gesture_ended = true;
          goto end_of_outer_loop;
          break;

        default:
          break;
      }

      geis_event_delete(event);

      event_status = geis_next_event(geis_, &event);
    }

    fd_set read_fds;
    FD_ZERO(&read_fds);
    FD_SET(geis_fd(), &read_fds);
    timeval tmo = { 5, 0 };
    int sstat = select(geis_fd() + 1, &read_fds, NULL, NULL, &tmo);
    EXPECT_GT(sstat, -1) << "error in select";
    if (sstat == 0)
      break;
    dispatch_status = geis_dispatch_events(geis_);
  }
end_of_outer_loop:

  EXPECT_EQ(gesture_accepted, true) << "gesture was never accepted";
  EXPECT_EQ(gesture_ended, true) << "gesture did not end";
  geis_subscription_delete(sub);
}

/**
 * Regression test for lp:968736: Windows are sometimes not ungrabbed on
 * subscription deactivation.
 *
 * When a subscription is deactivated, the touch grab on the window must be
 * ungrabbed. This test deactivates a subscription and then tries to grab the
 * window through a different X connection.
 */
TEST_F(GeisSubscriptionTests, WindowUngrab)
{
  GeisSubscription sub = geis_subscription_new(geis_,
                                               "subscription",
                                               GEIS_SUBSCRIPTION_NONE);
  EXPECT_TRUE(sub != NULL) << "can not create first subscription";

  GeisFilter filter = geis_filter_new(geis_, "root window");
  EXPECT_TRUE(filter != NULL) << "can not create filter";

  GeisStatus fs = geis_filter_add_term(filter, GEIS_FILTER_CLASS,
                                       GEIS_CLASS_ATTRIBUTE_NAME,
                                       GEIS_FILTER_OP_EQ, GEIS_GESTURE_ROTATE,
                                       GEIS_GESTURE_ATTRIBUTE_TOUCHES,
                                       GEIS_FILTER_OP_GT, 1, NULL);
  EXPECT_EQ(fs, GEIS_STATUS_SUCCESS) << "can not add class to filter";
  fs = geis_subscription_add_filter(sub, filter);
  EXPECT_EQ(fs, GEIS_STATUS_SUCCESS) << "can not subscribe filter";
  geis_filter_delete(filter);

  EXPECT_EQ(GEIS_STATUS_SUCCESS, geis_subscription_activate(sub))
              << "can not activate subscription";

  geis_subscription_deactivate(sub);

  ::Display* display2 = XOpenDisplay(NULL);
  
  XIEventMask mask;
  mask.deviceid = XIAllMasterDevices;
  mask.mask_len = XIMaskLen(XI_LASTEVENT);
  mask.mask = reinterpret_cast<unsigned char*>(calloc(mask.mask_len,
                                                      sizeof(char)));

  XISetMask(mask.mask, XI_TouchBegin);
  XISetMask(mask.mask, XI_TouchUpdate);
  XISetMask(mask.mask, XI_TouchEnd);
  XISetMask(mask.mask, XI_TouchOwnership);
  XISetMask(mask.mask, XI_HierarchyChanged);

  XIGrabModifiers mods = { XIAnyModifier, 0 };

  Window win = DefaultRootWindow(display2);

  XIGrabTouchBegin(display2, XIAllMasterDevices, win, 0, &mask, 1, &mods);

  free(mask.mask);

  EXPECT_EQ(XIGrabSuccess, mods.status);

  geis_subscription_delete(sub);
}
