/*****************************************************************************
 *
 * grail - Gesture Recognition And Instantiation Library
 *
 * Copyright (C) 2010-2012 Canonical Ltd.
 *
 * This library is free software: you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License version 3
 * as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranties of 
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
 * PURPOSE.  See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#include "grail-recognizer.h"
#include <math.h>
#include <stdio.h>

static const int getype[DIM_TOUCH + 1] = {
	0,
	0,
	GRAIL_TYPE_PINCH2,
	GRAIL_TYPE_PINCH3,
	GRAIL_TYPE_PINCH4,
	GRAIL_TYPE_PINCH5,
};

static const int fm_mask = 0x04;

static void set_props(const struct gesture_inserter *gin,
		      struct combo_model *s,
		      const struct move_model *m,
		      const struct utouch_frame *frame)
{
	s->prop[GRAIL_PROP_PINCH_DR] = m->fm[FM_R].action_delta;
	s->prop[GRAIL_PROP_PINCH_VR] = m->fm[FM_R].velocity;
	s->prop[GRAIL_PROP_PINCH_R] = m->fm[FM_R].value;
	s->prop[GRAIL_PROP_PINCH_X] = m->fm[FM_X].value;
	s->prop[GRAIL_PROP_PINCH_Y] = m->fm[FM_Y].value;
	s->nprop = 5;
	s->nprop += gin_add_contact_props(gin, s->prop + s->nprop, frame);
}

int gru_pinch(struct grail *ge,
	      const struct utouch_frame *frame)
{
	struct gesture_recognizer *gru = ge->gru;
	struct combo_model *state = &gru->pinch;
	struct move_model *move = &gru->move;
	int mask = state->active ? (move->active & fm_mask) : fm_mask;
	if (!move->multi && !move->single) {
		if (state->active) {
			gru_end(ge, state->gid, move,
				state->prop, state->nprop);
			state->active = 0;
		}
		return 0;
	}
	if ((move->timeout & fm_mask) == fm_mask) {
		if (state->active) {
			gin_gid_timeout(ge, state->gid);
		}
	}
	if (!(move->tickle & mask))
		return 0;
	if (!state->active) {
		int type = getype[move->ntouch];
		if (!type)
			return 0;
		state->gid = gin_gid_begin(ge, type, PRIO_GESTURE, frame);
		state->active = 1;
	}
	if (!(move->active & fm_mask))
		return 0;
	set_props(ge->gin, state, move, frame);
	gru_event(ge, state->gid, move, state->prop, state->nprop);
	return 1;
}

int gru_winpinch(struct grail *ge,
		 const struct utouch_frame *frame)
{
	struct gesture_recognizer *gru = ge->gru;
	struct combo_model *state = &gru->winpinch;
	struct move_model *move = &gru->move;
	int mask = state->active ? (move->active & fm_mask) : fm_mask;
	if (!move->multi) {
		if (state->active && out_of_bounds(state, move)) {
			gru_end(ge, state->gid, move,
				state->prop, state->nprop);
			state->active = 0;
		}
		return 0;
	}
	if ((move->timeout & fm_mask) == fm_mask) {
		if (state->active) {
			gin_gid_timeout(ge, state->gid);
		}
	}
	if (!(move->tickle & mask))
		return 0;
	if (!state->active) {
		if (move->ntouch == 4) {
			state->gid = gin_gid_begin(ge, GRAIL_TYPE_MPINCH,
						   PRIO_META, frame);
			state->mintouch = 2;
			state->maxtouch = 4;
			state->active = 1;
		} else if (move->ntouch == 3) {
			state->gid = gin_gid_begin(ge, GRAIL_TYPE_EPINCH,
						   PRIO_ENV, frame);
			state->mintouch = 2;
			state->maxtouch = 3;
			state->active = 1;
		} else {
			return 0;
		}
	}
	if (!(move->active & fm_mask))
		return 0;
	set_props(ge->gin, state, move, frame);
	gru_event(ge, state->gid, move, state->prop, state->nprop);
	return 1;
}
