/* ExpressionBuilder.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.conjecture.engine.apengine;

import org.grinvin.expr.Expression;
import org.grinvin.invariants.InvariantExpressionNode;

/**
 * Builds an {@link Expression} from a {@link LabeledBinaryTree}.
 * <p>
 * <b>Note:</b> this class is not thread safe.
 */
public class ExpressionBuilder {
    
    //
    private LabeledBinaryTree tree;
    
    /**
     * Set tree to be translated to an expression.
     */
    private void setTree(LabeledBinaryTree tree) {
        this.tree = tree;
    }
    
    //
    private InvariantExpressionNode[] invariants;
    
    /**
     * Set the invariants to be used in translation.
     */
    private void setInvariants (InvariantExpressionNode[] invariants) {
        this.invariants = invariants;
    }
    
    /**
     * Build an expression for the given tree.
     */
    public Expression buildExpression(LabeledBinaryTree tree, InvariantExpressionNode[] invariants) {
        setTree (tree);
        setInvariants (invariants);
        return buildExpression(0);
    }
    
    //
    private Expression buildExpression(int parent) {
        Operators.Operator operator = tree.operators[parent];
        assert operator != null;
        
        if (operator instanceof Operators.Invariant) {
            Operators.Invariant inv = (Operators.Invariant)operator;
            return invariants[inv.number];
        } else {
            
            int numberOfChildren = tree.childCount(parent);
            Expression[] sub = new Expression[numberOfChildren];
            int[] children = tree.children(parent);
            for (int i=0; i < numberOfChildren; i++)
                sub[i] = buildExpression(children[i]);
            return operator.toExpression(sub);
        }
    }
}
