/** \file src/views.cc */
/*
 * This file is part of assoGiate,
 * an editor of the file types database for GNOME.
 *
 * Copyright (C) 2007 Kevin Daughtridge <kevin@kdau.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "private.hh"
#include "views.hh"

#include <gtkmm/icontheme.h>

/******************************************************************************/
namespace assoGiate {
/******************************************************************************/

/******************************************************************************/
/* class CategoriesColumns                                                    */
/******************************************************************************/

CategoriesColumns&
CategoriesColumns::get()
{
	static CategoriesColumns the;
	return the;
}

CategoriesColumns::CategoriesColumns()
:	filter(), id(), icon(), name()
{
	add(filter);
	add(id);
	add(icon);
	add(name);
}

/******************************************************************************/
/* class CategoriesStore                                                      */
/******************************************************************************/

RefPtr<CategoriesStore>
CategoriesStore::get_filter()
{
	static RefPtr<CategoriesStore> the(new CategoriesStore(true));
	return the;
}

RefPtr<CategoriesStore>
CategoriesStore::get_selector()
{
	static RefPtr<CategoriesStore> the(new CategoriesStore(false));
	return the;
}

CategoriesStore::CategoriesStore(bool filter)
:	Gtk::ListStore(CategoriesColumns::get()),
	m_cols(CategoriesColumns::get()), m_is_filter(filter)
{
	if (m_is_filter) {
		append(CategoriesColumns::ALL, "folder", _("All types"));
		append(CategoriesColumns::OVERRIDE, "folder-new", _("Modified types"));
		append(CategoriesColumns::SEARCH, "folder-saved-search",
			_("Search results"));
		append(CategoriesColumns::SEPARATOR, ustring(), ustring());
	}
	append("application", "x-office-document", _("Multipurpose files"));
	append("audio", "audio-x-generic", _("Sound files"));
	append("image", "image-x-generic", _("Images"));
	append("inode", "emblem-system", _("Non-file objects"));
	append("message", "internet-mail", _("Messages"));
	append("model", "applications-development", _("3D models"));
	append("multipart", "package-x-generic", _("Multi-part files"));
	append("text", "text-x-generic", _("Text and source code"));
	append("video", "video-x-generic", _("Video files"));
}

Gtk::TreeIter
CategoriesStore::get_filter_item(CategoriesColumns::Filter item)
{
	RefPtr<CategoriesStore> filter = get_filter();
	FOREACH(Gtk::TreeNodeChildren, filter->children(), child)
		if (child->get_value(filter->m_cols.filter) == item)
			return child;
	
	return Gtk::TreeIter();
}

bool
CategoriesStore::row_separator_func(const RefPtr<Gtk::TreeModel>&,
	const Gtk::TreeIter& iter)
{
	return iter->get_value(CategoriesColumns::get().filter) ==
		CategoriesColumns::SEPARATOR;
}

CategoriesStore::iterator
CategoriesStore::append(const ustring& id, const ustring& icon,
	const ustring& name)
{
	Gtk::TreeIter i = Gtk::ListStore::append();
	i->set_value(m_cols.filter, CategoriesColumns::SINGLE);
	i->set_value(m_cols.id, id);
	try {
		i->set_value(m_cols.icon, Gtk::IconTheme::get_default()->
			load_icon(icon, m_is_filter ? 22 : 16, Gtk::IconLookupFlags(0)));
	} catch (...) {}
	i->set_value(m_cols.name, name);
	return i;
}

CategoriesStore::iterator
CategoriesStore::append(CategoriesColumns::Filter filter, const ustring& icon,
	const ustring& name)
{
	Gtk::TreeIter i = Gtk::ListStore::append();
	i->set_value(m_cols.filter, filter);
	i->set_value(m_cols.id, ustring());
	try {
		i->set_value(m_cols.icon, Gtk::IconTheme::get_default()->
			load_icon(icon, m_is_filter ? 22 : 16, Gtk::IconLookupFlags(0)));
	} catch (...) {}
	i->set_value(m_cols.name, name);
	return i;
}

/******************************************************************************/
/* class TypesColumns                                                         */
/******************************************************************************/

TypesColumns::TypesColumns()
:	obj(), alias(), style(), category(), name(), description()
{
	add(obj);
	add(alias); add(style);
	add(category); add(name);
	add(description);
}

TypesColumns&
TypesColumns::get()
{
	static TypesColumns the;
	return the;
}

/******************************************************************************/
/* class TypesStore                                                           */
/******************************************************************************/

RefPtr<TypesStore>
TypesStore::create(const RefPtr<MimeDatabase>& database)
{	return RefPtr<TypesStore>(new TypesStore(database)); }

TypesStore::TypesStore(const RefPtr<MimeDatabase>& database)
:	Gtk::ListStore(TypesColumns::get()),
	m_cols(TypesColumns::get()), m_database(database),
	m_filter(CategoriesColumns::ALL), m_param()
{
	set_default_sort_func
		(sigc::mem_fun(*this, &TypesStore::default_compare));
	reload();
	m_database->signal_changed().connect
		(sigc::mem_fun(*this, &TypesStore::reload));
}

void
TypesStore::show_category(const ustring& category)
{
	m_filter = CategoriesColumns::SINGLE;
	m_param = category;
	reload();
}

void
TypesStore::show_all()
{
	m_filter = CategoriesColumns::ALL;
	m_param.clear();
	reload();
}

void
TypesStore::show_override()
{
	m_filter = CategoriesColumns::OVERRIDE;
	m_param.clear();
	reload();
}
void
TypesStore::show_results(const ustring& query)
{
	m_filter = CategoriesColumns::SEARCH;
	m_param = query.casefold();
	reload();
}

void
TypesStore::reload()
{
	int column_id; Gtk::SortType order;
	get_sort_column_id(column_id, order);
	set_sort_column(DEFAULT_UNSORTED_COLUMN_ID, Gtk::SORT_ASCENDING);

	clear();
	FOREACH(MimeTypeMap, m_database->get_types(), i) {
		if (i->second.second) { /* alias */
			std::pair<ustring, ustring> alias;
			try {
				alias = MimeType::split_name(i->first);
			} catch (...) { continue; }

			load_type(*i->second.first, true, alias.first, alias.second,
				compose::ucompose(_("alias for %1"),
					i->second.first->get_full_name()));
		} else /* not alias */
			load_type(*i->second.first, false, i->second.first->m_type,
				i->second.first->m_subtype,
				i->second.first->m_description.get_translated());
	}
	
	set_sort_column(column_id, order);
}

void
TypesStore::load_type(MimeType& type, bool alias, const ustring& category,
	const ustring& name, const ustring& description)
{
	switch (m_filter) {
	case CategoriesColumns::SINGLE:
		if (m_param != category) return;
		break;
	case CategoriesColumns::ALL:
		break;
	case CategoriesColumns::OVERRIDE:
		if (alias ||
			(type.get_locations() & m_database->get_target()) == NO_LOCATION)
			return;
		break;
	case CategoriesColumns::SEARCH: {
		if (m_param.empty()) return;
		ustring haystack = category + "/" + name + ":" +
			(alias ? "=" + type.get_full_name() : description);
		if (haystack.casefold().find(m_param) != ustring::npos) break;
		return;
	} default: /* ??? */
		return;
	}

	Gtk::TreeIter iter = append();
	iter->set_value(m_cols.obj, &type);
	iter->set_value(m_cols.alias, alias);
	iter->set_value(m_cols.style, alias
		? Pango::STYLE_ITALIC : Pango::STYLE_NORMAL);
	iter->set_value(m_cols.category, category);
	iter->set_value(m_cols.name, name);
	iter->set_value(m_cols.description, description);
}

int
TypesStore::default_compare(const Gtk::TreeIter& lhs,
	const Gtk::TreeIter& rhs) const
{
	int res = lhs->get_value(m_cols.category).compare
		(rhs->get_value(m_cols.category));
	if (res == 0) res = lhs->get_value(m_cols.name).compare
		(rhs->get_value(m_cols.name));
	return res;
}

/******************************************************************************/
/* class TypesView                                                         */
/******************************************************************************/

TypesView::TypesView()
:	Gtk::TreeView(), m_cols(TypesColumns::get()), s_right_click()
{
	set_headers_clickable(true);
	set_search_equal_func(sigc::mem_fun
		(*this, &TypesView::on_search_equal));

	Gtk::TreeViewColumn *col = new Gtk::TreeViewColumn(_("Category"),
		m_cols.category);
	if cast(col->get_first_cell_renderer(), Gtk::CellRendererText, text)
		col->add_attribute(text->property_style(), m_cols.style);
	append_column(*Gtk::manage(col));

	col = new Gtk::TreeViewColumn(_("Name"), m_cols.name);
	if cast(col->get_first_cell_renderer(), Gtk::CellRendererText, text)
		col->add_attribute(text->property_style(), m_cols.style);
	append_column(*Gtk::manage(col));
	col->set_max_width(200);
	col->set_sizing(Gtk::TREE_VIEW_COLUMN_AUTOSIZE);
	col->set_sort_column(m_cols.name);

	col = new Gtk::TreeViewColumn(_("Description"), m_cols.description);
	if cast(col->get_first_cell_renderer(), Gtk::CellRendererText, text)
		col->add_attribute(text->property_style(), m_cols.style);
	append_column(*Gtk::manage(col));
	col->set_max_width(200);
	col->set_sizing(Gtk::TREE_VIEW_COLUMN_AUTOSIZE);
	col->set_sort_column(m_cols.description);
}

sigc::signal<void, guint32>
TypesView::signal_right_click()
{	return s_right_click; }

bool
TypesView::on_button_press_event(GdkEventButton* e)
{
	bool result = Gtk::TreeView::on_button_press_event(e);
	if (e && e->type == GDK_BUTTON_PRESS && e->button == 3)
		s_right_click.emit(e->time);
	return result;
}

bool
TypesView::on_search_equal(const RefPtr<Gtk::TreeModel>&, int,
	const ustring& key, const Gtk::TreeIter& iter) const
{
	MimeType *type = iter->get_value(TypesColumns::get().obj);
	return !(Glib::str_has_prefix(type->get_full_name(), key) ||
		Glib::str_has_prefix(type->m_subtype, key));
}

/******************************************************************************/
/* class MagicTypesColumns                                                    */
/******************************************************************************/

MagicTypesColumns&
MagicTypesColumns::get()
{
	static MagicTypesColumns the;
	return the;
}

MagicTypesColumns::MagicTypesColumns()
:	type(), name()
{
	add(type);
	add(name);
}

/******************************************************************************/
/* class MagicTypesStore                                                      */
/******************************************************************************/

RefPtr<MagicTypesStore>
MagicTypesStore::get()
{
	static RefPtr<MagicTypesStore> the(new MagicTypesStore());
	return the;
}

MagicTypesStore::MagicTypesStore()
:	Gtk::ListStore(MagicTypesColumns::get()), m_cols(MagicTypesColumns::get())
{
	append(MagicMatch::STRING, _("String"));
	append(MagicMatch::HOST16, _("32-bit host format"));
	append(MagicMatch::HOST32, _("16-bit host format"));
	append(MagicMatch::BIG16, _("16-bit big-endian"));
	append(MagicMatch::BIG32, _("32-bit big-endian"));
	append(MagicMatch::LITTLE16, _("16-bit little-endian"));
	append(MagicMatch::LITTLE32, _("32-bit little-endian"));
	append(MagicMatch::BYTE, _("Single byte"));
}

MagicTypesStore::iterator
MagicTypesStore::append(MagicMatch::Type type, const ustring& name)
{
	Gtk::TreeIter i = Gtk::ListStore::append();
	i->set_value(m_cols.type, type);
	i->set_value(m_cols.name, name);
	return i;
}

} /* namespace assoGiate */
