#include "audio_stream_mixier.hpp"
#include <algorithm>
#include <cstring>
#include <utility>

AudioStreamMixier::AudioStreamMixier(uint32_t rate, uint32_t duration, uint32_t intrRate, QObject* parent) :
	QIODevice(parent),
	rate_(rate),
	duration_(duration),
	intrRate_(intrRate),
	intrCountRest_(0),
	initializeRestCount_(-1)
{
	updateBufferSampleSize();
	updateIntrruptCount();
}

AudioStreamMixier::~AudioStreamMixier()
{
	if (hasRun()) stop();
}

void AudioStreamMixier::start()
{
	initializeRestCount_ = -1;
	open(QIODevice::ReadOnly);
}

void AudioStreamMixier::stop()
{
	close();
}

bool AudioStreamMixier::hasRun()
{
	return isOpen();
}

void AudioStreamMixier::setRate(uint32_t rate)
{
	rate_ = rate;
	updateBufferSampleSize();
	updateIntrruptCount();
}

void AudioStreamMixier::setDuration(uint32_t duration)
{
	duration_ = duration;
	updateBufferSampleSize();
}

void AudioStreamMixier::setInterruption(uint32_t rate)
{
	intrRate_ = rate;
	updateIntrruptCount();
}

void AudioStreamMixier::updateBufferSampleSize()
{
	bufferSampleSize_ = rate_ * duration_ / 1000;
}

void AudioStreamMixier::updateIntrruptCount()
{
	intrCount_ = rate_ / intrRate_;
}

qint64 AudioStreamMixier::readData(char* data, qint64 maxlen)
{
	qint64 generatedCount;
	switch (initializeRestCount_) {
	case -1:	// Clean device buffer in first read
		initializeRestCount_ = maxlen % 4;	// int16 * 2 (stereo)
		std::memset(data, 0, static_cast<size_t>(maxlen));
		return maxlen;
	case 0:		// Fill appropriate sample counts
		generatedCount = std::min(bufferSampleSize_, (maxlen >> 2));
		break;
	default:	// Clean device buffer to adjust L-R
		std::memset(data, 0, static_cast<size_t>(initializeRestCount_));
		return std::exchange(initializeRestCount_, 0);
	}
	size_t requiredCount = static_cast<size_t>(generatedCount);
	int16_t* destPtr = reinterpret_cast<int16_t*>(data);

	size_t count;
	while (requiredCount) {
		if (!intrCountRest_) {	// Interruption
			intrCountRest_ = intrCount_;    // Set counts to next interruption
			emit streamInterrupted();
		}

		count = std::min(intrCountRest_, requiredCount);
		requiredCount -= count;
		intrCountRest_ -= count;

		emit bufferPrepared(destPtr, count);

		destPtr += (count << 1);	// Move head
	}

	return generatedCount << 2; // Return generated bytes count
}

qint64 AudioStreamMixier::writeData(const char *data, qint64 len)   // No use
{
	Q_UNUSED(data);
	Q_UNUSED(len);
	return 0;
}
