# Copyright (C) 2006 Canonical Ltd
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""A simple LRU Cache"""

from collections import deque
import gc


class LRUCache(object):
    """A class which manages a cache of entries, removing unused ones."""

    def __init__(self, max_cache=100):
        self._max_cache = max_cache

        self._cache = {}
        self._cleanup = {}
        self._queue = deque() # Track when things are accessed
        self._refcount = {} # number of entries in self._queue for each key

    def __contains__(self, key):
        return key in self._cache

    def __getitem__(self, key):
        val = self._cache[key]
        self._record_access(key)
        return val

    def __len__(self):
        return len(self._cache)

    def add(self, key, value, cleanup=None):
        """Add a new value to the cache.

        Also, if the entry is ever removed from the queue, call cleanup.
        Passing it the key and value being removed.

        :param key: The key to store it under
        :param value: The object to store
        :param cleanup: None or a function taking (key, value) to indicate
                        'value' should be cleaned up.
        """
        if key in self._cache:
            self._remove(key)
        self._cache[key] = value
        self._cleanup[key] = cleanup
        self._record_access(key)

    def __setitem__(self, key, value):
        """Add a value to the cache, there will be no cleanup function."""
        self.add(key, value, cleanup=None)

    def _record_access(self, key):
        """Record that key was accessed."""
        self._queue.append(key)
        # Can't use setdefault because you can't += 1 the result
        self._refcount[key] = self._refcount.get(key, 0) + 1

        # Make sure the cache is shrunk to the correct size
        while len(self._cache) > self._max_cache:
            self._remove_lru()

        # If our access queue is too large, clean it up too
        if len(self._queue) > 4*self._max_cache:
            self._compact_queue()

    def _compact_queue(self):
        """Compact the queue, leaving things in sorted order."""
        for i in xrange(len(self._queue)):
            k = self._queue.popleft()
            if self._refcount[k] == 1:
                self._queue.append(k)
            else:
                self._refcount[k] -= 1
        # All entries should be of the same size. There should be one entry in
        # queue for each entry in cache, and all refcounts should == 1
        assert (len(self._queue) == len(self._cache) ==
                len(self._refcount) == sum(self._refcount.itervalues()))
        gc.collect()

    def _remove(self, key):
        """Remove an entry, making sure to maintain the invariants."""
        cleanup = self._cleanup.pop(key)
        val = self._cache.pop(key)
        if cleanup is not None:
            cleanup(key, val)
        del cleanup
        del val

    def _remove_lru(self):
        """Remove one entry from the lru, and handle consequences.

        If there are no more references to the lru, then this entry should be
        removed from the cache.
        """
        key = self._queue.popleft()
        self._refcount[key] -= 1
        if not self._refcount[key]:
            del self._refcount[key]
            self._remove(key)

    def clear(self):
        """Clear out all of the cache."""
        # Clean up in LRU order
        while self._cache:
            self._remove_lru()
