/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Regres      regres           Regression
*/

#include <cdi.h>

#include "cdo_int.h"

// Same code as Trend!
void *
Regres(void *process)
{
  int nrecs;
  int varID, levelID;
  size_t nmiss;

  cdoInitialize(process);

  CdoStreamID streamID1 = cdoOpenRead(0);

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = vlistDuplicate(vlistID1);

  vlistDefNtsteps(vlistID2, 1);

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  /*
  CdoStreamID streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);
  */
  CdoStreamID streamID3 = cdoOpenWrite(1);
  cdoDefVlist(streamID3, vlistID2);

  VarList varList;
  varListInit(varList, vlistID1);

  const int maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  const size_t gridsizemax = vlistGridsizeMax(vlistID1);

  Field field1, field2;
  field1.resize(gridsizemax);
  field2.resize(gridsizemax);

  constexpr size_t nwork = 5;
  FieldVector2D work[nwork];
  for (auto &w : work) fieldsFromVlist(vlistID1, w, FIELD_VEC, 0);

  int64_t vdate = 0;
  int vtime = 0;
  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      vdate = taxisInqVdate(taxisID1);
      vtime = taxisInqVtime(taxisID1);

      tsID++;  // don't move this line !!!
      const double zj = (double) tsID;

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          if (tsID == 1)
            {
              recList[recID].varID = varID;
              recList[recID].levelID = levelID;
              recList[recID].lconst = vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT;
            }

          cdoReadRecord(streamID1, field1.vec.data(), &nmiss);

          const size_t gridsize = varList[varID].gridsize;
          const double missval = varList[varID].missval;

          for (size_t i = 0; i < gridsize; i++)
            if (!DBL_IS_EQUAL(field1.vec[i], missval))
              {
                work[0][varID][levelID].vec[i] += zj;
                work[1][varID][levelID].vec[i] += zj * zj;
                work[2][varID][levelID].vec[i] += zj * field1.vec[i];
                work[3][varID][levelID].vec[i] += field1.vec[i];
                work[4][varID][levelID].vec[i]++;
              }
        }
    }

  taxisDefVdate(taxisID2, vdate);
  taxisDefVtime(taxisID2, vtime);
  /* cdoDefTimestep(streamID2, 0); */
  cdoDefTimestep(streamID3, 0);

  for (int recID = 0; recID < maxrecs; recID++)
    {
      const int varID = recList[recID].varID;
      const int levelID = recList[recID].levelID;
      const size_t gridsize = varList[varID].gridsize;
      const double missval = varList[varID].missval;
      const double missval1 = missval;
      const double missval2 = missval;
      field1.size = gridsize;
      field1.missval = missval;
      field2.size = gridsize;
      field2.missval = missval;

      double temp1, temp2;
      for (size_t i = 0; i < gridsize; i++)
        {
          temp1 = SUBMN(work[2][varID][levelID].vec[i], DIVMN(MULMN(work[0][varID][levelID].vec[i], work[3][varID][levelID].vec[i]),
                                                              work[4][varID][levelID].vec[i]));
          temp2 = SUBMN(work[1][varID][levelID].vec[i], DIVMN(MULMN(work[0][varID][levelID].vec[i], work[0][varID][levelID].vec[i]),
                                                              work[4][varID][levelID].vec[i]));

          field2.vec[i] = DIVMN(temp1, temp2);
          field1.vec[i] = SUBMN(DIVMN(work[3][varID][levelID].vec[i], work[4][varID][levelID].vec[i]),
                                MULMN(DIVMN(work[0][varID][levelID].vec[i], work[4][varID][levelID].vec[i]), field2.vec[i]));
        }

      cdoDefRecord(streamID3, varID, levelID);
      cdoWriteRecord(streamID3, field2.vec.data(), fieldNumMiss(field2));
    }

  cdoStreamClose(streamID3);
  cdoStreamClose(streamID1);

  cdoFinish();

  return 0;
}
