/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Timstat    timrange        Time range
      Timstat    timmin          Time minimum
      Timstat    timmax          Time maximum
      Timstat    timsum          Time sum
      Timstat    timmean         Time mean
      Timstat    timavg          Time average
      Timstat    timvar          Time variance
      Timstat    timvar1         Time variance [Normalize by (n-1)]
      Timstat    timstd          Time standard deviation
      Timstat    timstd1         Time standard deviation [Normalize by (n-1)]
      Hourstat   hourrange       Hourly range
      Hourstat   hourmin         Hourly minimum
      Hourstat   hourmax         Hourly maximum
      Hourstat   hoursum         Hourly sum
      Hourstat   hourmean        Hourly mean
      Hourstat   houravg         Hourly average
      Hourstat   hourvar         Hourly variance
      Hourstat   hourvar1        Hourly variance [Normalize by (n-1)]
      Hourstat   hourstd         Hourly standard deviation
      Hourstat   hourstd1        Hourly standard deviation [Normalize by (n-1)]
      Daystat    dayrange        Daily range
      Daystat    daymin          Daily minimum
      Daystat    daymax          Daily maximum
      Daystat    daysum          Daily sum
      Daystat    daymean         Daily mean
      Daystat    dayavg          Daily average
      Daystat    dayvar          Daily variance
      Daystat    dayvar1         Daily variance [Normalize by (n-1)]
      Daystat    daystd          Daily standard deviation
      Daystat    daystd1         Daily standard deviation [Normalize by (n-1)]
      Monstat    monrange        Monthly range
      Monstat    monmin          Monthly minimum
      Monstat    monmax          Monthly maximum
      Monstat    monsum          Monthly sum
      Monstat    monmean         Monthly mean
      Monstat    monavg          Monthly average
      Monstat    monvar          Monthly variance
      Monstat    monvar1         Monthly variance [Normalize by (n-1)]
      Monstat    monstd          Monthly standard deviation
      Monstat    monstd1         Monthly standard deviation [Normalize by (n-1)]
      Yearstat   yearrange       Yearly range
      Yearstat   yearmin         Yearly minimum
      Yearstat   yearmax         Yearly maximum
      Yearstat   yearsum         Yearly sum
      Yearstat   yearmean        Yearly mean
      Yearstat   yearavg         Yearly average
      Yearstat   yearvar         Yearly variance
      Yearstat   yearvar1        Yearly variance [Normalize by (n-1)]
      Yearstat   yearstd         Yearly standard deviation
      Yearstat   yearstd1        Yearly standard deviation [Normalize by (n-1)]
*/

#include <cdi.h>

#include "cdo_int.h"
#include "datetime.h"
#include "printinfo.h"

enum
{
  HOUR_LEN = 4,
  DAY_LEN = 6,
  MON_LEN = 8,
  YEAR_LEN = 10
};

static void
vlistSetFrequency(int vlistID, int comparelen)
{
  const char *freq = nullptr;
  // clang-format off
  if      (comparelen == DAY_LEN)  freq = "day";
  else if (comparelen == MON_LEN)  freq = "mon";
  else if (comparelen == YEAR_LEN) freq = "year";
  // clang-format on
  if (freq) cdiDefAttTxt(vlistID, CDI_GLOBAL, "frequency", (int) strlen(freq), freq);
}

static void
addOperators(void)
{
  // clang-format off
  cdoOperatorAdd("timrange",   func_range,  DATE_LEN, nullptr);
  cdoOperatorAdd("timmin",     func_min,    DATE_LEN, nullptr);
  cdoOperatorAdd("timmax",     func_max,    DATE_LEN, nullptr);
  cdoOperatorAdd("timminidx",  func_minidx, DATE_LEN, nullptr);
  cdoOperatorAdd("timmaxidx",  func_maxidx, DATE_LEN, nullptr);
  cdoOperatorAdd("timsum",     func_sum,    DATE_LEN, nullptr);
  cdoOperatorAdd("timmean",    func_mean,   DATE_LEN, nullptr);
  cdoOperatorAdd("timavg",     func_avg,    DATE_LEN, nullptr);
  cdoOperatorAdd("timvar",     func_var,    DATE_LEN, nullptr);
  cdoOperatorAdd("timvar1",    func_var1,   DATE_LEN, nullptr);
  cdoOperatorAdd("timstd",     func_std,    DATE_LEN, nullptr);
  cdoOperatorAdd("timstd1",    func_std1,   DATE_LEN, nullptr);
  cdoOperatorAdd("yearrange",  func_range,  YEAR_LEN, nullptr);
  cdoOperatorAdd("yearmin",    func_min,    YEAR_LEN, nullptr);
  cdoOperatorAdd("yearmax",    func_max,    YEAR_LEN, nullptr);
  cdoOperatorAdd("yearminidx", func_minidx, YEAR_LEN, nullptr);
  cdoOperatorAdd("yearmaxidx", func_maxidx, YEAR_LEN, nullptr);
  cdoOperatorAdd("yearsum",    func_sum,    YEAR_LEN, nullptr);
  cdoOperatorAdd("yearmean",   func_mean,   YEAR_LEN, nullptr);
  cdoOperatorAdd("yearavg",    func_avg,    YEAR_LEN, nullptr);
  cdoOperatorAdd("yearvar",    func_var,    YEAR_LEN, nullptr);
  cdoOperatorAdd("yearvar1",   func_var1,   YEAR_LEN, nullptr);
  cdoOperatorAdd("yearstd",    func_std,    YEAR_LEN, nullptr);
  cdoOperatorAdd("yearstd1",   func_std1,   YEAR_LEN, nullptr);
  cdoOperatorAdd("monrange",   func_range,  MON_LEN, nullptr);
  cdoOperatorAdd("monmin",     func_min,    MON_LEN, nullptr);
  cdoOperatorAdd("monmax",     func_max,    MON_LEN, nullptr);
  cdoOperatorAdd("monsum",     func_sum,    MON_LEN, nullptr);
  cdoOperatorAdd("monmean",    func_mean,   MON_LEN, nullptr);
  cdoOperatorAdd("monavg",     func_avg,    MON_LEN, nullptr);
  cdoOperatorAdd("monvar",     func_var,    MON_LEN, nullptr);
  cdoOperatorAdd("monvar1",    func_var1,   MON_LEN, nullptr);
  cdoOperatorAdd("monstd",     func_std,    MON_LEN, nullptr);
  cdoOperatorAdd("monstd1",    func_std1,   MON_LEN, nullptr);
  cdoOperatorAdd("dayrange",   func_range,  DAY_LEN, nullptr);
  cdoOperatorAdd("daymin",     func_min,    DAY_LEN, nullptr);
  cdoOperatorAdd("daymax",     func_max,    DAY_LEN, nullptr);
  cdoOperatorAdd("daysum",     func_sum,    DAY_LEN, nullptr);
  cdoOperatorAdd("daymean",    func_mean,   DAY_LEN, nullptr);
  cdoOperatorAdd("dayavg",     func_avg,    DAY_LEN, nullptr);
  cdoOperatorAdd("dayvar",     func_var,    DAY_LEN, nullptr);
  cdoOperatorAdd("dayvar1",    func_var1,   DAY_LEN, nullptr);
  cdoOperatorAdd("daystd",     func_std,    DAY_LEN, nullptr);
  cdoOperatorAdd("daystd1",    func_std1,   DAY_LEN, nullptr);
  cdoOperatorAdd("hourrange",  func_range,  HOUR_LEN, nullptr);
  cdoOperatorAdd("hourmin",    func_min,    HOUR_LEN, nullptr);
  cdoOperatorAdd("hourmax",    func_max,    HOUR_LEN, nullptr);
  cdoOperatorAdd("hoursum",    func_sum,    HOUR_LEN, nullptr);
  cdoOperatorAdd("hourmean",   func_mean,   HOUR_LEN, nullptr);
  cdoOperatorAdd("houravg",    func_avg,    HOUR_LEN, nullptr);
  cdoOperatorAdd("hourvar",    func_var,    HOUR_LEN, nullptr);
  cdoOperatorAdd("hourvar1",   func_var1,   HOUR_LEN, nullptr);
  cdoOperatorAdd("hourstd",    func_std,    HOUR_LEN, nullptr);
  cdoOperatorAdd("hourstd1",   func_std1,   HOUR_LEN, nullptr);
  // clang-format on
}

void *
Timstat(void *argument)
{
  TimeStat timestat_date = TimeStat::MEAN;
  int64_t vdate0 = 0;
  int vtime0 = 0;
  int nrecs;
  int varID, levelID;
  CdoStreamID streamID3;
  int vlistID3, taxisID3 = -1;
  bool lvfrac = false;
  char indate1[DATE_LEN + 1], indate2[DATE_LEN + 1];
  double vfrac = 1;

  cdoInitialize(argument);

  addOperators();

  const int operatorID = cdoOperatorID();
  const int operfunc = cdoOperatorF1(operatorID);
  const int comparelen = cdoOperatorF2(operatorID);

  const bool lrange = operfunc == func_range;
  const bool lminidx = operfunc == func_minidx;
  const bool lmaxidx = operfunc == func_maxidx;
  const bool lmean = operfunc == func_mean || operfunc == func_avg;
  const bool lstd = operfunc == func_std || operfunc == func_std1;
  const bool lvarstd = operfunc == func_std || operfunc == func_var || operfunc == func_std1 || operfunc == func_var1;
  const int divisor = operfunc == func_std1 || operfunc == func_var1;

  if (operfunc == func_mean)
    {
      const int oargc = operatorArgc();
      char **oargv = operatorArgv();

      if (oargc == 1)
        {
          lvfrac = true;
          vfrac = atof(oargv[0]);
          if (Options::cdoVerbose) cdoPrint("Set vfrac to %g", vfrac);
          if (vfrac < 0 || vfrac > 1) cdoAbort("vfrac out of range!");
        }
      else if (oargc > 1)
        cdoAbort("Too many arguments!");
    }

  const int cmplen = DATE_LEN - comparelen;

  CdoStreamID streamID1 = cdoOpenRead(0);

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = vlistDuplicate(vlistID1);

  if (cmplen == 0) vlistDefNtsteps(vlistID2, 1);

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = taxisDuplicate(taxisID1);
  taxisWithBounds(taxisID2);
  if (taxisInqType(taxisID2) == TAXIS_FORECAST) taxisDefType(taxisID2, TAXIS_RELATIVE);
  vlistDefTaxis(vlistID2, taxisID2);

  const int nvars = vlistNvars(vlistID1);

  if (lminidx || lmaxidx)
    {
      for (varID = 0; varID < nvars; ++varID)
        vlistDefVarDatatype(vlistID2, varID, CDI_DATATYPE_INT32);
    }

  vlistSetFrequency(vlistID2, comparelen);

  CdoStreamID streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  if (Options::cdoDiag)
    {
      char filename[8192];
      strcpy(filename, cdoOperatorName(operatorID));
      strcat(filename, "_");
      strcat(filename, cdoGetStreamName(1));
      streamID3 = cdoOpenWrite(filename);

      vlistID3 = vlistDuplicate(vlistID1);

      for (varID = 0; varID < nvars; ++varID)
        {
          vlistDefVarDatatype(vlistID3, varID, CDI_DATATYPE_INT32);
          vlistDefVarMissval(vlistID3, varID, -1);
          vlistDefVarUnits(vlistID3, varID, "");
          vlistDefVarAddoffset(vlistID3, varID, 0);
          vlistDefVarScalefactor(vlistID3, varID, 1);
        }

      taxisID3 = taxisDuplicate(taxisID1);
      taxisWithBounds(taxisID3);
      vlistDefTaxis(vlistID3, taxisID3);

      cdoDefVlist(streamID3, vlistID3);
    }

  const int maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  DateTimeList dtlist;
  dtlist.setStat(timestat_date);
  dtlist.setCalendar(taxisInqCalendar(taxisID1));

  size_t gridsizemax = vlistGridsizeMax(vlistID1);
  if (vlistNumber(vlistID1) != CDI_REAL) gridsizemax *= 2;

  int FIELD_MEMTYPE = 0;
  if (operfunc == func_mean && Options::CDO_Memtype == MEMTYPE_FLOAT) FIELD_MEMTYPE = MEMTYPE_FLOAT;

  Field field;
  field.memtype = FIELD_MEMTYPE;
  if (FIELD_MEMTYPE == MEMTYPE_FLOAT)
    field.resizef(gridsizemax);
  else
    field.resize(gridsizemax);

  const bool lvars2 = lvarstd || lrange || lminidx || lmaxidx;

  FieldVector2D samp1, vars1, vars2;
  fieldsFromVlist(vlistID1, samp1, FIELD_NONE);
  fieldsFromVlist(vlistID1, vars1, FIELD_VEC);
  if (lvars2) fieldsFromVlist(vlistID1, vars2, FIELD_VEC);

  int tsID = 0;
  int otsID = 0;
  while (true)
    {
      int nsets = 0;
      while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
        {
          dtlist.taxisInqTimestep(taxisID1, nsets);
          const int64_t vdate = dtlist.getVdate(nsets);
          const int vtime = dtlist.getVtime(nsets);

          if (nsets == 0) SET_DATE(indate2, vdate, vtime);
          SET_DATE(indate1, vdate, vtime);

          if (DATE_IS_NEQ(indate1, indate2, cmplen)) break;

          for (int recID = 0; recID < nrecs; recID++)
            {
              cdoInqRecord(streamID1, &varID, &levelID);

              if (tsID == 0)
                {
                  recList[recID].varID = varID;
                  recList[recID].levelID = levelID;
                  recList[recID].lconst = vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT;
                }

              Field &rsamp1 = samp1[varID][levelID];
              Field &rvars1 = vars1[varID][levelID];

              const size_t fieldsize = rvars1.size;

              if (nsets == 0)
                {
                  cdoReadRecord(streamID1, rvars1.vec.data(), &rvars1.nmiss);
                  if (lrange)
                    {
                      vars2[varID][levelID].nmiss = rvars1.nmiss;
                      vars2[varID][levelID].vec = rvars1.vec;
                    }
                  else if (lminidx || lmaxidx)
                    {
                      vars2[varID][levelID].nmiss = rvars1.nmiss;
                      vars2[varID][levelID].vec = rvars1.vec;
                      fieldFill(rvars1, 0);
                    }

                  if (rvars1.nmiss || !rsamp1.empty())
                    {
                      if (rsamp1.empty()) rsamp1.resize(fieldsize);
                      for (size_t i = 0; i < fieldsize; i++)
                        rsamp1.vec[i] = !DBL_IS_EQUAL(rvars1.vec[i], rvars1.missval);
                    }
                }
              else
                {
                  if (Options::CDO_Memtype == MEMTYPE_FLOAT)
                    cdoReadRecordF(streamID1, field.vecf.data(), &field.nmiss);
                  else
                    cdoReadRecord(streamID1, field.vec.data(), &field.nmiss);
                  field.size = fieldsize;
                  field.grid = rvars1.grid;
                  field.missval = rvars1.missval;

                  if (field.nmiss || !rsamp1.empty())
                    {
                      if (rsamp1.empty()) rsamp1.resize(fieldsize, nsets);

                      if (Options::CDO_Memtype == MEMTYPE_FLOAT)
                        {
                          for (size_t i = 0; i < fieldsize; i++)
                            if (!DBL_IS_EQUAL(field.vecf[i], rvars1.missval)) rsamp1.vec[i]++;
                        }
                      else
                        {
                          for (size_t i = 0; i < fieldsize; i++)
                            if (!DBL_IS_EQUAL(field.vec[i], rvars1.missval)) rsamp1.vec[i]++;
                        }
                    }

                  if (lvarstd)
                    {
                      vfarsumq(vars2[varID][levelID], field);
                      vfarsum(rvars1, field);
                    }
                  else if (lrange)
                    {
                      vfarmin(vars2[varID][levelID], field);
                      vfarmax(rvars1, field);
                    }
                  else if (lminidx)
                    {
                      vfarminidx(rvars1, vars2[varID][levelID], field, nsets);
                    }
                  else if (lmaxidx)
                    {
                      vfarmaxidx(rvars1, vars2[varID][levelID], field, nsets);
                    }
                  else
                    {
                      vfarfun(rvars1, field, operfunc);
                    }
                }
            }

          if (nsets == 0 && lvarstd)
            for (int recID = 0; recID < maxrecs; recID++)
              {
                if (recList[recID].lconst) continue;

                const int varID = recList[recID].varID;
                const int levelID = recList[recID].levelID;
                vfarmoq(vars2[varID][levelID], vars1[varID][levelID]);
              }

          vdate0 = vdate;
          vtime0 = vtime;
          nsets++;
          tsID++;
        }

      if (nrecs == 0 && nsets == 0) break;

      for (int recID = 0; recID < maxrecs; recID++)
        {
          if (recList[recID].lconst) continue;

          const int varID = recList[recID].varID;
          const int levelID = recList[recID].levelID;
          Field &rsamp1 = samp1[varID][levelID];
          Field &rvars1 = vars1[varID][levelID];

          if (lmean)
            {
              if (!rsamp1.empty())
                vfardiv(rvars1, rsamp1);
              else
                vfarcdiv(rvars1, (double) nsets);
            }
          else if (lvarstd)
            {
              Field &rvars2 = vars2[varID][levelID];
              if (!rsamp1.empty())
                {
                  if (lstd)
                    vfarstd(rvars1, rvars2, rsamp1, divisor);
                  else
                    vfarvar(rvars1, rvars2, rsamp1, divisor);
                }
              else
                {
                  if (lstd)
                    vfarcstd(rvars1, rvars2, nsets, divisor);
                  else
                    vfarcvar(rvars1, rvars2, nsets, divisor);
                }
            }
          else if (lrange)
            {
              Field &rvars2 = vars2[varID][levelID];
              vfarsub(rvars1, rvars2);
            }
        }

      if (Options::cdoVerbose)
        {
          char vdatestr[32], vtimestr[32];
          date2str(vdate0, vdatestr, sizeof(vdatestr));
          time2str(vtime0, vtimestr, sizeof(vtimestr));
          cdoPrint("%s %s  vfrac = %g, nsets = %d", vdatestr, vtimestr, vfrac, nsets);
        }

      if (lvfrac && operfunc == func_mean)
        for (int recID = 0; recID < maxrecs; recID++)
          {
            if (recList[recID].lconst) continue;

            const int varID = recList[recID].varID;
            const int levelID = recList[recID].levelID;
            Field &rsamp1 = samp1[varID][levelID];
            Field &rvars1 = vars1[varID][levelID];

            const size_t fieldsize = rvars1.size;
            const double missval = rvars1.missval;
            if (!rsamp1.empty())
              {
                int irun = 0;
                for (size_t i = 0; i < fieldsize; ++i)
                  {
                    if ((rsamp1.vec[i] / nsets) < vfrac)
                      {
                        rvars1.vec[i] = missval;
                        irun++;
                      }
                  }

                if (irun) rvars1.nmiss = fieldNumMiss(rvars1);
              }
          }

      dtlist.statTaxisDefTimestep(taxisID2, nsets);
      cdoDefTimestep(streamID2, otsID);

      if (Options::cdoDiag)
        {
          dtlist.statTaxisDefTimestep(taxisID3, nsets);
          cdoDefTimestep(streamID3, otsID);
        }

      for (int recID = 0; recID < maxrecs; recID++)
        {
          if (otsID && recList[recID].lconst) continue;

          const int varID = recList[recID].varID;
          const int levelID = recList[recID].levelID;
          Field &rsamp1 = samp1[varID][levelID];
          Field &rvars1 = vars1[varID][levelID];

          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, rvars1.vec.data(), rvars1.nmiss);

          if (Options::cdoDiag)
            {
              double *sampptr = field.vec.data();
              if (!rsamp1.empty())
                sampptr = rsamp1.vec.data();
              else
                {
                  const size_t fieldsize = rvars1.size;
                  for (size_t i = 0; i < fieldsize; ++i) sampptr[i] = nsets;
                }

              cdoDefRecord(streamID3, varID, levelID);
              cdoWriteRecord(streamID3, sampptr, 0);
            }
        }

      if (nrecs == 0) break;
      otsID++;
    }

  if (Options::cdoDiag) cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
