/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2006 Brockmann Consult
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Ymonpctl   ymonpctl        Multi-year monthly percentiles
*/

#include <cdi.h>

#include "cdo_int.h"
#include "param_conversion.h"
#include "percentiles_hist.h"
#include "percentiles.h"

#define NMONTH 17

static int
getmonth(int64_t date)
{
  int year, month, day;
  cdiDecodeDate(date, &year, &month, &day);
  return month;
}

void *
Ymonpctl(void *process)
{
  int varID;
  int gridID;
  int year, month, day;
  int levelID;
  size_t nmiss;
  int nrecs, nlevels;
  int64_t vdates1[NMONTH], vdates2[NMONTH];
  int vtimes1[NMONTH];
  long nsets[NMONTH];
  FieldVector2D vars1[NMONTH];
  HISTOGRAM_SET *hsets[NMONTH];

  cdoInitialize(process);
  cdoOperatorAdd("ymonpctl", func_pctl, 0, nullptr);

  operatorInputArg("percentile number");
  const double pn = parameter2double(operatorArgv()[0]);
  percentile_check_number(pn);

  for (month = 0; month < NMONTH; month++)
    {
      hsets[month] = nullptr;
      nsets[month] = 0;
    }

  CdoStreamID streamID1 = cdoOpenRead(0);
  CdoStreamID streamID2 = cdoOpenRead(1);
  CdoStreamID streamID3 = cdoOpenRead(2);

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = cdoStreamInqVlist(streamID2);
  const int vlistID3 = cdoStreamInqVlist(streamID3);
  const int vlistID4 = vlistDuplicate(vlistID1);

  vlistCompare(vlistID1, vlistID2, CMP_ALL);
  vlistCompare(vlistID1, vlistID3, CMP_ALL);

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = vlistInqTaxis(vlistID2);
  const int taxisID3 = vlistInqTaxis(vlistID3);
  /* TODO - check that time axes 2 and 3 are equal */

  const int taxisID4 = taxisDuplicate(taxisID1);
  if (taxisHasBounds(taxisID4)) taxisDeleteBounds(taxisID4);
  vlistDefTaxis(vlistID4, taxisID4);

  CdoStreamID streamID4 = cdoOpenWrite(3);
  cdoDefVlist(streamID4, vlistID4);

  const int nvars = vlistNvars(vlistID1);

  const int maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  const size_t gridsizemax = vlistGridsizeMax(vlistID1);

  Field field;
  field.resize(gridsizemax);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID2, tsID)))
    {
      if (nrecs != cdoStreamInqTimestep(streamID3, tsID))
        cdoAbort("Number of records at time step %d of %s and %s differ!", tsID + 1, cdoGetStreamName(1), cdoGetStreamName(2));

      const int64_t vdate = taxisInqVdate(taxisID2);
      const int vtime = taxisInqVtime(taxisID2);

      if (vdate != taxisInqVdate(taxisID3))
        cdoAbort("Verification dates at time step %d of %s and %s differ!", tsID + 1, cdoGetStreamName(1), cdoGetStreamName(2));

      if (Options::cdoVerbose) cdoPrint("process timestep: %d %d %d", tsID + 1, vdate, vtime);

      cdiDecodeDate(vdate, &year, &month, &day);
      if (month < 0 || month >= NMONTH) cdoAbort("Month %d out of range!", month);

      vdates2[month] = vdate;

      if (!vars1[month].size())
        {
          fieldsFromVlist(vlistID1, vars1[month], FIELD_VEC);
          hsets[month] = hsetCreate(nvars);

          for (varID = 0; varID < nvars; varID++)
            {
              gridID = vlistInqVarGrid(vlistID1, varID);
              nlevels = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));

              hsetCreateVarLevels(hsets[month], varID, nlevels, gridID);
            }
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID2, &varID, &levelID);
          cdoReadRecord(streamID2, vars1[month][varID][levelID].vec.data(), &nmiss);
          vars1[month][varID][levelID].nmiss = nmiss;
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID3, &varID, &levelID);
          cdoReadRecord(streamID3, field.vec.data(), &field.nmiss);
          field.grid = vars1[month][varID][levelID].grid;
          field.missval = vars1[month][varID][levelID].missval;

          hsetDefVarLevelBounds(hsets[month], varID, levelID, &vars1[month][varID][levelID], &field);
        }

      tsID++;
    }

  tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      const int64_t vdate = taxisInqVdate(taxisID1);
      const int vtime = taxisInqVtime(taxisID1);

      if (Options::cdoVerbose) cdoPrint("process timestep: %d %d %d", tsID + 1, vdate, vtime);

      cdiDecodeDate(vdate, &year, &month, &day);
      if (month < 0 || month >= NMONTH) cdoAbort("Month %d out of range!", month);

      vdates1[month] = vdate;
      vtimes1[month] = vtime;

      if (!vars1[month].size()) cdoAbort("No data for month %d in %s and %s", month, cdoGetStreamName(1), cdoGetStreamName(2));

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          if (tsID == 0)
            {
              recList[recID].varID = varID;
              recList[recID].levelID = levelID;
              recList[recID].lconst = vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT;
            }

          cdoReadRecord(streamID1, vars1[month][varID][levelID].vec.data(), &nmiss);
          vars1[month][varID][levelID].nmiss = nmiss;

          hsetAddVarLevelValues(hsets[month], varID, levelID, &vars1[month][varID][levelID]);
        }

      nsets[month]++;
      tsID++;
    }

  int otsID = 0;
  for (month = 0; month < NMONTH; month++)
    if (nsets[month])
      {
        if (getmonth(vdates1[month]) != getmonth(vdates2[month]))
          cdoAbort("Verification dates for the month %d of %s and %s are different!", month, cdoGetStreamName(0),
                   cdoGetStreamName(1));

        for (varID = 0; varID < nvars; varID++)
          {
            if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT) continue;
            nlevels = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));

            for (levelID = 0; levelID < nlevels; levelID++)
              hsetGetVarLevelPercentiles(&vars1[month][varID][levelID], hsets[month], varID, levelID, pn);
          }

        taxisDefVdate(taxisID4, vdates1[month]);
        taxisDefVtime(taxisID4, vtimes1[month]);
        cdoDefTimestep(streamID4, otsID);

        for (int recID = 0; recID < maxrecs; recID++)
          {
            if (otsID && recList[recID].lconst) continue;

            const int varID = recList[recID].varID;
            const int levelID = recList[recID].levelID;
            cdoDefRecord(streamID4, varID, levelID);
            cdoWriteRecord(streamID4, vars1[month][varID][levelID].vec.data(), vars1[month][varID][levelID].nmiss);
          }

        otsID++;
      }

  for (month = 0; month < NMONTH; month++)
    {
      if (vars1[month].size())
        {
          hsetDestroy(hsets[month]);
        }
    }

  cdoStreamClose(streamID4);
  cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
