/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2011 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#define _BSD_SOURCE /* lstat() */

#include <sys/stat.h>
#include <errno.h>
#include <unistd.h>

#include "main.h"
#include "cdw_config.h"
#include "cdw_task.h"
#include "cdw_widgets.h"
#include "gettext.h"
#include "cdw_debug.h"
#include "cdw_burn_disc.h"
#include "cdw_write_wizard.h"

#include "cdw_utils.h"
#include "cdw_drive.h"
#include "cdw_file_picker.h"
#include "cdw_file_manager.h"
#include "cdw_cdrecord.h"
#include "cdw_growisofs.h"
#include "cdw_xorriso.h"
#include "cdw_digest.h"
#include "cdw_logging.h"
#include "cdw_main_window.h"
#include "cdw_iso9660.h"
#include "cdw_ext_tools.h"
#include "cdw_config_window.h"
#include "cdw_processwin.h"
#include "cdw_read_disc_info.h"
#include "cdw_calculate_digest.h"


static cdw_rv_t cdw_burn_disc_dispatcher(cdw_task_t *parent_task, cdw_disc_t *disc);
static bool cdw_burn_disc_is_disc_ready(void);
static bool cdw_burn_disc_get_source(cdw_task_t *task);
static void cdw_burn_disc_release_source(cdw_task_t *task);
static bool cdw_burn_disc_check_wodim_large_image(cdw_task_t *task);
static bool cdw_burn_disc_check_overwrite_content(cdw_task_t *task, cdw_disc_t *disc);

/**
   \brief Check if everything is ready for writing to optical disc and then perform it

   Function does pre-writing checks, displays write wizard and
   calls burn_dispatcher() to do the job.

   Supported tasks are writing directly to disc and writing image to disc.
   Supported disc types are data cd and dvd.
   When combined the two things above we have following tasks:
    - write files to dvd
    - write image to dvd
    - write files to cd
    - write image to cd

   The decision is based on value of task->source and type of
   disc. Disc type is checked during this function call, before performing
   task.

   Function checks if image is available or if any files are selected for
   writing.

   \param task_id CDW_TASK_BURN_FROM_IMAGE or CDW_TASK_BURN_FROM_FILES

   \return CDW_OK on success
   \return CDW_NO if some preconditions were not met
   \return CDW_CANCEL if user cancels writing
   \return CDW_ERROR on failure or some error
*/
cdw_rv_t cdw_burn_disc(int task_id)
{
	cdw_assert (task_id == CDW_TASK_BURN_FROM_FILES || task_id == CDW_TASK_BURN_FROM_IMAGE,
		    "ERROR: called the function with incorrect task id %d\n", task_id);

	/* check if there is a disc in the drive and if it is writable */
	if (! cdw_burn_disc_is_disc_ready()) {
		return CDW_NO;
	}

	cdw_disc_t *current_disc = cdw_disc_get();
	cdw_task_t *task = cdw_task_new(task_id, current_disc);
	if (task == (cdw_task_t *) NULL) {
		cdw_vdm ("ERROR: failed to create a task\n");
		return CDW_NO;
	}

	bool source_available = cdw_burn_disc_get_source(task);
	if (! source_available) {
		cdw_task_delete(&task);
		return CDW_NO;
	}

	bool wodim_ok = cdw_burn_disc_check_wodim_large_image(task);
	if (!wodim_ok) {
		cdw_burn_disc_release_source(task);
		cdw_task_delete(&task);
		return CDW_CANCEL;
	}

	cdw_rv_t decision = cdw_write_wizard(task, current_disc);

	if (decision != CDW_OK) {
		cdw_burn_disc_release_source(task);
		cdw_task_delete(&task);
		if (decision == CDW_ERROR) {
			/* 2TRANS: this is the title of dialog window */
			cdw_buttons_dialog(_("Error"),
					   /* 2TRANS: this is the message in the dialog window */
					   _("cdw can't create write wizard. Please restart cdw."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		}
		return decision;
	}

	bool overwrite_content = cdw_burn_disc_check_overwrite_content(task, current_disc);
	if (!overwrite_content) {
		/* there was already some content on the disc, it might have
		   been overwritten, but user decided not to overwrite it */
		cdw_burn_disc_release_source(task);
		cdw_task_delete(&task);
		return CDW_CANCEL;
	}

	cdw_main_ui_main_window_wrefresh();

	cdw_assert (task->id == CDW_TASK_BURN_FROM_FILES || task->id == CDW_TASK_BURN_FROM_IMAGE,
		    "ERROR: task id has changed\n");

	/* 2TRANS: this is a title of process window;
	   "disc" is "optical disc" */
	cdw_processwin_create(_("Writing to disc"),
			      /* 2TRANS: this is message in process window;
				 "disc" is "optical disc" */
			      _("Writing to optical disc"), true);

	/* burn_dispatcher() calls actual function performing burning */
	cdw_rv_t command_return = cdw_burn_disc_dispatcher(task, current_disc);
	/* writing has been finished, erase information that was updated
	   (and up to date) only during actual writing; displaying it
	   after this point may be misleading - user might think that
	   it is somehow still valid */
	cdw_processwin_erase_fifo_and_speed();
	cdw_processwin_erase_eta();
	cdw_burn_disc_release_source(task);
	const char *drive_fullpath = cdw_drive_get_drive_fullpath();
	cdw_drive_eject_tray_with_ui_update(drive_fullpath);

	cdw_rv_t tool_status = cdw_task_check_tool_status(task);

	if (command_return == CDW_OK && tool_status == CDW_OK) {
		if (task->burn.verify) {
			cdw_vdm ("INFO: \"verify burn\" is true, attempting verification\n");
			cdw_drive_close_tray_with_ui_update(drive_fullpath);

			cdw_rv_t verification = cdw_calculate_digest(CDW_CALCULATE_DIGEST_MODE_DISC_FILE, task->burn.iso9660_file_fullpath);
			if (verification != CDW_OK) {
				cdw_vdm ("ERROR: verification function returns !CDW_OK\n");
				command_return = CDW_ERROR;
			}
		} else {
			cdw_vdm ("INFO: \"verify burn\" is false, not attempting to verify\n");
		}
	} else {
		/* incorrect situation, already covered by debug messages above */
		command_return = CDW_NO;
	}
	int status = cdw_drive_status(drive_fullpath);
	if (task->burn.eject) {
		if (status != CDW_CD_TRAY_OPEN) {
			cdw_drive_eject_tray_with_ui_update(drive_fullpath);
		}
	} else {
		if (status == CDW_CD_TRAY_OPEN) {
			cdw_drive_close_tray_with_ui_update(drive_fullpath);
		}
	}

	cdw_rv_t retval = CDW_OK;
	if (command_return == CDW_OK && tool_status == CDW_OK) {
		cdw_task_save(task);

		/* 2TRANS: this is message in dialog window:
		   operation finished with unknown result, probably success */
		cdw_processwin_destroy(_("Writing finished"), true);
		retval = CDW_OK;
	} else {
		cdw_vdm ("ERROR: returning CDW_ERROR due to errors reported above\n");
		/* 2TRANS: this is message in dialog window:
		   operation finished (most probably) with error */
		cdw_processwin_destroy(_("Writing probably failed"), true);
		retval = CDW_ERROR;
	}

	/* 2TRANS: this is the title of the dialog window, displaying messages
	   from the program writing iso image or selected files to CD disc */
	cdw_logging_display_log_conditional(_("\"Write\" log"));

	cdw_task_delete(&task);
	return retval;
}





bool cdw_burn_disc_check_wodim_large_image(cdw_task_t *task)
{
	bool ask = task->id == CDW_TASK_BURN_FROM_IMAGE
		&& task->burn.tool.id == CDW_TOOL_CDRECORD
		&& cdw_ext_tools_is_cdrecord_wodim();

	if (!ask) {
		return true;
	}

	/* sector = 2048 bytes */
	long sectors = cdw_iso_image_get_n_sectors();
	double gigabytes = ((float) sectors * 2.0) / (1024.0 * 1024.0);
	cdw_vdm ("INFO: sectors = %ld -> %fGB\n", sectors, gigabytes);
	if (sectors > 2 * 1024 * 1024) {
		cdw_vdm ("INFO: large image: %ld sectors -> %fGB\n", sectors, gigabytes);
		/* 2TRANS: this is the title of dialog window */
		cdw_rv_t crv = cdw_buttons_dialog(_("Warning"),
						  /* 2TRANS: this is the message in the dialog window */
						  _("You are attempting to use wodim to burn large image. This task will probably fail. You should try using another tool (cdrecord or growisofs) for this task. Do you want to continue?"),
						  CDW_BUTTONS_OK_CANCEL, CDW_COLORS_ERROR);
		if (crv == CDW_OK) {
			/* 2TRANS: this is message printed in log file;
			   %ld is size of ISO9660 image (in sectors) */
			cdw_logging_write(_("Attempting to burn ISO image of size %ld sectors with wodim, this may fail...\n"), sectors);
			return true;
		} else {
			return false;
		}
	} else {
		return true;
	}
}





bool cdw_burn_disc_check_overwrite_content(cdw_task_t *task, cdw_disc_t *disc)
{
	bool ask = disc->state_empty != CDW_TRUE
		&& (task->burn.session_mode == CDW_SESSION_MODE_START_MULTI
		    || task->burn.session_mode == CDW_SESSION_MODE_CREATE_SINGLE);

	if (!ask) {
		return true;
	}

	/* 2TRANS: this is the title of dialog window */
	cdw_rv_t crv = cdw_buttons_dialog(_("Warning"),
					  /* 2TRANS: this is the message in the dialog window */
					  _("The disc is not blank, you will overwrite its current content. Continue?"),
					  CDW_BUTTONS_OK_CANCEL, CDW_COLORS_WARNING);
	if (crv == CDW_OK) {
		return true;
	} else {
		return false;
	}
}





/**
   \brief Pass control to cdrecord, growisofs or xorriso code to perform burning

   This function is just a dispatcher, and calls cdw_cdrecord_run_task(),
   cdw_growisofs_run_task() or cdw_xorriso_run_task(), depending on
   task->burn.tool.id.

   The function checks all important assertions before doing actual
   dispatching, but only in debug build.

   \param task - variable describing current task
   \param disc - variable describing current disc

   \return return value from cdrecord, growisofs or xorriso cdw_*_run_task() function
*/
cdw_rv_t cdw_burn_disc_dispatcher(cdw_task_t *task, cdw_disc_t *disc)
{
	if (task->burn.tool.id != CDW_TOOL_XORRISO) {
		cdw_assert (task->burn.disc_mode != CDW_DISC_MODE_INIT, "ERROR: calling the function with uninitialized disc mode\n");
	}
	cdw_assert (task->burn.session_mode != CDW_SESSION_MODE_INIT, "ERROR: calling the function with uninitialized session mode\n");
	cdw_assert (task->id == CDW_TASK_BURN_FROM_FILES || task->id == CDW_TASK_BURN_FROM_IMAGE,
		    "ERROR: incorrect task id %lld\n", task->id);
	cdw_assert (task->burn.tool.id == CDW_TOOL_CDRECORD
		    || task->burn.tool.id == CDW_TOOL_GROWISOFS
		    || task->burn.tool.id == CDW_TOOL_XORRISO,
		    "ERROR: incorrect task->main_tool_id %lld\n", task->burn.tool.id);

	if (task->id == CDW_TASK_BURN_FROM_FILES) {
		if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
			/* 2TRANS: this is message in process window:
			   writing selected files to CD is in progress */
			cdw_processwin_display_main_info(_("Writing files to CD..."));
		} else {
			/* 2TRANS: this is message in process window:
			   writing selected files to DVD is in progress */
			cdw_processwin_display_main_info(_("Writing files to DVD..."));
		}
	} else {
		if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
			/* 2TRANS: this is message in process window -
			   writing iso image to CD is in progress */
			cdw_processwin_display_main_info(_("Writing image to CD..."));
		} else {
			/* 2TRANS: this is message in process window -
			   writing iso image to DVD is in progress */
			cdw_processwin_display_main_info(_("Writing image to DVD..."));
		}
	}

	/* at this point we don't care what kind of disc we have,
	   because "external tools" module already checked disc type and
	   decided which tool to use, e.g. to use cdrecord or growisofs for
	   writing to DVD; so we dispatch basing only on tool id */
	cdw_rv_t crv = CDW_OK;
	if (task->burn.tool.id == CDW_TOOL_CDRECORD) {
		cdw_vdm ("INFO: dispatching \"burn\" task to cdrecord\n");
		crv = cdw_cdrecord_run_task(task, disc);
	} else if (task->burn.tool.id == CDW_TOOL_GROWISOFS) {
		cdw_vdm ("INFO: dispatching \"burn\" task to growisofs\n");
		crv = cdw_growisofs_run_task(task, disc);
	} else if (task->burn.tool.id == CDW_TOOL_XORRISO) {
		cdw_vdm ("INFO: dispatching \"burn\" task to xorriso\n");
		crv = cdw_xorriso_run_task(task, disc);
	} else {
		cdw_assert (0, "ERROR: wrong tool id '%lld' for the job\n", task->burn.tool.id);
	}

	/* it is too early to check task->tool_status.ok */
	cdw_vdm ("INFO: burn dispatcher returns %s\n", cdw_utils_get_crv_label(crv));
	return crv;
}





bool cdw_burn_disc_get_source(cdw_task_t *task)
{
	bool source_available = false;

	if (task->id == CDW_TASK_BURN_FROM_IMAGE) {
		/* 2TRANS: this is the title of dialog window */
		cdw_rv_t crv = cdw_fs_ui_file_picker(_("Path to iso image"),
						     /* 2TRANS: this is the message in the dialog window;
						     "writing" means writing to optical disc */
						     _("Please enter FULL path to an existing iso image file for writing:"),
						     &(task->burn.iso9660_file_fullpath),
						     CDW_FS_FILE, R_OK, CDW_FS_EXISTING);

		cdw_main_ui_main_window_wrefresh();
		if (crv == CDW_OK) {
			/* TODO: replace with call to generic
			   cdw_fs_get_size_(const char *fullpath, int units); */
			task->burn.data_size_mb = cdw_iso_image_get_size_mb(task->burn.iso9660_file_fullpath);
			source_available = true;
		}
	} else if (task->id == CDW_TASK_BURN_FROM_FILES) {
		cdw_rv_t crv = cdw_file_manager_create_graftpoints_file();
		if (crv == CDW_OK) {
			long long size = cdw_selected_files_get_size();
			task->burn.data_size_mb = (((double) size) / 1024.0) / 1024.0;

			source_available = true;
			if (cdw_selected_files_file_over_4gb_present()) {
				cdw_vdm ("WARNING: selected file > 4 GB\n");
				/* 2TRANS: this is title of dialog window */
				cdw_buttons_dialog(_("Warning"),
						   /* 2TRANS: this is the message in the dialog window:
						      disc is not recognized as supported by
						      currently selected tools */
						   _("One of selected files has a size over 4GB. Choose your tools wisely (Configuration -> Tools), otherwise writing to disc will give incorrect results."),
						   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
			} else {
				cdw_sdm ("INFO: file size <= 4 GB\n");
			}
		} else if (crv == CDW_NO) {
			; /* probably no selected files */
		} else {
			cdw_vdm ("ERROR: failed to correctly create graftpoints file\n");
		}
	} else {
		; /* covered by caller's assert */
	}

	return source_available;
}





void cdw_burn_disc_release_source(cdw_task_t *task)
{
	if (task->id == CDW_TASK_BURN_FROM_FILES) {
		cdw_file_manager_delete_graftpoints_file();
	}

	return;
}





bool cdw_burn_disc_is_disc_ready(void)
{
	cdw_rv_t crv = cdw_read_disc_info();
	if (crv != CDW_OK) {
		return false;
	}

	/* update information about disc size and usage */
	cdw_main_window_volume_info_view_update(true, cdw_config_follow_symlinks());

	cdw_disc_t *current_disc = cdw_disc_get();
	//cdw_main_ui_disc_info_view_display_data(current_disc);

	if (current_disc->type == CDW_DISC_TYPE_UNKNOWN) {
		/* 2TRANS: this is the title of the dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is the message in the dialog window:
				      disc is not recognized as supported by
				      currently selected tools */
				   _("Can't recognize disc in drive. Try changing tools family in Configuration -> Tools or use different disc type."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

		return false;
	}

	if (current_disc->type_writable != CDW_TRUE) {
		/* CD-AUDIO, DVD-ROM etc. */
		/* 2TRANS: this is the title of the dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is the message in the dialog window - disc is not writable */
				   _("Disc in drive is read-only."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

		return false;
	}


	if (current_disc->state_writable == CDW_UNKNOWN) {
		/* perhaps device path is invalid? */

		/* 2TRANS: this is the title of the dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is the message in the dialog window: for some
				      reason cdw cannot read disc metainfo; this may be a
				      problem with the hardware configuration or with the family
				      of tools (cdrtools / dvd+rw-tools) used for the task */
				   _("Cannot get full information about disc. Please check configuration of hardware or tools. (Are you using wodim?)."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return false;
	} else if (current_disc->state_writable == CDW_FALSE) {

		/* 2TRANS: this is the title of the dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is the message in the dialog window - user tries to write to a closed disc */
				   _("Cannot write to this disc anymore, disc is closed."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return false;
	} else {
		/* current_disc->state_appendable == CDW_DISC_APPENDABLE_YES */
		if (current_disc->type == CDW_DVD_RP_DL
		    && cdw_config_support_dvd_rp_dl()) {

			/* 2TRANS: this is the title of the dialog window */
			cdw_rv_t crv2 = cdw_buttons_dialog(_("Information"),
							   /* 2TRANS: this is the message in the dialog window */
							  _("You are attempting to burn to a DVD+R DL disc. You REALLY need to use high quality discs and have a bit of luck to correctly burn data to DVD+R DL. Additionally cdw offers very limited support for DVD+RD DL discs. Continue?"),
							  CDW_BUTTONS_OK_CANCEL, CDW_COLORS_DIALOG);
			if (crv2 == CDW_OK) {
				return true;
			} else {
				return false;
			}
		}
		return true;
	}
}



