// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "base/file_util.h"
#include "base/files/file_path.h"
#include "base/strings/string_util.h"
#include "base/strings/stringprintf.h"
#include "mojo/public/cpp/bindings/lib/message_builder.h"
#include "mojo/public/cpp/bindings/lib/message_internal.h"
#include "mojo/public/cpp/bindings/message.h"

// This file is used to generate various files corresponding to mojo
// messages. The various binding implementations can parse these to verify they
// correctly decode messages.
//
// The output consists of each byte of the message encoded in a hex string with
// a newline after it.
namespace mojo {
namespace {

std::string BinaryToHex(const base::StringPiece& piece) {
  std::string result("// File generated by mojo_message_generator.\n");;
  result.reserve(result.size() + (piece.size() * 5));
  for (size_t i = 0; i < piece.size(); ++i)
    base::StringAppendF(&result, "0X%.2X\n", static_cast<int>(piece.data()[i]));
  return result;
}

void WriteMessageToFile(const Message& message, const base::FilePath& path) {
  const std::string hex_message(BinaryToHex(
      base::StringPiece(reinterpret_cast<const char*>(message.data()),
                        message.data_num_bytes())));
  CHECK_EQ(static_cast<int>(hex_message.size()),
           base::WriteFile(path, hex_message.data(),
                           static_cast<int>(hex_message.size())));
}

// Generates a message of type MessageData. The message uses the name 21,
// with 4 bytes of payload: 0x9, 0x8, 0x7, 0x6.
void GenerateMessageDataMessage() {
  internal::MessageBuilder builder(static_cast<uint32_t>(21),
                                   static_cast<size_t>(4));
  char* data = static_cast<char*>(builder.buffer()->Allocate(4));
  DCHECK(data);
  data[0] = 9;
  data[1] = 8;
  data[2] = 7;
  data[3] = 6;

  Message message;
  builder.Finish(&message);
  WriteMessageToFile(message,
                     base::FilePath(FILE_PATH_LITERAL("message_data")));
}

}  // namespace
}  // namespace mojo

int main(int argc, char** argv) {
  mojo::GenerateMessageDataMessage();
  return 0;
}
