/* $Id: cpl_fits.c,v 1.15 2010/12/20 15:30:19 llundin Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2008 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: llundin $
 * $Date: 2010/12/20 15:30:19 $
 * $Revision: 1.15 $
 * $Name: cpl-5_3_0-BRANCH $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <string.h>
#include <fitsio.h>

#include "cpl_fits.h"
#include "cpl_error_impl.h"
#include "cpl_memory.h"
#include "cpl_tools.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup cpl_fits   FITS related basic routines
 *
 * This module provides functions to get basic information on FITS files
 *
 * @par Synopsis:
 * @code
 *   #include "cpl_fits.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
                            Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief  Get the number of extensions contained in a FITS file
  @param  filename    The file name
  @return The number of extensions or -1 in case of error
  @note For a valid fits file without extensions zero is returned

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if the input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the input file is not FITS
 */
/*----------------------------------------------------------------------------*/
int cpl_fits_count_extensions(const char * filename)
{
    fitsfile * fptr;
    int        error = 0;
    int        next  = 0;

    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, -1);

    if (fits_open_diskfile(&fptr, filename, READONLY, &error)) {
        (void)cpl_error_set_fits(CPL_ERROR_FILE_IO, error, fits_open_diskfile,
                                 "filename='%s'", filename);
        return -1;
    }

    if (fits_get_num_hdus(fptr, &next, &error)) {
        (void)cpl_error_set_fits(CPL_ERROR_FILE_IO, error, fits_get_num_hdus,
                                 "filename='%s'", filename);
        error = 0; /* Reset so fits_close_file works */
        next = 0;
    }

    if (fits_close_file(fptr, &error)) {
        (void)cpl_error_set_fits(CPL_ERROR_FILE_IO, error, fits_close_file,
                                 "filename='%s'", filename);
    }

    /* The nb of HDUs includes the primary one, which is not an extension */
    return next - 1;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the place of a given extension in a FITS file
  @param    filename The file name
  @param    extname  The extension name
  @return   the extension number, 0 if not found or -1 on error

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if the file is not FITS
 */
/*----------------------------------------------------------------------------*/
int cpl_fits_find_extension(const char * filename,
                            const char * extname)
{
    fitsfile * fptr;
    int        error = 0;
    int        ext_num    = 0;
    char       c_extname[FLEN_VALUE];

    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, -1);
    cpl_ensure(extname  != NULL, CPL_ERROR_NULL_INPUT, -1);

    if (fits_open_diskfile(&fptr, filename, READONLY, &error)) {
        (void)cpl_error_set_fits(CPL_ERROR_FILE_IO, error, fits_open_diskfile,
                                 "filename='%s', extname='%s'",
                                 filename, extname);
        return -1;
    }

    do {
        const char * keyword = "EXTNAME";

        if (!fits_movrel_hdu(fptr, 1, NULL, &error) &&
            /* FIXME: Cast string literal to avoid compiler warnings for CFTISIO
               older than v. 3.18 */
            !fits_read_key_str(fptr, (char*)keyword, c_extname, NULL, &error))

            ext_num++;

    } while (!error && strncmp(extname, c_extname, FLEN_VALUE));

    if (error) {
        ext_num = error == END_OF_FILE ? 0 : -1;
        error = 0; /* Reset so fits_close_file works */
    }

    if (fits_close_file(fptr, &error)) {
        (void)cpl_error_set_fits(CPL_ERROR_FILE_IO, error, fits_close_file,
                                 "filename='%s', extname='%s'",
                                 filename, extname);
        ext_num = -1;
    }

    return ext_num;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the number of extensions contained in a FITS file
  @param    filename    The file name
  @return   the number of extensions or -1 in case of error
  @see cpl_fits_count_extensions()
  @deprecated Replace this call with cpl_fits_count_extensions().

 */
/*----------------------------------------------------------------------------*/

int cpl_fits_get_nb_extensions(const char * filename)
{
    return cpl_fits_count_extensions(filename);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the place of a given extension in a FITS file
  @param    filename    The file name
  @param    extname     The extension name
  @return   the extension place or -1 in case of error
  @see cpl_fits_find_extension
  @deprecated Replace this call with cpl_fits_find_extension().

 */
/*----------------------------------------------------------------------------*/

int cpl_fits_get_extension_nb(const char  *   filename,
                              const char  *   extname)
{
    return cpl_fits_find_extension(filename, extname);
}

/**@}*/
