/* $Id: cpl_image_io.c,v 1.213 2011/01/21 16:24:21 llundin Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2008 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: llundin $
 * $Date: 2011/01/21 16:24:21 $
 * $Revision: 1.213 $
 * $Name: cpl-5_3_0-BRANCH $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <math.h>
#include <float.h>
#include <complex.h>
#include <assert.h>

#include <fitsio.h>

#include "cpl_error_impl.h"
#include "cpl_tools.h"
#include "cpl_memory.h"
#include "cpl_propertylist_impl.h"
#include "cpl_image_io_impl.h"
#include "cpl_mask.h"
#include "cpl_image_bpm.h"

#include "cpl_image_defs.h"

/*-----------------------------------------------------------------------------
                                   Defines
 -----------------------------------------------------------------------------*/

#define CPL_IMAGE_IO_GET_PIXELS    2
#define CPL_IMAGE_IO_SET_BADPIXEL  7
#define CPL_IMAGE_IO_GET           8
#define CPL_IMAGE_IO_SET           9
#define CPL_IMAGE_COMPLEX          10

/* Needed for cpl_image_floodfill() */
#define FFSTACK_MAXLINES 10
#define FFSTACK_STACKSZ(IMAGE) (FFSTACK_MAXLINES * IMAGE->ny)
#define FFSTACK_BYTES(IMAGE) (FFSTACK_STACKSZ(IMAGE) * 4 * sizeof(int))

/*-----------------------------------------------------------------------------
                        Private function prototypes
 -----------------------------------------------------------------------------*/

static void cpl_image_floodfill(cpl_image *, void *, int, int, int);
static cpl_image * cpl_image_wrap_(int, int, cpl_type, void *) CPL_ATTR_ALLOC;

static
cpl_image * cpl_image_load_one(const char *, cpl_type, int, int,
                               cpl_boolean, int, int, int, int) CPL_ATTR_ALLOC;

/*-----------------------------------------------------------------------------
                            Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Allocate an image structure and pixel buffer for a image.
  @param    nx          Size in x
  @param    ny          Size in y
  @param    type        The pixel type
  @return   1 newly allocated cpl_image or NULL in case of an error

  Allocates space for the cpl_image structure and sets the dimensions and
  type of pixel data. The pixel buffer is allocated and initialised to zero.
  The pixel array will contain nx*ny values being the image pixels from the
  lower left to the upper right line by line. 

  Supported pixel types are CPL_TYPE_INT, CPL_TYPE_FLOAT, CPL_TYPE_DOUBLE,
  CPL_TYPE_FLOAT_COMPLEX and CPL_TYPE_DOUBLE_COMPLEX.

  The returned cpl_image must be deallocated using cpl_image_delete().

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_ILLEGAL_INPUT if nx or ny is non-positive
  - CPL_ERROR_INVALID_TYPE if the passed image type is not supported

 */
/*----------------------------------------------------------------------------*/
cpl_image * cpl_image_new(int         nx,
                          int         ny,
                          cpl_type    type)
{

    cpl_image * self = cpl_image_wrap_(nx, ny, type, NULL);

    if (self == NULL) (void)cpl_error_set_where(cpl_func);

    return self;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Create an image using an existing pixel buffer of the specified type
  @param    nx      Size in x
  @param    ny      Size in y
  @param    type    Pixel type
  @param    pixels  Pixel data of the specified type
  @return   1 newly allocated cpl_image or NULL on error
  @see      cpl_image_new()

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if nx or ny is non-positive or their product is
                            too big
  - CPL_ERROR_INVALID_TYPE if the passed image type is not supported
 */
/*----------------------------------------------------------------------------*/
cpl_image * cpl_image_wrap(int nx, int ny, cpl_type type, void * pixels)
{
    cpl_image  * self;

    cpl_ensure(pixels != NULL, CPL_ERROR_NULL_INPUT, NULL);

    self = cpl_image_wrap_(nx, ny, type, pixels);

    if (self == NULL) (void)cpl_error_set_where(cpl_func);

    return self;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Create a double image using an existing pixel buffer.
  @param    nx          Size in x
  @param    ny          Size in y
  @param    pixels      double * pixel data
  @return   1 newly allocated cpl_image or NULL in case of an error
  @see      cpl_image_new

  The pixel array is set to point to that of the argument.
  The pixel array must contain nx*ny doubles. 

  The allocated image must be deallocated with cpl_image_unwrap().

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if nx or ny is non-positive or zero.
 */
/*----------------------------------------------------------------------------*/
cpl_image * cpl_image_wrap_double(
        int                 nx, 
        int                 ny,
        double          *   pixels)
{
    cpl_image * self;

    cpl_ensure(pixels != NULL, CPL_ERROR_NULL_INPUT, NULL);

    self = cpl_image_wrap_(nx, ny, CPL_TYPE_DOUBLE, pixels);

    if (self == NULL) (void)cpl_error_set_where(cpl_func);

    return self;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Create a float image using an existing pixel buffer.
  @param    nx          Size in x
  @param    ny          Size in y
  @param    pixels      float * pixel data.
  @return   1 newly allocated cpl_image or NULL on error
  @see      cpl_image_wrap_double()
 */
/*----------------------------------------------------------------------------*/
cpl_image * cpl_image_wrap_float(
        int             nx, 
        int             ny,
        float       *   pixels)
{
    cpl_image * self;

    cpl_ensure(pixels != NULL, CPL_ERROR_NULL_INPUT, NULL);

    self = cpl_image_wrap_(nx, ny, CPL_TYPE_FLOAT, pixels);

    if (self == NULL) (void)cpl_error_set_where(cpl_func);

    return self;
}


/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Create a double complex image using an existing pixel buffer.
  @param    nx          Size in x
  @param    ny          Size in y
  @param    pixels      double complex * pixel data.
  @return   1 newly allocated cpl_image or NULL on error
  @see      cpl_image_wrap_double()
  @note This function is available iff the application includes complex.h
 */
/*----------------------------------------------------------------------------*/
cpl_image * cpl_image_wrap_double_complex(
        int                      nx, 
        int                      ny,
        double complex       *   pixels)
{
    cpl_image * self;

    cpl_ensure(pixels != NULL, CPL_ERROR_NULL_INPUT, NULL);

    self = cpl_image_wrap_(nx, ny, CPL_TYPE_DOUBLE_COMPLEX, pixels);

    if (self == NULL) (void)cpl_error_set_where(cpl_func);

    return self;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Create a float complex image using an existing pixel buffer.
  @param    nx          Size in x
  @param    ny          Size in y
  @param    pixels      float complex * pixel data.
  @return   1 newly allocated cpl_image or NULL on error
  @see      cpl_image_wrap_double_complex()
 */
/*----------------------------------------------------------------------------*/
cpl_image * cpl_image_wrap_float_complex(
        int             nx, 
        int             ny,
        float complex       *   pixels)
{
    cpl_image * self;

    cpl_ensure(pixels != NULL, CPL_ERROR_NULL_INPUT, NULL);

    self = cpl_image_wrap_(nx, ny, CPL_TYPE_FLOAT_COMPLEX, pixels);

    if (self == NULL) (void)cpl_error_set_where(cpl_func);

    return self;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Create an integer image using an existing pixel buffer.
  @param    nx          Size in x
  @param    ny          Size in y
  @param    pixels      int * pixel data.
  @return   1 newly allocated cpl_image or NULL on error 
  @see      cpl_image_wrap_double()
 */
/*----------------------------------------------------------------------------*/
cpl_image * cpl_image_wrap_int(
        int           nx, 
        int           ny,
        int       *   pixels)
{
    cpl_image * self;

    cpl_ensure(pixels != NULL, CPL_ERROR_NULL_INPUT, NULL);

    self = cpl_image_wrap_(nx, ny, CPL_TYPE_INT, pixels);

    if (self == NULL) (void)cpl_error_set_where(cpl_func);

    return self;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Load an image from a FITS file.
  @param    filename    Name of the file to load from.
  @param    im_type     Type of the created image       
  @param    pnum        Plane number in the Data Unit (0 for first)
  @param    xtnum       Extension number in the file (0 for primary HDU)
  @return   1 newly allocated image or NULL on error

  This function loads an image from a FITS file (NAXIS=2 or 3).

  The returned image has to be deallocated with cpl_image_delete().

  The passed type for the output image can be : CPL_TYPE_FLOAT, 
  CPL_TYPE_DOUBLE or CPL_TYPE_INT.

  This type is there to specify the type of the cpl_image that will be created
  by the function. It is totally independant from the way the data are stored
  in the FITS file. A FITS image containg float pixels can be loaded as a
  cpl_image of type double. In this case, the user would specify 
  CPL_TYPE_DOUBLE as im_type.

  Additionally, CPL_TYPE_UNSPECIFIED can be passed to inherit the FITS file
  type. The type of the image is defined based on the BITPIX information of the
  FITS file. After a successful call, the type of the created image can be
  accessed via cpl_image_get_type().
  
  'xtnum' specifies from which extension the image should be loaded.
  This could be 0 for the main data section (files without extension),
  or any number between 1 and N, where N is the number of extensions
  present in the file.

  The requested plane number runs from 0 to nplanes-1, where nplanes is
  the number of planes present in the requested data section.

  The created image has an empty bad pixel map.
  
  Examples:
  @code
  // Load as a float image the only image in FITS file (a.fits) without ext. 
  // and NAXIS=2.
  cpl_image * im = cpl_image_load("a.fits", CPL_TYPE_FLOAT, 0, 0);
  // Load as a double image the first plane in a FITS cube (a.fits) without 
  // extension, NAXIS=3 and NAXIS3=128
  cpl_image * im = cpl_image_load("a.fits", CPL_TYPE_DOUBLE, 0, 0);
  // Load as an integer image the third plane in a FITS cube (a.fits) without 
  // extension, NAXIS=3 and NAXIS3=128
  cpl_image * im = cpl_image_load("a.fits", CPL_TYPE_INT, 2, 0);
  // Load as a double image the first plane from extension 5
  cpl_image * im = cpl_image_load("a.fits", CPL_TYPE_DOUBLE, 0, 5);
  // Load as a double image the third plane in extension 5
  cpl_image * im = cpl_image_load("a.fits", CPL_TYPE_DOUBLE, 2, 5);
  @endcode

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_FILE_IO if the file cannot be opened or does not exist
  - CPL_ERROR_BAD_FILE_FORMAT if the data cannot be loaded from the file 
  - CPL_ERROR_INVALID_TYPE if the passed image type is not supported
  - CPL_ERROR_ILLEGAL_INPUT if the passed extension number is negative
  - CPL_ERROR_DATA_NOT_FOUND if the specified extension has no image data
 */
/*----------------------------------------------------------------------------*/
cpl_image * cpl_image_load(const char  *   filename,
                           cpl_type        im_type,
                           int             pnum,
                           int             xtnum)
{
    cpl_image * self = cpl_image_load_one(filename, im_type, pnum, xtnum,
                                          CPL_FALSE, 0, 0, 0, 0);

    if (self == NULL) cpl_error_set_where(cpl_func);

    return self;
}


/*----------------------------------------------------------------------------*/
/**
  @internal
  @brief  Load an image from a FITS file.
  @param  filename    Name of the file to load from.
  @param  im_type     Type of the created image       
  @param  pnum        Plane number in the Data Unit (0 for first)
  @param  xtnum       Extension number in the file (0 for primary HDU)
  @param  do_window   True for (and only for) a windowed load
  @param  llx         Lower left  x position (FITS convention, 1 for leftmost)
  @param  lly         Lower left  y position (FITS convention, 1 for lowest)
  @param  urx         Upper right x position (FITS convention)
  @param  ury         Upper right y position (FITS convention)
  @return 1 newly allocated image or NULL on error
  @see cpl_image_load()
*/
/*----------------------------------------------------------------------------*/
static cpl_image * cpl_image_load_one(const char * filename,
                                      cpl_type     im_type,
                                      int          pnum,
                                      int          xtnum,
                                      cpl_boolean  do_window,
                                      int          llx,
                                      int          lly,
                                      int          urx,
                                      int          ury)
{

    int         error = 0;
    int         naxis = 0;
    long int    naxes[3] ={0, 0, 0}; /* May read from Data Unit w. NAXIS=[23] */
    cpl_type    pix_type = im_type;
    fitsfile  * fptr;
    cpl_image * self;

    /* FIXME: Version 3.24 of fits_open_diskfile() seg-faults on NULL.
       If ever fixed in CFITSIO, this check should be removed */
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(xtnum   >= 0,     CPL_ERROR_ILLEGAL_INPUT, NULL);

    if (fits_open_diskfile(&fptr, filename, READONLY, &error)) {
        /* If the file did not exist:    error = 104 (FILE_NOT_OPENED) */
        /* If the file had permission 0: error = 104 (FILE_NOT_OPENED) */
        /* If the file was empty:        error = 107 (END_OF_FILE) */
        /* If the file was a directory:  error = 108 (READ_ERROR) */
        (void)cpl_error_set_fits(CPL_ERROR_FILE_IO, error, fits_open_diskfile,
                                 "filename='%s', im_type=%u, pnum=%d, xtnum=%d",
                                 filename, im_type, pnum, xtnum);
        return NULL;
    }

    self = cpl_image_load_(fptr, &naxis, naxes, &pix_type, filename, pnum,
                           xtnum, do_window, llx, lly, urx, ury);

    if (fits_close_file(fptr, &error)) {
        cpl_image_delete(self);
        self = NULL;
        (void)cpl_error_set_fits(CPL_ERROR_BAD_FILE_FORMAT, error,
                                 fits_close_file, "filename='%s', "
                                 "im_type=%u, pnum=%d, xtnum=%d",
                                 filename, im_type, pnum, xtnum);
    } else if (self == NULL) {
        cpl_error_set_where(cpl_func);
    }

    return self;
}


/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief  Load an image from a FITS file.
  @param  filename    Name of the file to load from.
  @param  im_type     Type of the created image       
  @param  pnum        Plane number in the Data Unit (0 for first)
  @param  xtnum       Extension number in the file.
  @param  llx         Lower left  x position (FITS convention, 1 for leftmost)
  @param  lly         Lower left  y position (FITS convention, 1 for lowest)
  @param  urx         Upper right x position (FITS convention)
  @param  ury         Upper right y position (FITS convention)
  @return 1 newly allocated image or NULL on error
  @see    cpl_image_load()

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_FILE_IO if the file does not exist
  - CPL_ERROR_BAD_FILE_FORMAT if the data cannot be loaded from the file 
  - CPL_ERROR_INVALID_TYPE if the passed image type is not supported
  - CPL_ERROR_ILLEGAL_INPUT if the passed position is invalid
 */
/*----------------------------------------------------------------------------*/
cpl_image * cpl_image_load_window(const char  *   filename,
                                  cpl_type        im_type,
                                  int             pnum,
                                  int             xtnum,
                                  int             llx,
                                  int             lly,
                                  int             urx,
                                  int             ury)
{

    cpl_image * self = cpl_image_load_one(filename, im_type, pnum, xtnum,
                                          CPL_TRUE, llx, lly, urx, ury);

    if (self == NULL) cpl_error_set_where(cpl_func);

    return self;

}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Create an int image from a mask
  @param    mask    the original mask
  @return   1 newly allocated cpl_image or NULL on error 

  The created image is of type CPL_TYPE_INT.
  
  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_image * cpl_image_new_from_mask(const cpl_mask * mask)
{
    const int          nx   = cpl_mask_get_size_x(mask);
    const int          ny   = cpl_mask_get_size_y(mask);
    const cpl_binary * data = cpl_mask_get_data_const(mask);
    cpl_image        * out;
    int              * pout;
    int                i;


    cpl_ensure(mask, CPL_ERROR_NULL_INPUT, NULL);

    /* Create the INT image */
    pout = (int*)cpl_malloc((size_t)nx * ny * sizeof(*pout));
    out = cpl_image_wrap_(nx, ny, CPL_TYPE_INT, pout);

    /* Fill the image */
    for (i=0; i < nx * ny; i++) {
        pout[i] = (data[i] == CPL_BINARY_0) ? 0 : 1;
    }

    return out;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Labelise a mask to differentiate different objects
  @param    in      mask to labelise
  @param    nbobjs  number of objects found.
  @return   A newly allocated label image or NULL on error
  
  This function labelises all blobs in a mask. All 4-neighbour connected zones 
  set to 1 in the input mask will end up in the returned integer image as zones
  where all pixels are set to the same (unique for this blob in this image) 
  label.
  A non-recursive flood-fill is applied to label the zones. The flood-fill is 
  dimensioned by the number of lines in the image, and the maximal number of
  lines possibly covered by a blob.
  The returned image must be deallocated with cpl_image_delete()

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if the input mask is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_image * cpl_image_labelise_mask_create(const cpl_mask * in,
                                           int            * nbobjs)
{
    cpl_image        * intimage;
    void             * fftemp;
    int              * pio;
    int                label;
    int                offpos;
    const int          nx  = cpl_mask_get_size_x(in);
    const int          ny  = cpl_mask_get_size_y(in);
    const cpl_binary * pin = cpl_mask_get_data_const(in);
    int                i, j;

    /* Test entries */
    cpl_ensure(in != NULL, CPL_ERROR_NULL_INPUT, NULL);

    /* Create output integer image and initialise it with the input map */
    pio = (int*)cpl_malloc((size_t)nx * ny * sizeof(*pio));
    cpl_ensure(pio != NULL, CPL_ERROR_ILLEGAL_INPUT, NULL);
    intimage = cpl_image_wrap_(nx, ny, CPL_TYPE_INT, pio);

    for (i=0; i<nx*ny; i++) {
        pio[i] = (pin[i] == CPL_BINARY_0) ? 0 : -1;
    }

    /* Now work on intimage */

    /* Allocate temporary work-space for cpl_image_floodfill() */
    fftemp = cpl_malloc(FFSTACK_BYTES(intimage));

    label = 1;
    for (j=0; j<intimage->ny; j++) {
        offpos = j*intimage->nx;
        for (i=0; i<intimage->nx; i++) {
            /* Look up if unprocessed pixel */
            if (pio[i+offpos]==(int)-1) {
                /* Flood fill from this pixel with the assigned label */
                cpl_image_floodfill(intimage, fftemp, i, j, label);
                label++;
            }
        }
    }
    cpl_free(fftemp);

    if (nbobjs!=NULL) (*nbobjs)=label-1;

    return intimage;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Get the image type
  @param    img     a cpl_image object 
  @return   The image type or CPL_TYPE_INVALID on NULL input.

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_type cpl_image_get_type(const cpl_image * img)
{
    cpl_ensure(img, CPL_ERROR_NULL_INPUT, 
            CPL_TYPE_INVALID);
    return img->type;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Get the image x size
  @param    img     a cpl_image object 
  @return   The image x size, or -1 on NULL input

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
int cpl_image_get_size_x(const cpl_image * img)
{
    cpl_ensure(img, CPL_ERROR_NULL_INPUT, -1);
    return img->nx;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Get the image y size
  @param    img     a cpl_image object 
  @return   The image y size, or -1 on NULL input

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
int cpl_image_get_size_y(const cpl_image * img)
{
    cpl_ensure(img, CPL_ERROR_NULL_INPUT, -1);
    return img->ny;
}

#define CPL_OPERATION CPL_IMAGE_IO_GET
/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Get the value of a pixel at a given position
  @param    image          Input image.
  @param    xpos           Pixel x position (FITS convention, 1 for leftmost)
  @param    ypos           Pixel y position (FITS convention, 1 for lowest)
  @param    pis_rejected   1 if the pixel is bad, 0 if good, negative on error
  @return   The pixel value (cast to a double) or undefined if *pis_rejected

  The return value is defined if the pixel is not flagged as rejected, i. e.
  when *pis_rejected == 0.

  In case of an error, the #_cpl_error_code_ code is set.

  Images can be CPL_TYPE_FLOAT, CPL_TYPE_INT or CPL_TYPE_DOUBLE.

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_INVALID_TYPE if the passed image type is not supported
 */ 
/*----------------------------------------------------------------------------*/
double cpl_image_get(
        const cpl_image * image,
        int               xpos,
        int               ypos,
        int             * pis_rejected) 
{
    double value;
    int    pos;

    /* Check entries */
    cpl_ensure(pis_rejected != NULL, CPL_ERROR_NULL_INPUT, -1.0);

    /* This call will check the validity of image, xpos and ypos */
    *pis_rejected = cpl_image_is_rejected(image, xpos, ypos);
    cpl_ensure( *pis_rejected >= 0,  cpl_error_get_code(), -2.0);

    /* The pixel is flagged as rejected */
    if (*pis_rejected) return 0.0;

    assert( image->pixels );

    pos = (xpos - 1) + image->nx * (ypos - 1);

    /* Get the value */
    switch (image->type) {
#define CPL_CLASS CPL_CLASS_DOUBLE
#include "cpl_image_io_body.h"
#undef CPL_CLASS

#define CPL_CLASS CPL_CLASS_FLOAT
#include "cpl_image_io_body.h"
#undef CPL_CLASS

#define CPL_CLASS CPL_CLASS_INT
#include "cpl_image_io_body.h"
#undef CPL_CLASS

        default:
            cpl_ensure(0, CPL_ERROR_INVALID_TYPE, -3);
    }

    return value;

}
#undef CPL_OPERATION

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Get the value of a complex pixel at a given position
  @param    image          Input complex image.
  @param    xpos           Pixel x position (FITS convention, 1 for leftmost)
  @param    ypos           Pixel y position (FITS convention, 1 for lowest)
  @param    pis_rejected   1 if the pixel is bad, 0 if good, negative on error
  @return   The pixel value (cast to a double complex) or
     undefined if *pis_rejected
  @see cpl_image_get()
  @note This function is available iff the application includes complex.h

  The return value is defined if the pixel is not flagged as rejected, i. e.
  when *pis_rejected == 0.

  In case of an error, the #_cpl_error_code_ code is set.

  Images can be CPL_TYPE_FLOAT_COMPLEX or CPL_TYPE_DOUBLE_COMPLEX.

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_INVALID_TYPE if the passed image type is not supported
 */ 
/*----------------------------------------------------------------------------*/
double complex cpl_image_get_complex(
        const cpl_image * image,
        int               xpos,
        int               ypos,
        int             * pis_rejected) 
{
    double complex value;
    int    pos;

    /* Check entries */
    cpl_ensure(pis_rejected != NULL, CPL_ERROR_NULL_INPUT, -1.0);

    cpl_ensure(image->type == CPL_TYPE_FLOAT_COMPLEX ||
           image->type == CPL_TYPE_DOUBLE_COMPLEX,
           CPL_ERROR_INVALID_TYPE, 0.0);

    /* This call will check the validity of image, xpos and ypos */
    *pis_rejected = cpl_image_is_rejected(image, xpos, ypos);
    cpl_ensure( *pis_rejected >= 0,  cpl_error_get_code(), -2.0);

    /* The pixel is flagged as rejected */
    if (*pis_rejected) return 0.0;

    assert( image->pixels );

    pos = (xpos - 1) + image->nx * (ypos - 1);

    if (image-> type == CPL_TYPE_FLOAT_COMPLEX) {
        const float complex * pi = (const float complex *)image->pixels;
        value = (double complex)pi[pos];
    } else {
        const double complex * pi = (const double complex *)image->pixels;
        value = pi[pos];
    }

    return value;
}


/*----------------------------------------------------------------------------*/
/**
  @internal
  @ingroup cpl_image
  @brief  Fill an image of type integer with a constant
  @param  self  The image to fill
  @param  value The value to fill with
  @return The #_cpl_error_code_ or CPL_ERROR_NONE
  @note   Any bad pixels are accepted

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT    if an input pointer is NULL
  - CPL_ERROR_TYPE_MISMATCH if the image type is not int
 */
/*----------------------------------------------------------------------------*/
cpl_error_code cpl_image_fill_int(cpl_image * self, int value)
{

    cpl_ensure_code(self       != NULL,         CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->type == CPL_TYPE_INT, CPL_ERROR_TYPE_MISMATCH);

    if (value == 0) {
        (void)memset(self->pixels, 0, self->nx * sizeof(int) * self->ny);
    } else {
        int i = self->nx * self->ny;

        do {
            ((int*)self->pixels)[--i] = value;
        } while (i);
    }

    return CPL_ERROR_NONE;
}


#define CPL_OPERATION CPL_IMAGE_COMPLEX

#undef CPL_COMPLEX_OPERATOR_F
#undef CPL_COMPLEX_OPERATOR

/*----------------------------------------------------------------------------*/
/**
  @internal
  @ingroup cpl_image
  @brief    Extract the real part image of a complex image
  @param    image       input image
  @return   a newly allocated real image or NULL on error

  Image can be CPL_TYPE_FLOAT_COMPLEX or CPL_TYPE_DOUBLE_COMPLEX.

  The returned image must be deallocated using cpl_image_delete().

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if the input pointer is NULL
  - CPL_ERROR_INVALID_TYPE if the passed image type is non-complex
 */ 
/*----------------------------------------------------------------------------*/
cpl_image * cpl_image_extract_real(const cpl_image * image)
{
#define CPL_COMPLEX_OPERATOR_F crealf
#define CPL_COMPLEX_OPERATOR   creal
#include "cpl_image_io_body.h"
#undef CPL_COMPLEX_OPERATOR_F
#undef CPL_COMPLEX_OPERATOR
}

/*----------------------------------------------------------------------------*/
/**
  @internal
  @ingroup cpl_image
  @brief    Extract the imaginary part image of a complex image
  @param    image       input image
  @return   a newly allocated imaginary image or NULL on error

  Image can be CPL_TYPE_FLOAT_COMPLEX or CPL_TYPE_DOUBLE_COMPLEX.

  The returned image must be deallocated using cpl_image_delete().

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if the input pointer is NULL
  - CPL_ERROR_INVALID_TYPE if the passed image type is non-complex
 */ 
/*----------------------------------------------------------------------------*/
cpl_image * cpl_image_extract_imag(const cpl_image * image)
{
#define CPL_COMPLEX_OPERATOR_F cimagf
#define CPL_COMPLEX_OPERATOR   cimag
#include "cpl_image_io_body.h"
#undef CPL_COMPLEX_OPERATOR_F
#undef CPL_COMPLEX_OPERATOR
}

/*----------------------------------------------------------------------------*/
/**
  @internal
  @ingroup cpl_image
  @brief    Extract the absolute part (modulus) of a complex image
  @param    image       input image
  @return   a newly allocated imaginary image or NULL on error

  Image can be CPL_TYPE_FLOAT_COMPLEX or CPL_TYPE_DOUBLE_COMPLEX.

  The returned image must be deallocated using cpl_image_delete().

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if the input pointer is NULL
  - CPL_ERROR_INVALID_TYPE if the passed image type is non-complex
 */ 
/*----------------------------------------------------------------------------*/
cpl_image * cpl_image_extract_mod(const cpl_image * image)
{
#define CPL_COMPLEX_OPERATOR_F cabsf
#define CPL_COMPLEX_OPERATOR   cabs
#include "cpl_image_io_body.h"
#undef CPL_COMPLEX_OPERATOR_F
#undef CPL_COMPLEX_OPERATOR
}

/*----------------------------------------------------------------------------*/
/**
  @internal
  @ingroup cpl_image
  @brief    Extract the argument (phase) of a complex image
  @param    image       input image
  @return   a newly allocated imaginary image or NULL on error

  Image can be CPL_TYPE_FLOAT_COMPLEX or CPL_TYPE_DOUBLE_COMPLEX.

  The returned image must be deallocated using cpl_image_delete().

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if the input pointer is NULL
  - CPL_ERROR_INVALID_TYPE if the passed image type is non-complex
 */ 
/*----------------------------------------------------------------------------*/
cpl_image * cpl_image_extract_phase(const cpl_image * image)
{
#define CPL_COMPLEX_OPERATOR_F cargf
#define CPL_COMPLEX_OPERATOR   carg
#include "cpl_image_io_body.h"
#undef CPL_COMPLEX_OPERATOR_F
#undef CPL_COMPLEX_OPERATOR
}
#undef CPL_OPERATION

#define CPL_OPERATION CPL_IMAGE_IO_SET
/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Set the pixel at the given position to the given value
  @param    image       input image.
  @param    xpos        Pixel x position (FITS convention, 1 for leftmost)
  @param    ypos        Pixel y position (FITS convention, 1 for lowest)
  @param    value       New pixel value 
  @return   CPL_ERROR_NONE or the relevant #_cpl_error_code_ on error

  Images can be CPL_TYPE_FLOAT, CPL_TYPE_INT, CPL_TYPE_DOUBLE.

  If the pixel is flagged as rejected, this flag is removed.

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if the passed position is not in the image
  - CPL_ERROR_INVALID_TYPE if the passed image type is not supported
 */ 
/*----------------------------------------------------------------------------*/
cpl_error_code cpl_image_set(
        cpl_image * image,
        int         xpos,
        int         ypos,
        double      value) 
{
    int    pos;

    /* Check entries */
    cpl_ensure_code(image != NULL,     CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(xpos >= 1,         CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_ensure_code(ypos >= 1,         CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_ensure_code(xpos <= image->nx, CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_ensure_code(ypos <= image->ny, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    assert( image->pixels );

    pos = (xpos - 1) + image->nx * (ypos - 1);

    /* Get the value */
    switch (image->type) {
#define CPL_CLASS CPL_CLASS_DOUBLE
#include "cpl_image_io_body.h"
#undef CPL_CLASS

#define CPL_CLASS CPL_CLASS_FLOAT
#include "cpl_image_io_body.h"
#undef CPL_CLASS

#define CPL_CLASS CPL_CLASS_INT
#include "cpl_image_io_body.h"
#undef CPL_CLASS
        default:
            return cpl_error_set_(CPL_ERROR_INVALID_TYPE);
    }

    return cpl_image_accept(image, xpos, ypos)
        ? cpl_error_set_where_() : CPL_ERROR_NONE;

}
#undef CPL_OPERATION

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Set the pixel at the given position to the given complex value
  @param    image       input image.
  @param    xpos        Pixel x position (FITS convention, 1 for leftmost)
  @param    ypos        Pixel y position (FITS convention, 1 for lowest)
  @param    value       New pixel value 
  @return   CPL_ERROR_NONE or the relevant #_cpl_error_code_ on error
  @see cpl_image_set()
  @note This function is available iff the application includes complex.h

  Images can be CPL_TYPE_FLOAT_COMPLEX or CPL_TYPE_DOUBLE_COMPLEX.

  If the pixel is flagged as rejected, this flag is removed.

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if the passed position is not in the image
  - CPL_ERROR_INVALID_TYPE if the passed image type is not supported
 */ 
/*----------------------------------------------------------------------------*/
cpl_error_code cpl_image_set_complex(
        cpl_image *    image,
        int            xpos,
        int            ypos,
        double complex value) 
{
    int    pos;

    /* Check entries */
    cpl_ensure_code(image != NULL,     CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(xpos >= 1,         CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_ensure_code(ypos >= 1,         CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_ensure_code(xpos <= image->nx, CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_ensure_code(ypos <= image->ny, CPL_ERROR_ACCESS_OUT_OF_RANGE);
/*    cpl_ensure_code(image -> type & CPL_TYPE_COMPLEX,
            CPL_ERROR_INVALID_TYPE);*/

    assert( image->pixels );

    pos = (xpos - 1) + image->nx * (ypos - 1);

    if (image-> type == CPL_TYPE_FLOAT_COMPLEX) {
        float complex * pi = (float complex *)image->pixels;
        pi[pos] = (float complex)value;
    } else {
        double complex * pi = (double complex *)image->pixels;
        pi[pos] = value;
    }

    return cpl_image_accept(image, xpos, ypos)
        ? cpl_error_set_where_() : CPL_ERROR_NONE;

}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Gets the pixel data.
  @param    img  Image to query.
  @return   A pointer to the image pixel data or NULL on error.
  
  The returned pointer refers to already allocated data. 

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
void * cpl_image_get_data(cpl_image * img)
{
    cpl_ensure(img, CPL_ERROR_NULL_INPUT, NULL);
    return img->pixels;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Gets the pixel data.
  @param    img  Image to query.
  @return   A pointer to the image pixel data or NULL on error.
  @see cpl_image_get_data
 */
/*----------------------------------------------------------------------------*/
const void * cpl_image_get_data_const(const cpl_image * img)
{
    cpl_ensure(img, CPL_ERROR_NULL_INPUT, NULL);
    return img->pixels;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Remove the bad pixel map from the image
  @param    self  image to process
  @return   A pointer to the cpl_mask of bad pixels, or NULL
  @note NULL is returned if the image has no bad pixel map
  @note The returned mask must be deallocated using cpl_mask_delete().
  @see cpl_image_get_bpm_const()

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_mask * cpl_image_unset_bpm(cpl_image * self)
{
    cpl_mask * bpm;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    bpm = self->bpm;

    self->bpm = NULL;
 
    return bpm;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Gets the bad pixels map
  @param    img  Image to query.
  @return   A pointer to the mask identifying the bad pixels or NULL.
  
  The returned pointer refers to already allocated data. 
  If the bad pixel map is NULL, an empty one is created.

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_mask * cpl_image_get_bpm(cpl_image * img)
{
    cpl_ensure(img != NULL, CPL_ERROR_NULL_INPUT, NULL);

    if (img->bpm == NULL) img->bpm = cpl_mask_new(img->nx, img->ny);

    return img->bpm;
}
    
/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Gets the bad pixels map
  @param    img  Image to query.
  @return   A pointer to the mask identifying the bad pixels or NULL.
  @note NULL is returned if the image has no bad pixel map
  @see cpl_image_get_bpm

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
const cpl_mask * cpl_image_get_bpm_const(const cpl_image * img)
{
    cpl_ensure(img != NULL, CPL_ERROR_NULL_INPUT, NULL);

    return img->bpm;
}
    
#define CPL_OPERATION CPL_IMAGE_IO_GET_PIXELS
#define CPL_CONST /* const */
/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Get the data as a double array
  @param    img     a cpl_image object 
  @return   pointer to the double data array or NULL on error.
  
  The returned pointer refers to already allocated data. 

  The pixels are stored in a one dimensional array. The pixel value PIXVAL at 
  position (i, j) in the image - (0, 0) is the lower left pixel, i gives the 
  column position from left to right, j gives the row position from bottom to
  top - is given by :
  PIXVAL = array[i + j*nx];
  where nx is the x size of the image and array is the data array returned by 
  this function.
  array can be used to access or modify the pixel value in the image.
  
  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_TYPE_MISMATCH if the passed image type is not double
 */
/*----------------------------------------------------------------------------*/
double * cpl_image_get_data_double(cpl_image * img)
{
#define CPL_CLASS CPL_CLASS_DOUBLE
#include "cpl_image_io_body.h"
#undef CPL_CLASS
}


/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Get the data as a float array
  @param    img     a cpl_image object 
  @return   pointer to the float data array or NULL on error.
  @see      cpl_image_get_data_double()
 */
/*----------------------------------------------------------------------------*/
float * cpl_image_get_data_float(cpl_image * img)
{
#define CPL_CLASS CPL_CLASS_FLOAT
#include "cpl_image_io_body.h"
#undef CPL_CLASS
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Get the data as a double complex array
  @param    img     a cpl_image object 
  @return   pointer to the double complex data array or NULL on error.
  @see      cpl_image_get_data_double()
  @note This function is available iff the application includes complex.h

 */
/*----------------------------------------------------------------------------*/
double complex * cpl_image_get_data_double_complex(cpl_image * img)
{
#define CPL_CLASS CPL_CLASS_DOUBLE_COMPLEX
#include "cpl_image_io_body.h"
#undef CPL_CLASS
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Get the data as a float complex array
  @param    img     a cpl_image object 
  @return   pointer to the float complex data array or NULL on error.
  @see      cpl_image_get_data_double_complex()
 */
/*----------------------------------------------------------------------------*/
float complex * cpl_image_get_data_float_complex(cpl_image * img)
{
#define CPL_CLASS CPL_CLASS_FLOAT_COMPLEX
#include "cpl_image_io_body.h"
#undef CPL_CLASS
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Get the data as a integer array
  @param    img     a cpl_image object 
  @return   pointer to the integer data array or NULL on error.
  @see      cpl_image_get_data_double()
 */
/*----------------------------------------------------------------------------*/
int * cpl_image_get_data_int(cpl_image * img)
{
#define CPL_CLASS CPL_CLASS_INT
#include "cpl_image_io_body.h"
#undef CPL_CLASS
}
   
#undef CPL_CONST
#define CPL_CONST const

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Get the data as a double array
  @param    img     a cpl_image object 
  @return   pointer to the double data array or NULL on error.
  @see cpl_image_get_data_double
 */
/*----------------------------------------------------------------------------*/
const double * cpl_image_get_data_double_const(const cpl_image * img)
{
#define CPL_CLASS CPL_CLASS_DOUBLE
#include "cpl_image_io_body.h"
#undef CPL_CLASS
}


/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Get the data as a float array
  @param    img     a cpl_image object 
  @return   pointer to the float data array or NULL on error.
  @see      cpl_image_get_data_float()
 */
/*----------------------------------------------------------------------------*/
const float * cpl_image_get_data_float_const(const cpl_image * img)
{
#define CPL_CLASS CPL_CLASS_FLOAT
#include "cpl_image_io_body.h"
#undef CPL_CLASS
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Get the data as a double complex array
  @param    img     a cpl_image object 
  @return   pointer to the double complex data array or NULL on error.
  @see      cpl_image_get_data_double_complex()
 */
/*----------------------------------------------------------------------------*/
const double complex * 
cpl_image_get_data_double_complex_const(const cpl_image * img)
{
#define CPL_CLASS CPL_CLASS_DOUBLE_COMPLEX
#include "cpl_image_io_body.h"
#undef CPL_CLASS
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Get the data as a float complex array
  @param    img     a cpl_image object 
  @return   pointer to the float complex data array or NULL on error.
  @see      cpl_image_get_data_double_complex()
 */
/*----------------------------------------------------------------------------*/
const float complex * 
cpl_image_get_data_float_complex_const(const cpl_image * img)
{
#define CPL_CLASS CPL_CLASS_FLOAT_COMPLEX
#include "cpl_image_io_body.h"
#undef CPL_CLASS
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Get the data as a integer array
  @param    img     a cpl_image object 
  @return   pointer to the integer data array or NULL on error.
  @see      cpl_image_get_data_int()
 */
/*----------------------------------------------------------------------------*/
const int * cpl_image_get_data_int_const(const cpl_image * img)
{
#define CPL_CLASS CPL_CLASS_INT
#include "cpl_image_io_body.h"
#undef CPL_CLASS
}
#undef CPL_OPERATION
  

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Free memory associated to an cpl_image object.
  @param    d    Image to destroy.
  @return   void

  Frees all memory associated with a cpl_image.
  If the passed image is NULL, the function returns without doing
  anything.
 */
/*----------------------------------------------------------------------------*/
void cpl_image_delete(cpl_image * d)
{
    if (d == NULL) return;

    /* Delete pixels and bad pixel map */
    cpl_free(d->pixels);
    cpl_mask_delete(d->bpm);

    cpl_free(d);
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Free memory associated to an cpl_image object, but the pixel buffer.
  @param    d   Image to destroy.
  @return   A pointer to the data array or NULL if the input is NULL.

  Frees all memory associated to an cpl_image, except the pixel buffer.
  cpl_image_unwrap() is provided for images that are constructed by passing
  a pixel buffer to one of cpl_image_wrap_{double,float,int}().

  @note
     The pixel buffer must subsequently be deallocated. Failure to do so will
     result in a memory leak.
 */
/*----------------------------------------------------------------------------*/
void * cpl_image_unwrap(cpl_image * d)
{

    void * data;

    if (d == NULL) return NULL;

    /* Delete bad pixel map */
    cpl_mask_delete(d->bpm);

    data = (void *) d->pixels;

    cpl_free(d);

    return data;
}


/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Copy an image.
  @param    src    Source image.
  @return   1 newly allocated image, or NULL on error.
 
  Copy an image into a new image object. The pixels and the bad pixel map are
  also copied. The returned image must be deallocated using cpl_image_delete().

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_image * cpl_image_duplicate(const cpl_image * src)
{
    cpl_image * self;

    cpl_ensure(src != NULL, CPL_ERROR_NULL_INPUT, NULL);

    /* Allocate and copy the image properties into a new image
       - must update all (one or two) non-NULL pointer members */
    self = memcpy(cpl_malloc(sizeof(cpl_image)), src, sizeof(cpl_image));

    /* 1: Allocate and copy the pixel buffer */
    self->pixels = cpl_malloc((size_t)src->nx * (size_t)src->ny
                              * cpl_type_get_sizeof(src->type));
    (void)memcpy(self->pixels, src->pixels, (size_t)src->nx * (size_t)src->ny
                 * cpl_type_get_sizeof(src->type));

    /* 2: Duplicate the bad pixel map, if any */
    if (src->bpm != NULL) self->bpm = cpl_mask_duplicate(src->bpm);

    return self;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Convert a cpl_image to a given type
  @param    self  The image to convert
  @param    type  The destination type
  @return   the newly allocated cpl_image or NULL on error

  Casting to non-complex types is only supported for non-complex types.

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the passed type is invalid
  - CPL_ERROR_TYPE_MISMATCH if the passed image type is complex and requested
  casting type is non-complex.
  - CPL_ERROR_INVALID_TYPE if the passed image type is not supported
 */
/*----------------------------------------------------------------------------*/
cpl_image * cpl_image_cast(const cpl_image * self,
                           cpl_type          type)
{
    cpl_image            * new_im;
    const float          * pfi; 
    float                * pfo;
    const double         * pdi;
    double               * pdo;
    const int            * pii;
    int                  * pio;
    const float complex  * pfci;
    float complex        * pfco;
    const double complex * pdci;
    double complex       * pdco;
    int                    i;

    /* Check entries */
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(type != CPL_TYPE_INVALID, CPL_ERROR_ILLEGAL_INPUT, NULL);
    
    /* The image has already the destination type */
    if (self->type == type) return cpl_image_duplicate(self);

    /* Switch on the requested type */
    switch (type) {
        case CPL_TYPE_FLOAT:
            pfo = (float*)cpl_malloc((size_t)self->nx * self->ny * sizeof(*pfo));
            new_im = cpl_image_wrap_(self->nx, self->ny, type, pfo);
            /* Switch on the original type */
            switch(self->type) {
                case CPL_TYPE_DOUBLE:
                    pdi = (const double*)self->pixels;
                    for (i=0; i<self->nx*self->ny; i++) pfo[i] = (float)pdi[i];
                    break;
                case CPL_TYPE_INT:
                    pii = (const int*)self->pixels;
                    for (i=0; i<self->nx*self->ny; i++) pfo[i] = (float)pii[i];
                    break;
                case CPL_TYPE_FLOAT_COMPLEX:
                case CPL_TYPE_DOUBLE_COMPLEX:
                    cpl_image_delete(new_im);
                    cpl_ensure(0, CPL_ERROR_TYPE_MISMATCH, NULL);
            break;
                default:
                    cpl_image_delete(new_im);
                    cpl_ensure(0, CPL_ERROR_INVALID_TYPE, NULL);
            }
            break;
        case CPL_TYPE_DOUBLE:
            pdo = (double*)cpl_malloc((size_t)self->nx * self->ny * sizeof(*pdo));
            new_im = cpl_image_wrap_(self->nx, self->ny, type, pdo);
            /* Switch on the original type */
            switch(self->type) {
                case CPL_TYPE_FLOAT:
                    pfi = (const float*)self->pixels;
                    for (i=0; i<self->nx*self->ny; i++) pdo[i] = (double)pfi[i];
                    break;
                case CPL_TYPE_INT:
                    pii = (const int*)self->pixels;
                    for (i=0; i<self->nx*self->ny; i++) pdo[i] = (double)pii[i];
                    break;
                case CPL_TYPE_FLOAT_COMPLEX:
                case CPL_TYPE_DOUBLE_COMPLEX:
                    cpl_image_delete(new_im);
                    cpl_ensure(0, CPL_ERROR_TYPE_MISMATCH, NULL);
            break;
                default:
                    cpl_image_delete(new_im);
                    cpl_ensure(0, CPL_ERROR_INVALID_TYPE, NULL);
            }
            break;
        case CPL_TYPE_INT:
            pio = (int*)cpl_malloc((size_t)self->nx * self->ny * sizeof(*pio));
            new_im = cpl_image_wrap_(self->nx, self->ny, type, pio);
            /* Switch on the original type */
            switch(self->type) {
                case CPL_TYPE_FLOAT:
                    pfi = (const float*)self->pixels;
                    for (i=0; i<self->nx*self->ny; i++) pio[i] = (int)pfi[i];
                    break;
                case CPL_TYPE_DOUBLE:
                    pdi = (const double*)self->pixels;
                    for (i=0; i<self->nx*self->ny; i++) pio[i] = (int)pdi[i];
                    break;
                case CPL_TYPE_FLOAT_COMPLEX:
                case CPL_TYPE_DOUBLE_COMPLEX:
                    cpl_image_delete(new_im);
                    cpl_ensure(0, CPL_ERROR_TYPE_MISMATCH, NULL);
            break;
                default:
                    cpl_image_delete(new_im);
                    cpl_ensure(0, CPL_ERROR_INVALID_TYPE, NULL);
            }
            break;
        case CPL_TYPE_FLOAT_COMPLEX:
            pfco = (float complex*)cpl_malloc((size_t)self->nx * self->ny
                                              * sizeof(*pfco));
            new_im = cpl_image_wrap_(self->nx, self->ny, type, pfco);
            /* Switch on the original type */
            switch(self->type) {
            case CPL_TYPE_DOUBLE_COMPLEX:
                pdci = (const double complex *)self->pixels;
                for (i=0; i<self->nx*self->ny; i++)
                    pfco[i] = (float complex)pdci[i];
                break;
            case CPL_TYPE_INT:
                pii = (const int *)self->pixels;
                for (i=0; i<self->nx*self->ny; i++)
                    pfco[i] = (float complex)pii[i];
                break;
            case CPL_TYPE_FLOAT:
                pfi = (const float*)self->pixels;
                for (i=0; i<self->nx*self->ny; i++)
                    pfco[i] = (float complex)pfi[i];
                break;
            case CPL_TYPE_DOUBLE:
                pdi = (const double*)self->pixels;
                for (i=0; i<self->nx*self->ny; i++)
                    pfco[i] = (float complex)pdi[i];
                break;
            default:
                cpl_image_delete(new_im);
                cpl_ensure(0, CPL_ERROR_INVALID_TYPE, NULL);
            }
            break;
        case CPL_TYPE_DOUBLE_COMPLEX:
            pdco = (double complex*)cpl_malloc((size_t)self->nx * self->ny
                                              * sizeof(*pdco));
            new_im = cpl_image_wrap_(self->nx, self->ny, type, pdco);
            /* Switch on the original type */
            switch(self->type) {
            case CPL_TYPE_FLOAT_COMPLEX:
                pfci = (const float complex*)self->pixels;
                for (i=0; i<self->nx*self->ny; i++) 
                    pdco[i] = (double complex)pfci[i];
                break;
            case CPL_TYPE_INT:
                pii = (const int *)self->pixels;
                for (i=0; i<self->nx*self->ny; i++)
                    pdco[i] = (double complex)pii[i];
                break;
            case CPL_TYPE_FLOAT:
                pfi = (const float*)self->pixels;
                for (i=0; i<self->nx*self->ny; i++)
                    pdco[i] = (double complex)pfi[i];
                break;
            case CPL_TYPE_DOUBLE:
                pdi = (const double*)self->pixels;
                for (i=0; i<self->nx*self->ny; i++)
                    pdco[i] = (double complex)pdi[i];
                break;
            default:
                cpl_image_delete(new_im);
                cpl_ensure(0, CPL_ERROR_INVALID_TYPE, NULL);
            }
            break;
    default:
        cpl_ensure(0, CPL_ERROR_INVALID_TYPE, NULL);
    }

    /* Bad pixel map */
    if (self->bpm != NULL) new_im->bpm = cpl_mask_duplicate(self->bpm);
    
    return new_im;
}

#define CPL_OPERATION CPL_IMAGE_IO_SET_BADPIXEL
/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Set the bad pixels in an image to a fixed value
  @param    im  The image to modify.
  @param    a   The fixed value
  @return   the #_cpl_error_code_ or CPL_ERROR_NONE

  Images can be CPL_TYPE_FLOAT, CPL_TYPE_INT, CPL_TYPE_DOUBLE.

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT
  - CPL_ERROR_INVALID_TYPE if the passed image type is not supported
 */
/*----------------------------------------------------------------------------*/
cpl_error_code cpl_image_fill_rejected(
        cpl_image   *   im, 
        double          a)
{
    cpl_binary  *       pbpm;

    /* Check entries and Initialise */
    cpl_ensure_code(im, CPL_ERROR_NULL_INPUT);
    
    /* If no bad pixel */
    if (im->bpm == NULL) return CPL_ERROR_NONE;
    
    /* Return if no bad pixel map to update */
    if (cpl_mask_is_empty(im->bpm)) return CPL_ERROR_NONE;

    pbpm = cpl_mask_get_data(im->bpm);
    
    switch (im->type) {
#define CPL_CLASS CPL_CLASS_DOUBLE
#include "cpl_image_io_body.h"
#undef CPL_CLASS

#define CPL_CLASS CPL_CLASS_FLOAT
#include "cpl_image_io_body.h"
#undef CPL_CLASS

#define CPL_CLASS CPL_CLASS_INT
#include "cpl_image_io_body.h"
#undef CPL_CLASS

#define CPL_CLASS CPL_CLASS_DOUBLE_COMPLEX
#include "cpl_image_io_body.h"
#undef CPL_CLASS

#define CPL_CLASS CPL_CLASS_FLOAT_COMPLEX
#include "cpl_image_io_body.h"
#undef CPL_CLASS

      default:
        return cpl_error_set_(CPL_ERROR_INVALID_TYPE);
    }

    return CPL_ERROR_NONE;
}
#undef CPL_OPERATION

/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Save an image to a FITS file
  @param    self       Image to write to disk or NULL
  @param    filename   Name of the file to write
  @param    bpp        Bits per pixel
  @param    pl         Property list for the output header or NULL
  @param    mode       The desired output options
  @return   CPL_ERROR_NONE or the relevant #_cpl_error_code_ on error

  This function saves an image to a FITS file. If a property list
  is provided, it is written to the header where the image is written.
  The image may be NULL, in this case only the propertylist is written.
  
  The requested bits per pixel (bpp) follows the FITS convention. Possible
  values are CPL_BPP_8_UNSIGNED, CPL_BPP_16_SIGNED,
  CPL_BPP_32_SIGNED, CPL_BPP_IEEE_FLOAT, CPL_BPP_IEEE_DOUBLE.

  Additionally, the mode CPL_BPP_16_UNSIGNED is supported.
  In this case the data are saved as CPL_BPP_16_SIGNED, with 32768 subtracted
  from each pixel and BZERO set to 32768 as specified by the FITS standard.

  Supported image types are CPL_TYPE_DOUBLE, CPL_TYPE_FLOAT, CPL_TYPE_INT.

  Supported output modes are CPL_IO_CREATE (create a new file) and
  CPL_IO_EXTEND (append a new extension to an existing file)
 
  Note that in append mode the file must be writable (and do not take for
  granted that a file is writable just because it was created by the same
  application, as this depends from the system @em umask).
  
  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the bpp value or the mode is not supported
  - CPL_ERROR_INVALID_TYPE if the passed image type is not supported
  - CPL_ERROR_FILE_NOT_CREATED if the output file cannot be created
  - CPL_ERROR_FILE_IO if the data cannot be written to the file
 */
/*----------------------------------------------------------------------------*/
cpl_error_code cpl_image_save(const cpl_image        * self,
                              const char             * filename,
                              cpl_type_bpp             bpp,
                              const cpl_propertylist * pl,
                              unsigned                 mode)
{
    cpl_error_code error = CPL_ERROR_NONE;

    cpl_ensure_code(((mode & CPL_IO_CREATE) != 0) ^ 
            ((mode & CPL_IO_EXTEND) != 0),
            CPL_ERROR_ILLEGAL_INPUT);

    if (self == NULL) {
        error = cpl_propertylist_save(pl, filename, mode);
        cpl_ensure_code(!error, error); /* Propagate error, if any */
    } else {
        fitsfile        *  fptr;
        int                fio_status=0;
        const char       * badkeys = mode & CPL_IO_EXTEND
            ? CPL_FITS_BADKEYS_EXT  "|" CPL_FITS_COMPRKEYS
            : CPL_FITS_BADKEYS_PRIM "|" CPL_FITS_COMPRKEYS;
        const long naxes[]     = {self->nx, self->ny};
        const int  cfitsiotype = self->type == CPL_TYPE_DOUBLE ? TDOUBLE
            : (self->type == CPL_TYPE_FLOAT ? TFLOAT : TINT);


        cpl_ensure_code(filename != NULL, CPL_ERROR_NULL_INPUT);
        cpl_ensure_code(bpp==CPL_BPP_8_UNSIGNED  || bpp==CPL_BPP_16_SIGNED || 
                        bpp==CPL_BPP_16_UNSIGNED || bpp==CPL_BPP_32_SIGNED || 
                        bpp==CPL_BPP_IEEE_FLOAT  || bpp==CPL_BPP_IEEE_DOUBLE, 
                        CPL_ERROR_ILLEGAL_INPUT);
        cpl_ensure_code(mode <= CPL_IO_MAX, CPL_ERROR_ILLEGAL_INPUT);

        cpl_ensure_code(self->type == CPL_TYPE_DOUBLE ||
                        self->type == CPL_TYPE_FLOAT ||
                        self->type == CPL_TYPE_INT, CPL_ERROR_INVALID_TYPE);

        if (mode & CPL_IO_EXTEND) {
            /* Open the file */
            fits_open_diskfile(&fptr, filename, READWRITE, &fio_status);
            if (fio_status != 0) {
                return cpl_error_set_fits(CPL_ERROR_FILE_IO, fio_status,
                                          fits_open_diskfile,
                                          "filename='%s', bpp=%d, mode=%d",
                                          filename, bpp, mode);
            }
        } else {
            /* Create the file */
            char * sval = cpl_sprintf("!%s", filename);

            fits_create_file(&fptr, sval, &fio_status);
            cpl_free(sval);
            if (fio_status != 0) {
                int fio_status2 = 0;
                fits_close_file(fptr, &fio_status2);
                return cpl_error_set_fits(CPL_ERROR_FILE_IO, fio_status,
                                          fits_create_file, "filename='%s', "
                                          "bpp=%d, mode=%d", filename, bpp,
                                          mode);
            }
        }

        fits_create_img(fptr, bpp, 2, (long*)naxes, &fio_status);
    
        if (fio_status != 0) {
            int fio_status2 = 0;
            fits_close_file(fptr, &fio_status2);
            return cpl_error_set_fits(CPL_ERROR_FILE_IO, fio_status,
                                      fits_create_img, "filename='%s', bpp=%d"
                                      ", mode=%d", filename, bpp, mode);
        }
    
        if (mode & CPL_IO_CREATE) {
            /* Add DATE */
            fits_write_date(fptr, &fio_status);
            if (fio_status != 0) {
                int fio_status2 = 0;
                fits_close_file(fptr, &fio_status2);
                return cpl_error_set_fits(CPL_ERROR_FILE_IO, fio_status,
                                          fits_write_date, "filename='%s', "
                                          "bpp=%d, mode=%d", filename, bpp,
                                          mode);
            }
        }

        /* Add the property list */
        if (cpl_fits_add_properties(fptr, pl, badkeys) != CPL_ERROR_NONE) {
            fits_close_file(fptr, &fio_status);
            return cpl_error_set_(CPL_ERROR_ILLEGAL_INPUT);
        }

        /* Write the pixels */
        fits_write_img(fptr, cfitsiotype, 1, self->nx * self->ny,
                       (void*)self->pixels, &fio_status);

        if (fio_status != 0) {
            error = cpl_error_set_fits(CPL_ERROR_FILE_IO, fio_status,
                                       fits_write_img, "filename='%s', "
                                       "bpp=%d, mode=%d", filename, bpp, mode);
            fio_status = 0;
        }

        /* Close and write on disk */
        fits_close_file(fptr, &fio_status);

        /* Check  */
        if (fio_status != 0) {
            error = cpl_error_set_fits(CPL_ERROR_FILE_IO, fio_status,
                                       fits_close_file, "filename='%s', "
                                       "bpp=%d, mode=%d", filename, bpp, mode);
        }
    }
    
    return error;
}


/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Dump structural information of a CPL image
  @param    self    Image to dump
  @param    stream  Output stream, accepts @c stdout or @c stderr
  @return   CPL_ERROR_NONE or the relevant #_cpl_error_code_ on error

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_error_code cpl_image_dump_structure(const cpl_image * self, FILE * stream)
{

    const char * msg = "Image with %d X %d pixel(s) of type '%s' and %d bad "
        "pixel(s)\n";
    const int msgmin = strlen(msg) - 5;

    cpl_ensure_code(self   != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(stream != NULL, CPL_ERROR_NULL_INPUT);

    cpl_ensure_code( fprintf(stream,  msg,
                             self->nx, 
                             self->ny,
                             cpl_type_get_name(self->type),
                             self->bpm ? cpl_mask_count(self->bpm) : 0)
                     >= msgmin, CPL_ERROR_FILE_IO);

    return CPL_ERROR_NONE;

}
 
/*----------------------------------------------------------------------------*/
/**
  @ingroup cpl_image
  @brief    Dump pixel values in a CPL image
  @param    self    Image to dump
  @param    llx     Lower left x position (FITS convention, 1 for leftmost)
  @param    lly     Lower left y position (FITS convention, 1 for lowest)
  @param    urx     Specifies the window position
  @param    ury     Specifies the window position
  @param    stream  Output stream, accepts @c stdout or @c stderr
  @return   CPL_ERROR_NONE or the relevant #_cpl_error_code_ on error

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if the defined window is not in the image
  - CPL_ERROR_ILLEGAL_INPUT if the window definition is wrong (e.g llx > urx)
 */
/*----------------------------------------------------------------------------*/
cpl_error_code cpl_image_dump_window(const cpl_image * self, int llx, int lly,
                                     int urx, int ury, FILE * stream)
{
    const cpl_error_code err = CPL_ERROR_FILE_IO;
    int i, j;
    cpl_boolean has_bad;

    cpl_ensure_code(self   != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(stream != NULL, CPL_ERROR_NULL_INPUT);

    cpl_ensure_code(llx > 0 && llx <= self->nx && urx > 0 && urx <= self->nx,
           CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_ensure_code(lly > 0 && lly <= self->ny && ury > 0 && ury <= self->ny,
           CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_ensure_code(urx >= llx && ury >= lly, CPL_ERROR_ILLEGAL_INPUT);

    has_bad = self->bpm != NULL && !cpl_mask_is_empty(self->bpm);

    cpl_ensure_code( fprintf(stream, "#----- image: %d <= x <= %d, %d <= y <= "
                             "%d -----\n", llx, urx, lly, ury) > 0, err);

    cpl_ensure_code( fprintf(stream, "\tX\tY\tvalue\n") > 0, err);

    for (i = llx; i <= urx; i++) {
        for (j = lly; j <= ury; j++) {
            const char * badtxt = has_bad && cpl_mask_get(self->bpm, i, j)
                ? " (rejected)" : "";
            if (self->type == CPL_TYPE_INT) {
                const int * pint  = (const int*)self->pixels;
                const int   value = pint[(i-1) + (j-1) * self->nx];
                cpl_ensure_code( fprintf(stream, "\t%d\t%d\t%d%s\n", i, j,
                                         value, badtxt) > 0, err);
            } else if (self->type == CPL_TYPE_FLOAT_COMPLEX ||
               self->type == CPL_TYPE_DOUBLE_COMPLEX ) {
                int dummy;
                const double complex value = 
            cpl_image_get_complex(self, i, j, &dummy);
                cpl_ensure_code( fprintf(stream, "\t%d\t%d\t%g + %g I %s\n",
                     i, j, creal(value), cimag(value),
                     badtxt) > 0, err);
            } else {
                int dummy;
                const double value = cpl_image_get(self, i, j, &dummy);
                cpl_ensure_code( fprintf(stream, "\t%d\t%d\t%g%s\n", i, j,
                                         value, badtxt) > 0, err);
            }
        }
    }

    return CPL_ERROR_NONE;

}
 

#define FFSTACK_PUSH(Y, XL, XR, DY) \
    if (sp<stack+stacksz && Y+(DY)>=wy1 && Y+(DY)<=wy2) \
    {sp->y = Y; sp->xl = XL; sp->xr = XR; sp->dy = DY; sp++;}
#define FFSTACK_POP(Y, XL, XR, DY) \
    {sp--; Y = sp->y+(DY = sp->dy); XL = sp->xl; XR = sp->xr;}

/*----------------------------------------------------------------------------*/
/*
  @internal
  @ingroup cpl_image
  @brief    Fill a zone with label.
  @param    intimage    input label image
  @param    fftemp      Pre-allocated work-space
  @param    x           x position
  @param    y           y position
  @param    label       current label
  
  This code was pulled out the Usenet and freely adapted to cpl. 
  Credits - Paul Heckbert (posted on comp.graphics 28 Apr 1988)
  It is highly unreadable and makes use of goto and other fairly bad 
  programming practices, but works fine and fast.
 */
/*----------------------------------------------------------------------------*/
static void cpl_image_floodfill(cpl_image   * lab,
                                void        * fftemp,
                                int           x,
                                int           y,
                                int           label)
{
    struct { int y, xl, xr, dy; } * stack, * sp;
    int             wx1, wx2, wy1, wy2;
    int             l, x1, x2, dy;
    int             ov;
    const int       stacksz = FFSTACK_STACKSZ(lab);
    int         *   pii;

    stack = fftemp;
    sp = stack;
    wx1 = 0;
    wx2 = lab->nx-1;
    wy1 = 0;
    wy2 = lab->ny-1;
    pii = cpl_image_get_data_int(lab);
    ov = pii[x+y*lab->nx];
    if (ov==label || x<wx1 || x>wx2 || y<wy1 || y>wy2) return;
    FFSTACK_PUSH(y, x, x, 1);           /* needed in some cases */
    FFSTACK_PUSH(y+1, x, x, -1);        /* seed segment (popped 1st) */

    while (sp>stack) {
        /* pop segment off stack and fill a neighboring scan line */
        FFSTACK_POP(y, x1, x2, dy);
        /*
         * segment of scan line y-dy for x1<=x<=x2 was previously filled,
         * now explore adjacent pixels in scan line y
         */
        for (x=x1; x>=wx1 && pii[x+y*lab->nx]==ov; x--)
            pii[x+y*lab->nx] = label;
        if (x>=x1) goto skip;
        l = x+1;
        if (l<x1) FFSTACK_PUSH(y, l, x1-1, -dy);        /* leak on left? */
        x = x1+1;
        do {
            for (; x<=wx2 && pii[x+y*lab->nx]==ov; x++)
            pii[x+y*lab->nx] = label;
            FFSTACK_PUSH(y, l, x-1, dy);
            if (x>x2+1) FFSTACK_PUSH(y, x2+1, x-1, -dy);    /* leak on right? */
skip:       for (x++; x<=x2 && pii[x+y*lab->nx]!=ov; x++);
            l = x;
        } while (x<=x2);
    }
}


/*----------------------------------------------------------------------------*/
/*
  @internal
  @brief Internal image loading function
  @param fptr      CFITSIO structure of the already opened FITS file
  @param pnaxis    If it points to 0, set *pnaxis to NAXIS else use as such
  @param naxes     If 1st is 0, fill w. NAXIS[12[3]] else use as such
  @param ppix_type If points to _UNSPECIFIED, set else use as such
  @param filename  The named of the opened file (only for error messages)
  @param pnum      Plane number in the Data Unit (0 for first)
  @param hdumov    Relative extension number to move to first (0 for no move)
  @param do_window True for (and only for) a windowed load
  @param llx       Lower left x position (FITS convention, 1 for leftmost)
  @param lly       Lower left y position (FITS convention, 1 for lowest)
  @param urx       Specifies the window position
  @param ury       Specifies the window position
  @return   1 newly allocated image or NULL if image cannot be loaded.
  @see cpl_image_load_one()

  This function reads from an already opened FITS-file, which is useful when
  multiple images are to be loaded from the same file.

  To avoid repeated calls to determine NAXIS, NAXISi and optionally the
  CPL pixel-type most suitable for the the actual FITS data, the parameters
  pnaxis, naxes and ppix_type can be used both for input and for output.

  Also the extension number, hdumov, is relative to the most recently accessed
  HDU which is defined as zero for the main HDU. When the file is opened the
  most recently accessed HDU is defined to be the main HDU.

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_BAD_FILE_FORMAT if a CFITSIO call fails or if in auto-pixel mode
                              the CFITSIO data type is unsupported.
  - CPL_ERROR_DATA_NOT_FOUND if the specified extension has no image data.
                             This code is relied on as being part of the API!
  - CPL_ERROR_INCOMPATIBLE_INPUT if NAXIS is OK but the data unit is empty
  - CPL_ERROR_ILLEGAL_INPUT If the plane number is out of range, or in a windowed
                             read the window is invalid
  - CPL_ERROR_INVALID_TYPE if the passed image type is not supported

 */
/*----------------------------------------------------------------------------*/
cpl_image * cpl_image_load_(fitsfile   * fptr,
                            int        * pnaxis,
                            long int     naxes[],
                            cpl_type   * ppix_type,
                            const char * filename,
                            int          pnum,
                            int          hdumov,
                            cpl_boolean  do_window,
                            int          llx,
                            int          lly,
                            int          urx,
                            int          ury)
{

    int            error = 0;
    cpl_image    * self;
    void         * pixels;
    const long int inc[3] = {1, 1, 1};
    long int       fpixel[3];
    long int       lpixel[3];
    int            loadtype;
    int            nx, ny;

    cpl_ensure(fptr      != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(pnaxis    != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(naxes     != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(ppix_type != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(filename  != NULL, CPL_ERROR_NULL_INPUT, NULL);

    if (hdumov != 0 && fits_movrel_hdu(fptr, hdumov, NULL, &error)) {
        (void)cpl_error_set_fits(CPL_ERROR_BAD_FILE_FORMAT, error,
                                 fits_movrel_hdu, "filename='%s', pnum=%d, "
                                 "hdumov=%d", filename, pnum, hdumov);
        return NULL;
    }

    /* Get NAXIS, if needed */
    if (*pnaxis == 0 && fits_get_img_dim(fptr, pnaxis, &error)) {
        (void)cpl_error_set_fits(CPL_ERROR_BAD_FILE_FORMAT, error,
                                 fits_get_img_dim, "filename='%s', pnum=%d, "
                                 "hdumov=%d", filename, pnum, hdumov);
        return NULL;
    }
    /* Verify NAXIS before trying anything else */
    if (*pnaxis != 2 && *pnaxis != 3) {
        (void)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                    "filename='%s', pnum=%d, hdumov=%d, "
                                    "NAXIS=%d", filename, pnum, hdumov,
                                    *pnaxis);
        return NULL;
    }

    /* Get NAXIS[12[3]], if needed */
    if (naxes[0] == 0 && fits_get_img_size(fptr, *pnaxis, naxes, &error)) {
        (void)cpl_error_set_fits(CPL_ERROR_BAD_FILE_FORMAT, error,
                                 fits_get_img_size, "filename='%s', "
                                 "pnum=%d, hdumov=%d, NAXIS=%d", filename,
                                 pnum, hdumov, *pnaxis);
        return NULL;
    }

    /* Verify NAXIS[123] */
    if (naxes[0] == 0 || naxes[1] == 0) {
        /* FIXME: Is this actually possible with a non-zero NAXIS ? */
        (void)cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                    "filename='%s', pnum=%d, hdumov=%d, "
                                    "NAXIS=%d, NAXIS1=%ld, NAXIS2=%ld",
                                    filename, pnum, hdumov, *pnaxis,
                                    naxes[0], naxes[1]);
        return NULL;
    }
    if (*pnaxis == 3 && naxes[2] == 0) {
        /* FIXME: Is this actually possible with a non-zero NAXIS ? */
        (void)cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                    "filename='%s', pnum=%d, hdumov=%d, "
                                    "NAXIS=%d, NAXIS1=%ld, NAXIS2=%ld "
                                    "NAXIS3=0", filename, pnum, hdumov,
                                    *pnaxis, naxes[0], naxes[1]);
        return NULL;
    }

    if (do_window) {
        /* Verify the window size */
        /* If the naxes[] passed is from a previous succesful call here,
           then this check is redundant. Don't rely on that. */
        if (llx < 1 || lly < 1 || urx > naxes[0] || ury > naxes[1]
            || urx < llx || ury < lly) {
            (void)cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                        "filename='%s', pnum=%d, hdumov=%d, "
                                        "llx=%d, lly=%d, urx=%d, ury=%d, "
                                        "NAXIS=%d, NAXIS1=%ld, NAXIS2=%ld",
                                        filename, pnum, hdumov, llx, lly,
                                        urx, ury, *pnaxis, naxes[0], naxes[1]);
            return NULL;
        }
    } else {
        llx = lly = 1;
        urx = naxes[0];
        ury = naxes[1];
    }

    /* Create the zone definition. The 3rd element not defined for NAXIS = 2 */
    fpixel[0] = llx;
    fpixel[1] = lly;
    lpixel[0] = urx;
    lpixel[1] = ury;
    if (*pnaxis == 3) {

        /* Verify plane number */
        if (pnum + 1 > naxes[2]) {
            (void)cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                        "filename='%s', pnum=%d, hdumov=%d, "
                                        "NAXIS=3, NAXIS1=%ld, NAXIS2=%ld, "
                                        "NAXIS3=%ld", filename, pnum, hdumov,
                                        naxes[0], naxes[1], naxes[2]);
            return NULL;
        }

        fpixel[2] = lpixel[2] = pnum + 1;
    } else if (pnum != 0) {
        /* May not ask for any plane but the first when NAXIS == 2 */
        (void)cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                    "filename='%s', pnum=%d, hdumov=%d, "
                                    "NAXIS=%d, NAXIS1=%ld, NAXIS2=%ld",
                                    filename, pnum, hdumov, *pnaxis,
                                    naxes[0], naxes[1]);
        return NULL;
    }

    if (*ppix_type == CPL_TYPE_UNSPECIFIED) {

        /* The pixel type of the created image is determined
           by the pixel type of the loaded FITS image */

        int imgtype;

        if (fits_get_img_type(fptr, &imgtype, &error)) {
            (void)cpl_error_set_fits(CPL_ERROR_BAD_FILE_FORMAT, error,
                                     fits_get_img_type, "filename='%s', "
                                     "pnum=%d, hdumov=%d, NAXIS=%d, "
                                     "NAXIS1=%ld, NAXIS2=%ld", filename,
                                     pnum, hdumov, *pnaxis, naxes[0], naxes[1]);
            return NULL;
        }
        
        switch (imgtype) {
        case BYTE_IMG    :
        case SHORT_IMG   :
        case LONG_IMG    :
        case LONGLONG_IMG:
            *ppix_type = CPL_TYPE_INT;
            break;
        case FLOAT_IMG   :
            *ppix_type = CPL_TYPE_FLOAT;
            break;
        case DOUBLE_IMG  :
            *ppix_type = CPL_TYPE_DOUBLE;
            break;
        default:
            break;
        }

        if (*ppix_type == CPL_TYPE_UNSPECIFIED) {
            (void)cpl_error_set_message(cpl_func, CPL_ERROR_BAD_FILE_FORMAT,
                                        "filename='%s', pnum=%d, hdumov=%d, "
                                        "NAXIS=%d, NAXIS1=%ld, NAXIS2=%ld, "
                                        "imgtype=%d", filename, pnum, hdumov,
                                        *pnaxis, naxes[0], naxes[1], imgtype);
            return NULL;
        }
    }

    if (*ppix_type == CPL_TYPE_DOUBLE) {
        loadtype = TDOUBLE;
    } else if (*ppix_type == CPL_TYPE_FLOAT) {
        loadtype = TFLOAT;
    } else if (*ppix_type == CPL_TYPE_INT) {
        loadtype = TINT;
    } else {
        (void)cpl_error_set_message(cpl_func, CPL_ERROR_INVALID_TYPE,
                                    "filename='%s', pnum=%d, hdumov=%d, "
                                    "NAXIS=%d, NAXIS1=%ld, NAXIS2=%ld, "
                                    "im_type=%d", filename, pnum, hdumov,
                                    *pnaxis, naxes[0], naxes[1], *ppix_type);
        return NULL;
    }

    nx = urx - llx + 1;
    ny = ury - lly + 1;

    pixels = cpl_malloc((size_t)nx * ny * cpl_type_get_sizeof(*ppix_type));

    if (fits_read_subset(fptr, loadtype, fpixel, lpixel, (long*)inc,
                         NULL, pixels, NULL, &error)) {
        cpl_free(pixels);
        (void)cpl_error_set_fits(CPL_ERROR_BAD_FILE_FORMAT, error,
                                 fits_read_subset, "filename='%s', "
                                 "pnum=%d, hdumov=%d, NAXIS=%d, "
                                 "NAXIS1=%ld, NAXIS2=%ld", filename,
                                 pnum, hdumov, *pnaxis, naxes[0], naxes[1]);
        return NULL;
    }

    self = cpl_image_wrap_(nx, ny, *ppix_type, pixels);

    if (self == NULL) {
        cpl_free(pixels);
        cpl_error_set_where(cpl_func);
    }

    return self;
}

/*----------------------------------------------------------------------------*/
/**
  @internal
  @ingroup cpl_image
  @brief    Create an image, optionally using an existing pixel buffer.
  @param    nx      Size in x
  @param    ny      Size in y
  @param    type    Pixel type
  @param    pixels  Pixel data, or NULL
  @return   1 newly allocated cpl_image or NULL on error
  @see      cpl_image_new(), cpl_image_wrap_double()
  @note if pixels is NULL, a new buffer will be cpl_calloc'ed.

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_ILLEGAL_INPUT if nx or ny is non-positive or their product is too big
  - CPL_ERROR_INVALID_TYPE if the passed image type is not supported
 */
/*----------------------------------------------------------------------------*/
static cpl_image * cpl_image_wrap_(int nx, int ny, cpl_type type, void * pixels)
{
    cpl_image  * self;
    const int    npix = nx * ny;
    const size_t upix = (size_t)nx * (size_t)ny;

    cpl_ensure( nx > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure( ny > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    /* Check for overflow */
    cpl_ensure( npix > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure( (size_t)npix == upix, CPL_ERROR_ILLEGAL_INPUT, NULL);

    cpl_ensure( type == CPL_TYPE_INT    || type == CPL_TYPE_FLOAT ||
                type == CPL_TYPE_DOUBLE || type == CPL_TYPE_FLOAT_COMPLEX ||
        type == CPL_TYPE_DOUBLE_COMPLEX, CPL_ERROR_INVALID_TYPE, NULL);

    self         = cpl_malloc(sizeof(cpl_image));
    self->nx     = nx;
    self->ny     = ny;
    self->type   = type;
    self->bpm    = NULL;
    self->pixels = pixels != NULL ? pixels
        : cpl_calloc(upix, cpl_type_get_sizeof(type));

    return self;
}
