/* $Id: cpl_apertures_img.c,v 1.5 2010/11/11 09:23:18 llundin Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2008 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: llundin $
 * $Date: 2010/11/11 09:23:18 $
 * $Revision: 1.5 $
 * $Name: cpl-5_3_0-BRANCH $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <stdio.h>
#include <math.h>

#include <cpl_image_basic.h>
#include <cpl_error.h>
#include <cpl_errorstate.h>

#include "cpl_apertures_img.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup cpl_apertures_img  High-level functions that uses apertures
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
                            Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Compute FWHM values in x and y for a list of apertures
  @param    in      the input image
  @param    aperts  the list of apertures
  @return   A newly allocated object containing the fwhms in x and y or NULL
  @see cpl_image_get_fwhm()
  @note On success the created cpl_bivector must be deallocated with
  cpl_bivector_delete().
 
  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the number of apertures is lower than 1
  - CPL_ERROR_DATA_NOT_FOUND if no aperture has a valid FWHM
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if an aperture specifies a pixel outside the
    image size range
 */
/*----------------------------------------------------------------------------*/
cpl_bivector * cpl_apertures_get_fwhm(const cpl_image     * in,
                                      const cpl_apertures * aperts)
{
    cpl_errorstate prevstate = cpl_errorstate_get();
    cpl_bivector * fwhms;
    double       * fwhms_x;
    double       * fwhms_y;
    int            naperts;
    int            i, nok = 0;


    cpl_ensure(in     != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(aperts != NULL, CPL_ERROR_NULL_INPUT, NULL);

    naperts = cpl_apertures_get_size(aperts);
    cpl_ensure(naperts >= 1, CPL_ERROR_ILLEGAL_INPUT, NULL);

    /* Allocate storage */
    fwhms = cpl_bivector_new(naperts);
    fwhms_x = cpl_bivector_get_x_data(fwhms);
    fwhms_y = cpl_bivector_get_y_data(fwhms);


    /* Compute FWHM on all apertures */
    for (i=0; i < naperts; i++) {
        const cpl_error_code error
            = cpl_image_get_fwhm(in,
                                 (int)cpl_apertures_get_max_x(aperts, i+1),
                                 (int)cpl_apertures_get_max_y(aperts, i+1),
                                 &(fwhms_x[i]),
                                 &(fwhms_y[i]));

        if (!error) {
            nok++;
        } else if (error == CPL_ERROR_DATA_NOT_FOUND) {
            /* This error can be ignored */
            cpl_errorstate_set(prevstate);
        } else {
            /* This unexpected error cannot be ignored */
            cpl_bivector_delete(fwhms);
            cpl_ensure(0, error, NULL);
        }
    }

    if (!nok) {
        /* Require at least one fwhm to be OK */
        cpl_bivector_delete(fwhms);
        cpl_ensure(0, CPL_ERROR_DATA_NOT_FOUND, NULL);
    }

    return fwhms;
}

/**@}*/

