package org.eclipse.webdav.dom;

/*
 * (c) Copyright IBM Corp. 2000, 2001.
 * All Rights Reserved.
 */

import java.util.*;
import org.eclipse.webdav.Policy;
import org.w3c.dom.*;

/**
 * An element editor for the WebDAV propertybehavior element. See
 * RFC2518 section 12.12 for the element's definition.
 * <p>
 * <b>Note:</b> This class/interface is part of an interim API that is still under 
 * development and expected to change significantly before reaching stability. 
 * It is being made available at this early stage to solicit feedback from pioneering 
 * adopters on the understanding that any code that uses this API will almost 
 * certainly be broken (repeatedly) as the API evolves.
 * </p>
 */
public class PropertyBehavior extends ElementEditor {
	/**
	 * An ordered collection of the element names of the
	 * propertybehavior element's children.
	 */
	protected static final String[] childNames = new String[] {"omit", "keepalive"}; 

	/**
	 * An ordered collection of the element names of the
	 * propertybehavior element's children in the "omit" form
	 */
	public static String[] fgNamesOmit = new String[] {"omit"};

	/**
	 * An ordered collection of the element names of the
	 * propertybehavior element's children in the "keep alive" form
	 */
	public static String[] fgNamesKeepAlive = new String[] {"keepalive"};
/**
 * Creates a new editor on the given WebDAV propertybehavior element.
 * The element is assumed to be well formed.
 *
 * @param root a propertybehavior element
 * @throws        MalformedElementException if there is reason to
 *                believe that the element is not well formed
 */
public PropertyBehavior(Element root) throws MalformedElementException {

	super(root, "propertybehavior");
}
/**
 * Adds the given property href to this propertybehavior's list of live
 * properties. The property href must not be <code>null</code> and the
 * form of this property behavior must not already be omit or
 * keepAllAlive.
 *
 * @param propertyHref the property href to add
 */
public void addProperty(String propertyHref) {

	Assert.isNotNull(propertyHref);
	Assert.isTrue(getFirstChild(root, "omit") == null);

	Element keepalive = getFirstChild(root, "keepalive");

	if (keepalive == null)
		keepalive = addChild(root, "keepalive", fgNamesKeepAlive, true);
	else
		Assert.isTrue(!"*".equals(getFirstText(keepalive)));

	addChild(
		keepalive, 
		"href", 
		encodeHref(propertyHref), 
		new String[] {"href"}, 
		false); 
}
/**
 * Creates a new WebDAV propertybehavior element and sets it as the root
 * of the given document. Returns an editor on the new propertybehavior
 * element. The document must not be <code>null</code>, and must not
 * already have a root element.
 *
 * @param document the document that will become the root of a new
 *                 propertybehavior element
 * @return         an element editor on a propertybehavior element
 */
public static PropertyBehavior create(Document document) {

	Assert.isNotNull(document);
	Assert.isTrue(document.getOwnerDocument() == null);

	Element element = create(document, "propertybehavior");

	PropertyBehavior result = null;
	try {
		result = new PropertyBehavior(element);
	} catch (MalformedElementException e) {
		Assert.isTrue(false, Policy.bind("assert.internalError"));
	}

	return result;
}
/**
 * Returns an <code>Enumeration</code> over this propertybehavior's
 * property hrefs. The methods <code>isMerge()</code> and
 * <code>isKeepAllAlive</code> return false if this propertybehavior is
 * in the "keep some alive" form.
 *
 * @return an <code>Enumeration</code> of <code>String</code>s
 * @throws MalformedElementException if there is reason to believe that
 *         this editor's underlying element is not well formed, or this
 *         propertybehavior is not in the "keep some alive" form
 * @see    isKeepAllAlive()
 * @see    isOmit()
 */
public Enumeration getProperties() throws MalformedElementException {

	Element keepalive = getFirstChild(root, "keepalive");
	ensureNotNull(Policy.bind("ensure.missingKeealiveElmt"), keepalive);
	ensure(
		!"*".equals(getFirstText(keepalive)), 
		Policy.bind("ensure.wrongForm")); 

	final Element firstHref = getFirstChild(keepalive, "href");
	ensureNotNull(Policy.bind("ensure.missingHrefElmt"), firstHref);

	Enumeration e = new Enumeration() {
		Element currentHref = firstHref;

		public boolean hasMoreElements() {

			return currentHref != null;
		}

		public Object nextElement() {

			if (!hasMoreElements())
				throw new NoSuchElementException();

			String href = getFirstText(currentHref);
			currentHref = getTwin(currentHref, true);

			return decodeHref(href);
		}
	};

	return e;
}
/**
 * Returns <code>true</code> if this propertybehavior is in the
 * "keep all alive" form, otherwise, returns <code>false</code>.
 *
 * @return a boolean indicating whether this propertybehavior is in the
 *         "keep all alive" form or not
 * @throws MalformedElementException if there is reason to believe that
 *         this editor's underlying element is not well formed
 */
public boolean isKeepAllAlive() throws MalformedElementException {

	Element child = getFirstChild(root, childNames);
	ensureNotNull(Policy.bind("ensure.expectingOmitOrKeepaliveElmt"), child);

	boolean isKeepAllAlive = false;
	if (isDAVElement(child, "keepalive")) {
		isKeepAllAlive = "*".equals(getFirstText(child));
		ensureNull(Policy.bind("ensure.conflictingHrefElmt"), getFirstChild(child, "href"));
	}

	child = getNextSibling(child, childNames);
	ensureNull(Policy.bind("ensure.conflictingOmitOrKeepaliveElmt"), child);

	return isKeepAllAlive;
}
/**
 * Returns <code>true</code> if this propertybehavior is in the
 * "omit" form, otherwise, returns <code>false</code>.
 *
 * @return a boolean indicating whether this propertybehavior is in the
 *         "omit" form or not
 * @throws MalformedElementException if there is reason to believe that
 *         this editor's underlying element is not well formed
 */
public boolean isOmit() throws MalformedElementException {

	Element child = getFirstChild(root, childNames);
	ensureNotNull(Policy.bind("ensure.expectingOmitOrKeepaliveElmt"), child);

	boolean isOmit = isDAVElement(child, "omit");

	child = getNextSibling(child, childNames);
	ensureNull(Policy.bind("ensure.conflictingOmitOrKeepaliveElmt"), child);

	return isOmit;
}
/**
 * Sets whether this propertybehavior is in the "keep all alive" form or
 * not.
 *
 * @param isKeepAllAlive a boolean indicating whether this
 *                       propertybehavior will be in the "keep all
 *                       alive" form
 */
public void setIsKeepAllAlive(boolean isKeepAllAlive) {

	Element child = getFirstChild(root, childNames);

	boolean isAlreadyKeepAllAlive = false;
	if (isDAVElement(child, "keepalive"))
		isAlreadyKeepAllAlive = "*".equals(getFirstText(child));

	if (isKeepAllAlive) {
		if (!isAlreadyKeepAllAlive) {
			if (child != null)
				root.removeChild(child);
			appendChild(root, "keepalive", "*");
		}
	} else if (isAlreadyKeepAllAlive)
		root.removeChild(child);
}
/**
 * Sets whether this propertybehavior is in the "omit" form or not.
 *
 * @param isOmit a boolean indicating whether this propertybehavior will
 *               be in the "omit" form
 */
public void setIsOmit(boolean isOmit) {

	Element child = getFirstChild(root, childNames);
	boolean isAlreadyOmit = isDAVElement(child, "omit");

	if (isOmit) {
		if (!isAlreadyOmit) {
			if (child != null)
				root.removeChild(child);
			appendChild(root, "omit");
		}
	} else if (isAlreadyOmit)
		root.removeChild(child);
}
}
