/*
 
    Ejecter - Safely, easily remove external peripherals
    Copyright 2008-2011, Federico Pelloni <federico.pelloni@gmail.com>
 
    
    This file is part of Ejecter.
 
    Ejecter is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Ejecter is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with Ejecter.  If not, see <http://www.gnu.org/licenses/>.
   
*/ 

using GLib, Gdu;


namespace Ejecter {


public class Device: GLib.Object {


    public GLib.Drive drive;
    public Gdu.Device device;
    private GLib.SList<GLib.Volume> volumes;
    private GLib.SList<GLib.Mount> mounts;
    private Gtk.Image icon;
    private Gtk.ImageMenuItem menuitem;
    private Gtk.Menu menu;
    private bool was_removed = false;
    private bool icon_updated = false;
    private uint volume_count = 0;
    private uint mount_count = 0;
    public string name;
    public string description;



    
    public Device (GLib.Drive drive, Gdu.Device device) {
    
        this.drive = drive;
        this.device = device;
        
        this.volumes = new GLib.SList<GLib.Volume>();
        this.mounts = new GLib.SList<GLib.Mount>();
        
        this.menuitem = new Gtk.ImageMenuItem();
        this.menuitem.set_label("");

        this.icon = new Gtk.Image.from_gicon(this.drive.get_icon(), Gtk.IconSize.BUTTON);
        this.menuitem.set_always_show_image(true);
        this.menuitem.set_image(this.icon);
        
        this.menuitem.show_all();
        this.menuitem.show();
        this.menuitem.activate.connect((a) => this.clicked_eject(a));

        if (this.device.is_optical_disc()) this.set_mounted(true);
        else this.set_mounted(false);

        this.drive.disconnected.connect(this.handle_removed_drive);
        
    }
    
    
    private void update_label () {
    
        if (this.volume_count == 0) {

            this.name = this.drive.get_name();
            this.description = "";

        } else if (this.volume_count == 1) {

            GLib.Volume v = this.volumes.nth_data(0);
            
            this.name = v.get_name();
            this.description = this.drive.get_name();
            
        } else {
            
            string volumes = "";
            bool first = true;
            
            foreach (GLib.Volume v in this.volumes)
                if (first) {
                    volumes += v.get_name();
                    first = false;
                } else {
                    volumes += ", " + v.get_name();
                }
            
            this.name = this.drive.get_name();
            this.description = volumes;
            
        }
        
        this.menuitem.set_label(this.name + " (" + this.description + ")");
    
    }
        
        
    public void attach (Gtk.Menu menu) {
    
        menu.append(this.menuitem);
        this.menu = menu;
        
    }
        
    
    public signal void removed ();
    public signal void ejected ();
    public signal void unmounted ();
    
    public void remove () {
    
        if (this.was_removed) return;
        
        this.menu.remove(this.menuitem);

        this.icon.destroy();
        this.menuitem.destroy();
        
        this.was_removed = true;
        this.removed();
        
    }
        
        
    public void set_mounted (bool mounted) {
        
        mounted = (mounted || this.device.is_optical_disc());
        
        this.menuitem.set_sensitive(mounted);
        
    }
    
    
    private async void clicked_eject (Gtk.MenuItem menuitem) {
    
        Gtk.MountOperation op = new Gtk.MountOperation(null);
        
        if (this.device.is_removable() && this.drive.can_eject()) {

            debug("Eject: " + this.drive.get_name());        
            
            try {
                if (yield this.drive.eject_with_operation(GLib.MountUnmountFlags.NONE, op)) {
                    debug("EJECT OK");
                    this.unmounted();
                } else {
                    debug("EJECT FAILED");
                }
            } catch (GLib.Error err) {
                debug("EJECT ERROR: "+err.message);
            }

        } else {
            
            debug("Unmount: " + this.drive.get_name());

            foreach (GLib.Volume v in (GLib.List<GLib.Volume>) this.drive.get_volumes()) {
                try {
                    GLib.Mount m = v.get_mount();
                    if (yield m.eject_with_operation(GLib.MountUnmountFlags.NONE, op)) {
                        debug("UNMOUNT OK");
                        this.unmounted();
                    } else {
                        debug("UNMOUNT FAILED");
                    }
                } catch (GLib.Error err2) {
                    debug("UNMOUNT ERROR: "+err2.message);
                }
            }
            
        }
        
    }
    
    

    private void handle_unmounted (GLib.Mount mount) {
    
        this.mounts.remove(mount);
        this.mount_count--;
        this.update_label();
        
        if (this.mount_count <= 0) {
            this.set_mounted(false);
            this.unmounted();
        }
    
    }
    
    
    private void handle_removed_drive (GLib.Drive drive) {
    
        this.remove();
    
    }
    
    
    private void handle_removed_volume (GLib.Volume volume) {
    
        this.volumes.remove(volume);
        this.volume_count--;
        this.update_label();
        
        if (this.volume_count == 0)
            this.remove();
    
    }
    
    
    public void add_volume (GLib.Volume volume) {
        
        this.volumes.prepend(volume);
        this.volume_count++;
        volume.removed.connect(this.handle_removed_volume);
        
        if (!this.icon_updated) {
            this.icon.set_from_gicon(volume.get_icon(), Gtk.IconSize.BUTTON);
            this.icon_updated = true;
        }
        
        this.update_label();
    
    }
    
    
    public void add_mount (GLib.Mount mount) {
    
        GLib.Volume volume = mount.get_volume();
        if (this.volumes.index(volume) < 0) this.add_volume(volume);
    
        this.mounts.prepend(mount);
        this.mount_count++;
        this.set_mounted(true);
        mount.unmounted.connect(this.handle_unmounted);
        this.update_label();
    
    }

}

}


