/* $Id: svg.c,v 1.54 2003/06/12 21:16:55 sjoerd Exp $ */

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <assert.h>
#include <ctype.h>
#include <locale.h>
#include <limits.h>

#include <libxml/xmlmemory.h>
#include <libxml/parser.h>

#include "global.h"
#include "svg.h"
#include "sprite.h"
#include "svgcolors.h"

#ifndef GDEBUG
#define GDEBUG(...) DEBUG(DGRAPHICS,"Graphics",__VA_ARGS__)
#endif

#define SVG_TMP_SCALE 1000

/******************************************************************************
 * SVG to Graphics SVG Series
 *****************************************************************************/

Graphics_svg_picture *new_gsvg_picture(Vector *norm) {
  	Graphics_svg_picture *gsvg_picture;

	gsvg_picture = malloc(sizeof(Graphics_svg_picture));
  memset(gsvg_picture,0,sizeof(Graphics_svg_picture));
	gsvg_picture->normal = copy_vector(norm);

	return gsvg_picture;
}

void del_gsvg_picture(Graphics_svg_picture *gsvg_picture) {
	if (gsvg_picture->type == PT_BEZIER) {
		free(gsvg_picture->path);
		free(gsvg_picture->transform);
	}
	free(gsvg_picture->normal);
	free(gsvg_picture->style);
	free(gsvg_picture);
}

Graphics_svg *new_gsvg(void) {
  	Graphics_svg *gsvg;

	gsvg = malloc(sizeof(Graphics_svg));
	gsvg->pic = new_list();
	
	return gsvg;
}

void del_gsvg(Graphics_svg *gsvg) {
	Graphics_svg_picture *gsvg_picture;
	
	while((gsvg_picture = (Graphics_svg_picture *)list_pop(gsvg->pic))) {
		del_gsvg_picture(gsvg_picture);
	}
	
	del_list(gsvg->pic);
	
	free(gsvg);
}

Graphics_svg_state *new_gsvg_state(void) {
	Graphics_svg_state *gsvg_state;

	gsvg_state = malloc(sizeof(Graphics_svg_state));
	gsvg_state->svg = new_list();
  gsvg_state->state = NULL;
  gsvg_state->fpp = 0;

	return gsvg_state;
}

void del_gsvg_state(Graphics_svg_state *gsvg_state) {
	Graphics_svg *gsvg;
	
	while((gsvg = (Graphics_svg *)list_pop(gsvg_state->svg))) {
		del_gsvg(gsvg);
	}
	del_list(gsvg_state->svg);

  free(gsvg_state->state);
	free(gsvg_state);
}

Graphics_svg_series *new_gss(void) {
	Graphics_svg_series *gss;

	gss = malloc(sizeof(Graphics_svg_series));
	gss->states = new_list();

	return gss;
}

void del_gss(Graphics_svg_series *gss) {
	Graphics_svg_state *gsvg_state;
	
	while((gsvg_state = (Graphics_svg_state *)list_pop(gss->states))) {
		del_gsvg_state(gsvg_state);
	}
	
	del_list(gss->states);
	
	free(gss);
}

Graphics_svg_picture *svgs_parse_rect(xmlDocPtr doc, xmlNodePtr cur, Vector *norm) {
  	/* Watch out, this function can possibly return NULL! */
	
	Graphics_svg_picture *rect; /* Rectangle in a SVG XML. */
	xmlChar *style; /* Style of the bezier curve (stroke, line-fill, etc) */
	xmlChar *width; /* Width of the rectangle */
	xmlChar *height; /* Height of the rectangle */
	xmlChar *x; /* x position of the rectangle */
	xmlChar *y; /* y position of the rectangle */
  
	rect = new_gsvg_picture(norm);
	rect->type = PT_RECT;

	style = xmlGetProp(cur, (const xmlChar *)"style");
	if (style == NULL) {
		GDEBUG("No style defined for rectangle.");
		del_gsvg_picture(rect);
	 	return NULL;
  }
  rect->style = (char *)xmlStrdup(style);
  xmlFree(style);

	width = xmlGetProp(cur, (const xmlChar *)"width");
	if  (width == NULL) {
		GDEBUG("No width defined for rectangle.");
		del_gsvg_picture(rect);
	  return NULL;
  }
  rect->width = atoi((char *)width);
  xmlFree(width);

	height = xmlGetProp(cur, (const xmlChar *)"height");
	if  (height == NULL) {
		GDEBUG("No width defined for rectangle.");
		del_gsvg_picture(rect);
	  return NULL;
  }
	rect->height = atoi((char *)height);
  xmlFree(height);

	x = xmlGetProp(cur, (const xmlChar *)"x");
	if  (x == NULL) {
		GDEBUG("No width defined for rectangle.");
		del_gsvg_picture(rect);
	  return NULL;
  }
	rect->x = atoi((char *)x);
  xmlFree(x);

	y = xmlGetProp(cur, (const xmlChar *)"y");
	if  (y == NULL) {
		GDEBUG("No width defined for rectangle.");
		del_gsvg_picture(rect);
	  return NULL;
  }
	rect->y = atoi((char *)y);
  xmlFree(y);
	
	GDEBUG("Rectangle: x = %d, y = %d, width = %d, height = %d",
		rect->x, rect->y, rect->width, rect->height);
	return rect;
}

Graphics_svg_picture *svgs_parse_bezier(xmlDocPtr doc, xmlNodePtr cur, Vector *norm) {
  	/* Watch out, this function can possibly return NULL! */
	
	Graphics_svg_picture *bezier; /* Bezier curve in a SVG XML. */
	xmlChar *style; /* Style of the bezier curve (stroke, line-fill, etc) */
	xmlChar *d; /* Bezier curve */
	xmlChar *transform; /* Transformation information */
  
	style = xmlGetProp(cur, (const xmlChar *)"style");
	if (style == NULL) {
		GDEBUG("No style defined in SVG.");
	  return NULL;
  }

	d = xmlGetProp(cur, (const xmlChar *)"d");
	if  (d == NULL) {
		GDEBUG("No bezier data defined in SVG.");
    xmlFree(style);
	 	return NULL;
	}

	bezier = new_gsvg_picture(norm);
	bezier->type = PT_BEZIER;

  bezier->style = (char *)xmlStrdup(style);
  xmlFree(style);
	bezier->path = (char *)xmlStrdup(d);
  xmlFree(d);

	transform = xmlGetProp(cur, (const xmlChar *)"transform");
	if (transform == NULL) {
		GDEBUG("No transformation data defined in SVG."); 
		GDEBUG("Applying standard (=no) transformation.");
		bezier->transform = strdup("translate(0.0,0.0)");
	}
	else {
		bezier->transform = (char *)xmlStrdup(transform);
    xmlFree(transform);
	}  	
	GDEBUG("bezier->type: %d", bezier->type);
	GDEBUG("bezier->style: %s", bezier->style);
	GDEBUG("bezier->transform: %s", bezier->transform);
	GDEBUG("bezier->path: %s", bezier->path);
	
	return bezier;
}


Graphics_svg *svgs_parse_svg(xmlDocPtr doc, xmlNodePtr cur) {
	Graphics_svg_picture *pic;
	Graphics_svg *svg; /* SVG data */
	xmlChar *width;
	xmlChar *height;
	float fwidth = 1000;
	float fheight = 1000;
	Vector *norm; /* Normalization vector */

	width = xmlGetProp(cur, (const xmlChar *)"width");
	if (width != NULL)
		fwidth = atof((char *)width);
	
	height = xmlGetProp(cur, (const xmlChar *)"height");
	if (height != NULL)
		fheight = atof((char *)height);

	norm = new_vector((SVG_TMP_SCALE * SVG_TMP_SCALE) / fwidth, 
                    (SVG_TMP_SCALE * SVG_TMP_SCALE) / fheight);
	
	GDEBUG("width, height: %f, %f", fwidth, fheight);
	GDEBUG("norm: <%d, %d>", norm->x, norm->y); 
	xmlFree(width);
	xmlFree(height);
	
	svg = new_gsvg();
		
	cur = cur->xmlChildrenNode;
			
	while (cur != NULL) {
		if ((!xmlStrcmp(cur->name, (const xmlChar *)"path"))){			
			GDEBUG("Parsing the path.");
			pic = svgs_parse_bezier(doc, cur, norm);
			if (pic == NULL)
				GDEBUG("Error in bezier curve.");
			else
				list_append(svg->pic, pic);
   	}						  
		else if ((!xmlStrcmp(cur->name, (const xmlChar *)"rect"))){
			GDEBUG("Parsing the rect.");
			pic = svgs_parse_rect(doc, cur, norm);
			if (pic == NULL)
				GDEBUG("Error in rectangle.");
			else
				list_append(svg->pic, pic);
		}
		else if ((!xmlStrcmp(cur->name, (const xmlChar *)"g"))){
		 	GDEBUG("Group defined but not supported.");
			GDEBUG("All elements in this group will not be parsed.");
		}  			
	  cur = cur->next;
	}	

	return svg;
}

Graphics_svg_state *svgs_parse_data(xmlDocPtr doc, xmlNodePtr cur) {
 	/* Watch out, this function can possibly return NULL! */
	
	xmlChar *fpp; /* Frames per picture for each animation. */
	xmlChar *state; /* State of the model (walking, etc). */
	Graphics_svg *svg; /* SVG data */
	Graphics_svg_state *gsvg_state; 

 	gsvg_state = new_gsvg_state();
	cur = cur->xmlChildrenNode;
	
  while (cur != NULL) {
		GDEBUG("Element type: %d %d", cur->type, XML_ELEMENT_NODE);
		GDEBUG("Contents: %s", cur->content);
		if (cur->type == XML_ELEMENT_NODE) {
			GDEBUG("Parsing xml node: %s", cur->name);
			if ((!xmlStrcmp(cur->name, (const xmlChar *)"info"))){
				GDEBUG("Parsing info");
		  
				fpp = xmlGetProp(cur, (const xmlChar *)"fpp");
				if (fpp == NULL) {
					GDEBUG("No fpp defined in SVGserie->data->info.");
          del_gsvg_state(gsvg_state);
					return NULL;
				} 
				GDEBUG("fpp: %s", fpp);
				gsvg_state->fpp=atoi((char *)fpp);
        xmlFree(fpp);

				state = xmlGetProp(cur, (const xmlChar *)"state");
        if  (state == NULL) {
					GDEBUG("No state defined in SVGserie->data->info.");
          del_gsvg_state(gsvg_state);
					return NULL;
				} 
				GDEBUG("state: %s", state);
				gsvg_state->state = (char *)xmlStrdup(state);
        xmlFree(state);

			  GDEBUG("Info parsed");
			} else if ((!xmlStrcmp(cur->name, (const xmlChar *)"svg"))){
				GDEBUG("Parsing svg");
				svg = svgs_parse_svg(doc, cur);

				if (svg == NULL)
					GDEBUG("Error in SVGseries->data->svg.");
				else
					list_append(gsvg_state->svg, svg);
			} else {
				GDEBUG("Unknown element: %s.", cur->name);
        del_gsvg_state(gsvg_state);
				return NULL;
			}
		}
  	cur = cur->next;
  } 

  if (gsvg_state->state == NULL) {
    GDEBUG("No state information");
    del_gsvg_state(gsvg_state);
		return NULL;
  }
 	return gsvg_state;
}

Graphics_svg_series *parse_svg_series(char *SVGserie) {
  	/* Watch out, this function can possibly return NULL! */
	
	xmlDocPtr doc; /* Pointer to the XML document */
	xmlNodePtr cur; /* Pointer to the current node */
	Graphics_svg_series *gss; /* Graphics SVG series struct */
	Graphics_svg_state *data; /* SVG data */
	GDEBUG("Parsing svgserie");

	gss = new_gss();
	
	/* Open xml file */
	doc = xmlParseDoc((xmlChar *)SVGserie);

  if (doc == NULL ) {
		GDEBUG("SVG serie not parsed successfully.");
		del_gss(gss);
		return NULL;
  }

	cur = xmlDocGetRootElement(doc);

  if (cur == NULL) {
  	GDEBUG("Empty SVG serie.");
  	xmlFreeDoc(doc);
		del_gss(gss);
		return NULL;
  }
	/* XML opened and checked */
	GDEBUG("XML opened and checked.");

	GDEBUG("SVGserie: %s", SVGserie);

	/* Traverse the tree, using svg_parse_data on each data field */
  cur = cur->xmlChildrenNode;
  while (cur != NULL) {
  	if ((!xmlStrcmp(cur->name, (const xmlChar *)"data"))){
			GDEBUG("Parsing data");
			data = svgs_parse_data(doc, cur);
			if (data != NULL) {
				GDEBUG("Data successfully parsed.");
				list_append(gss->states, data);
				GDEBUG("Data successfully appended.");
			} else {
        xmlFreeDoc(doc);
				del_gss(gss);
				return NULL;
			}
    }
    cur = cur->next;
  }
	
  xmlFreeDoc(doc);

	GDEBUG("Returning gss");
	return gss;
}

/******************************************************************************
 * Graphics_SVG_series to Sprite_source functions
 *****************************************************************************/

int term(int *i, char sym, char *string) {
  GDEBUG("Term: %c == %c", string[*i], sym);
	if (string[*i] == sym) {
		(*i)++;
		return TRUE;
	}
	else
		return FALSE;
}

float
parse_num(int *i, char *bez) {
	char* temp;
	float res;
  char *oldlocale;

  oldlocale = strdup (setlocale (LC_NUMERIC, NULL));
  setlocale (LC_NUMERIC, "C");
  /* LC_NUMERIC is set to C so the decimal seperator is . */
  res = strtod(bez + *i,&temp);
  setlocale (LC_NUMERIC, oldlocale);
  free(oldlocale);

  /* strip whitespace */
  while (*temp != '\0' && isspace(*temp )) {
    temp++;
  }
  *i = temp - bez;
	return res;
}

static void
parse_named_color(char *style, int *i, Graphics_sprite_rgba *col) {
	/* FIXME: What to do with stroke color ``none''? */
	int endpos;
	char *name_end = strstr(style + *i, ";");
	char *colorname;
	int cli; /* index in colorlist */

	endpos = (name_end == NULL ? strlen(style) : name_end - style);
	colorname = strndup(style + *i, endpos - *i);
	GDEBUG("Got color name: [%s]", colorname);
	*i = endpos;
	/* Do a bounded linear search for color in colorlist */
	for (cli = 0; cli < sizeof colorlist / sizeof(struct ColorList); cli++)
		if (!strcmp(colorname, colorlist[cli].name)) {
			/* Found the right color */
			col->red = colorlist[cli].r;
			col->green = colorlist[cli].g;
			col->blue = colorlist[cli].b;
			free(colorname);
			return;
		}
	GDEBUG("Didn't find specified color in color list");
	col->red = 0x00;
	col->green = 0xFF;
	col->blue = 0x00;
	free(colorname);
}
 
void
gsp_parse_transform(char *trans,float *matrix) {
  int i = 0;

  /* start with the matrix without transformation */ 
  matrix[0] = 1; matrix[1] = 0; matrix[2] = 0;
  matrix[3] = 0; matrix[4] = 1; matrix[5] = 0;
  if (trans == NULL) return;

  if (!strncmp("matrix(",trans,7)) {
      i = 7;
      for (;isspace(trans[i]); i++)
        ;

			/* parse the translation and scale information */
      matrix[0] = parse_num(&i, trans);
	    term(&i, ',', trans);
      for (;isspace(trans[i]); i++)
        ;

      matrix[3]=parse_num(&i, trans);
      term(&i, ',', trans);
      while (isspace(trans[i])) i++;
			
      matrix[1]=parse_num(&i, trans);
      term(&i, ',', trans);
      while (isspace(trans[i])) i++;
			
      matrix[4]=parse_num(&i, trans);
      term(&i, ',', trans);
      while (isspace(trans[i])) i++;
			
      matrix[2]=parse_num(&i, trans);
      term(&i, ',', trans);
      while (isspace(trans[i])) i++;
			
      matrix[5]=parse_num(&i, trans);
      term(&i, ')', trans);

		} else if (!strncmp("translate(",trans,10)) {
			i = 10;
			while (isspace(trans[i])) i++;
			
			/* parse the translation information */
			matrix[2]=parse_num(&i, trans);
			term(&i, ',', trans);
			while (isspace(trans[i])) i++;
			matrix[5]=parse_num(&i, trans);
			term(&i, ')', trans);
			
			/* create the scale and translation vectors */
		} else {
			GDEBUG("Unknown element in transformation element.");
		}
}

int 
gsp_parse_picture(Graphics_svg_picture *svg_picture, 
                  Graphics_sprite_path *sprite_path) {
	Graphics_sprite_path_elem *sprite_pelem;
		
	int i, j;	/* Counter */
	char *bez; /* Bezier curve */
  float x,y;
  float matrix[6]; /* matrix for the transform */
  float num[8];

	Vector *norm_vec = NULL;
	
	Vector *p1 = NULL; 
	Vector *p2 = NULL; 
	Vector *p3 = NULL; 
	Vector *p4 = NULL; 
	
	Bezier_Vectors *bezvecs = NULL;
	
	norm_vec = svg_picture->normal;
	
  gsp_parse_transform(svg_picture->transform,matrix);

	if (svg_picture->type == PT_BEZIER) {
    GDEBUG("Parsing bezier curve");
		bez = svg_picture->path;
		GDEBUG("Bezier path %s", bez);
		
		i = 0;
		/* The first element must be a move M */
		if (!(term(&i, 'M', bez))) {
	  	GDEBUG("Missing first coordinate in bezier curve.");
			return FALSE;
		}
		while (isspace(bez[i])) i++;

    x = parse_num(&i,bez);
    y = parse_num(&i,bez);
    num[0] = ((x * matrix[0] + y * matrix[1] + matrix[2]) * norm_vec->x);
    num[1] = ((x * matrix[3] + y * matrix[4] + matrix[5]) * norm_vec->y);
		/* The rest of the Bezier curve should only consist of 
		   Bezier curves and straight lines */ 
		while(bez[i] != '\0' && bez[i] != 'z') {
			GDEBUG("New element in curve.");
		 	if (bez[i] == 'C') {
				i++;
		    while (isspace(bez[i])) i++;
				for ( j = 2; j < 8; j += 2) {
          x = parse_num(&i,bez);
          y = parse_num(&i,bez);
          num[j] = ((x * matrix[0] + y * matrix[1] + matrix[2])  * norm_vec->x);
          num[j+1] = ((x * matrix[3] + y * matrix[4] + matrix[5])* norm_vec->y);
	  		}
				p1 = new_vector(num[0], num[1]);
				p2 = new_vector(num[2], num[3]);			
				p3 = new_vector(num[4], num[5]);			
				p4 = new_vector(num[6], num[7]);			

		    GDEBUG("C: (%f, %f) (%f, %f) (%f, %f), (%f, %f)", 
                                                num[0], num[1],
                                                num[2], num[3],
                                                num[4], num[5],
                                                num[6], num[7]
                                               );

				bezvecs = new_bezier_vectors(p1, p2, p3, p4);

        del_vector(p1); del_vector(p2); del_vector(p3); del_vector(p4);

				sprite_pelem = new_sprite_path_elem();

				sprite_path_elem_set_type(sprite_pelem, ET_BEZIER);
				sprite_path_elem_set_vector(sprite_pelem, bezvecs);
				
				list_append(sprite_path->elems, sprite_pelem);
				
				num[0] = num[6];
				num[1] = num[7];
			}
			else if (bez[i] == 'L') {
				i++;
		    while (isspace(bez[i])) i++;
        x = parse_num(&i,bez);
        y = parse_num(&i,bez);
        num[2] = (x * matrix[0] + y * matrix[1] + matrix[2]) * norm_vec->x;
        num[3] = (x * matrix[3] + y * matrix[4] + matrix[5]) * norm_vec->y;

				p1 = new_vector(num[0], num[1]);
				p2 = NULL;			
				p3 = NULL;			
				p4 = new_vector(num[2], num[3]);
				
				bezvecs = new_bezier_vectors(p1, p2, p3, p4);
        del_vector(p1);
        del_vector(p4);
				
				sprite_pelem = new_sprite_path_elem();

				sprite_path_elem_set_type(sprite_pelem, ET_LINE);
				sprite_path_elem_set_vector(sprite_pelem, bezvecs);
				
				list_append(sprite_path->elems, sprite_pelem);	
				
	  		num[0] = num[2];
				num[1] = num[3];
			}
			else {
			 	GDEBUG("Unsupported element in bezier curve.");
				while (bez[i] != '\0' && !(bez[i]=='C' || bez[i]=='L' || bez[i] == 'z'))
					i++;
			}
		}
	}
	else if (svg_picture->type == PT_RECT) {
		sprite_pelem = new_sprite_path_elem();
				
    x = svg_picture->x;
    y = svg_picture->y;
    p1 = new_vector((x * matrix[0] + y * matrix[1] + matrix[2]) * norm_vec->x,
                    (x * matrix[3] + y * matrix[4] + matrix[5]) * norm_vec->y);
		p2 = NULL;			
		p3 = NULL;			
    x = svg_picture->x + svg_picture->width;
    y = svg_picture->y + svg_picture->height;
    p4 = new_vector((x * matrix[0] + y * matrix[1] + matrix[2]) * norm_vec->x,
                    (x * matrix[3] + y * matrix[4] + matrix[5]) * norm_vec->y);

		bezvecs = new_bezier_vectors(p1, p2, p3, p4);
    del_vector(p1);
    del_vector(p4);

		sprite_path_elem_set_type(sprite_pelem, ET_RECT);
		sprite_path_elem_set_vector(sprite_pelem, bezvecs);
		sprite_path_add_elem(sprite_path, sprite_pelem);
	} else {
    assert(FALSE && "Unknown svg type");
  }
 	return TRUE;
}

int gsp_parse_path(void *gsvg_pic, void *gsp_pic) {
	Graphics_svg_picture *svg_picture;
	Graphics_sprite_pic *sprite_pic;
	Graphics_sprite_path *sprite_path;
	Graphics_sprite_rgba *fillcol;
	Graphics_sprite_rgba *linecol;

	
	int style_len;	/* Lenght of the style string. */
	char *style; /* Style of the SVG picture. */
	char color[8]; /* Temporary string, used to cut a RGB string into parts. */
	int i; /* Counter */
	
	svg_picture = (Graphics_svg_picture *) gsvg_pic;
	sprite_pic = (Graphics_sprite_pic *) gsp_pic;
	sprite_path = new_sprite_path();
                        
	/* Parse the svg style for the stroke and fill values */

	style = svg_picture->style;
	i = 0;
	style_len = strlen(style);


	/* Set default values of line- and fillcol */
	linecol = malloc(sizeof(Graphics_sprite_rgba));
	fillcol = malloc(sizeof(Graphics_sprite_rgba));
	
	linecol->red = 0;
	linecol->green = 0;
	linecol->blue = 0;
	linecol->alpha = 1;
	
	fillcol->red = 0;
	fillcol->green = 0;
	fillcol->blue = 0;
	fillcol->alpha = 1;
	
	GDEBUG("Style len: %d", style_len);
	GDEBUG("Parse style string: %s", style);
	/* Parse the style string */
	while ( style[i] != '\0') {
		assert (i <= style_len);
		if (style[i] == 'f') {
			i++;
			/* parse fill */
			if (term(&i, 'i', style)) {
				if (term(&i, 'l', style)) {
					if (term(&i, 'l', style)) {
						if (term(&i, ':', style)) {
						 	if (style[i] == '#') {
								i++;
								
								color[0] = style[i++];
								color[1] = style[i++];
								color[2] = '\0';
								
								fillcol->red = strtol(color, NULL, 16);
								
								color[0] = style[i++];
								color[1] = style[i++];
								color[2] = '\0';
								fillcol->green = strtol(color, NULL, 16);
								
								color[0] = style[i++];
								color[1] = style[i++];
								color[2] = '\0';
								fillcol->blue = strtol(color, NULL, 16);
								
							} else if (style[i] == 'n') {
								i++;
							 	if(term(&i, 'o', style)) {
							 		if(term(&i, 'n', style)) {
							 			if(term(&i, 'e', style)) {
											fillcol->red = 0;
											fillcol->green = 0;
											fillcol->blue = 0;
											fillcol->alpha = 0;
										}
									}
								}
							}
							else {
								parse_named_color(style, &i, fillcol);
							  while (style[i] != '\0' && style[i] != ';')
									i++;
							}
						  GDEBUG("Fillcol (r,g,b) = (%d,%d,%d)", fillcol->red,
                                                    fillcol->green,
                                                    fillcol->blue);
						}
					}
				}
			}
		}
		else if (style[i] == 's') {
			GDEBUG("s");
			i++;
		 	/* parse stroke */
			if (term(&i, 't', style)) {
				if (term(&i, 'r', style)) {
					if (term(&i, 'o', style)) {
						if (term(&i, 'k', style)) {
							if (term(&i, 'e', style)) {
								GDEBUG("stroke");
								if (term(&i, ':', style)) {
						  		if (style[i] == '#') {
										i++;
								
										color[0] = style[i++];
										color[1] = style[i++];
										color[2] = '\0';
										linecol->red = strtol(color, NULL, 16);
										
										color[0] = style[i++];
										color[1] = style[i++];
										color[2] = '\0';
										linecol->green = strtol(color, NULL, 16);

										color[0] = style[i++];
										color[1] = style[i++];
										color[2] = '\0';
										linecol->blue = strtol(color, NULL, 16);
								
									} else {
										parse_named_color(style, &i, linecol);
									}  	
						      GDEBUG("Linecol (r,g,b) = (%d,%d,%d)", linecol->red,
                                                         linecol->green,
                                                         linecol->blue);
								}
							}
						}
					}
				}
			}
		}
		GDEBUG("Skip?");	
		while (style[i+1] != '\0' && style[i] != ';') {
			GDEBUG(">Skipping %c<", style[i]);
		 	i++;
		}
		assert (i < style_len);
	  GDEBUG("-> %c",style[i]);
	  i++;
	}
	GDEBUG("->%d %d %d",linecol->red,linecol->green,linecol->blue);
	GDEBUG("->%d %d %d",fillcol->red,fillcol->green,fillcol->blue);
	
	sprite_path_set_linecolor(sprite_path, linecol);
	sprite_path_set_fillcolor(sprite_path, fillcol);

	gsp_parse_picture(svg_picture, sprite_path);

	sprite_pic_add_path(sprite_pic, sprite_path);
	GDEBUG("Parse style string: %s done", style);

	return TRUE;
}
         
int gsp_parse_pic(void *gsvg, void *gsp_state) {
        Graphics_svg *svg;
        Graphics_sprite_state *sprite_state;
        Graphics_sprite_pic *sprite_pic;

        svg = (Graphics_svg *) gsvg;
        sprite_state = (Graphics_sprite_state *) gsp_state;
        sprite_pic = new_sprite_pic();
                        
				GDEBUG("Parse all paths in the pic");
				list_foreach(svg->pic, gsp_parse_path, sprite_pic);

				GDEBUG("Appending sprite_pic.");
				sprite_state_add_pic(sprite_state, sprite_pic);

        return TRUE;
}
         
int gsp_parse_state(void *gsvg_state, void *gsp_source) {
        Graphics_svg_state *svg_state;
        Graphics_sprite_source *sprite_source;
        Graphics_sprite_state *sprite_state;
 
        svg_state = (Graphics_svg_state *) gsvg_state;
        sprite_source = (Graphics_sprite_source *) gsp_source;
        sprite_state = new_sprite_state();
 
				sprite_state_set_state(sprite_state, svg_state->state);
				GDEBUG("sprite_state->state: %d", sprite_state->state);
				
				sprite_state_set_fpp(sprite_state, svg_state->fpp);
				GDEBUG("sprite_state->fpp: %d", sprite_state->fpp);
        
				GDEBUG("Parse all pics in the state");
        list_foreach(svg_state->svg, gsp_parse_pic, sprite_state);

				GDEBUG("Appending sprite_state.");
				sprite_source_add_state(sprite_source, sprite_state);
        
        return TRUE;
}
        
Graphics_sprite_source *gss_to_sprite_source(Graphics_svg_series *gss) {
  Graphics_sprite_source *gsp_source;

	GDEBUG("Converting SVG series to Sprite source");
  gsp_source = new_sprite_source();
				
  list_foreach(gss->states, gsp_parse_state, gsp_source);
 
  return gsp_source;
}

